/* GMM (C) 2018 Stephane Charette <stephanecharette@gmail.com>
 * $Id: SerialCommWin32.cpp 2629 2018-10-12 07:02:08Z stephane $
 */


/** @file
 * This file is only included when building the Windows version of GMM.
 * See SerialCommStub.cpp for the Linux equivalent which does nothing.
 */


#include "GMM.hpp"


void TabCanvas::open_serial_device(const std::string & device_name)
{
	comm_handle = CreateFile(device_name.c_str(), GENERIC_READ | GENERIC_WRITE, 0, 0, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
	if (comm_handle == INVALID_HANDLE_VALUE)
	{
		const std::string str = "failed to open serial communication device " + device_name;
		warnings_and_errors.push_back(str);
		throw Lox::Exception(LOX_WHERE, str);
	}

	const std::string comm_str = cfg().get_str(prefix + "_comm_str");
	LOG_MSG("device=" << device_name << ", str=" << comm_str);

	DCB dcb = {0};
	dcb.DCBlength = sizeof(dcb);
	BOOL result = BuildCommDCBA(comm_str.c_str(), &dcb);
	if (result == FALSE)
	{
		const std::string str = "failed to build DCB with \"" + comm_str + "\"";
		warnings_and_errors.push_back(str);
		throw Lox::Exception(LOX_WHERE, str);
	}

	result = SetCommState(comm_handle, &dcb);
	if (result == FALSE)
	{
		const std::string str = "failure in call to SetCommState() for device " + device_name;
		warnings_and_errors.push_back(str);
		throw Lox::Exception(LOX_WHERE, str);
	}

	COMMTIMEOUTS timeouts					= {0};
	timeouts.ReadIntervalTimeout			= 10;
	timeouts.ReadTotalTimeoutConstant		= 10;
	timeouts.ReadTotalTimeoutMultiplier		= 10;
	timeouts.WriteTotalTimeoutConstant		= 10;
	timeouts.WriteTotalTimeoutMultiplier	= 10;
	LOG_MSG(
		"max time between read: "			<< timeouts.ReadIntervalTimeout			<<
		" read multiplier of chars: "		<< timeouts.ReadTotalTimeoutMultiplier	<<
		" read constant in millisecs: "		<< timeouts.ReadTotalTimeoutConstant	<<
		" write multiplier of chars: "		<< timeouts.WriteTotalTimeoutMultiplier	<<
		" write constant in millisecs: "	<< timeouts.WriteTotalTimeoutConstant	);

	result = SetCommTimeouts(comm_handle, &timeouts);
	if (result == FALSE)
	{
		const std::string str = "failure in call to SetCommTimeouts() for device " + device_name;
		warnings_and_errors.push_back(str);
		throw Lox::Exception(LOX_WHERE, str);
	}

	GetCommState(comm_handle, &dcb);
	LOG_MSG(
		"device="				<< device_name					<<
		" Baud="				<< dcb.BaudRate					<<
		" Binary="				<< (int)dcb.fBinary				<<
		" Parity="				<< (int)dcb.fParity				<<
		" OutxCtsFlow="			<< (int)dcb.fOutxCtsFlow		<<
		" OutxDsrFlow="			<< (int)dcb.fOutxDsrFlow		<<
		" DtrControl="			<< (int)dcb.fDtrControl			<<
		" DsrSensitivity="		<< (int)dcb.fDsrSensitivity		<<
		" TXContinueOnXoff="	<< (int)dcb.fTXContinueOnXoff	<<
		" OutX="				<< (int)dcb.fOutX				<<
		" InX="					<< (int)dcb.fInX				<<
		" ErrorChar="			<< (int)dcb.ErrorChar			<<
		" Null="				<< (int)dcb.fNull				<<
		" RtsControl="			<< (int)dcb.fRtsControl			<<
		" AbortOnError="		<< (int)dcb.fAbortOnError		<<
		" XonLim="				<< (int)dcb.XonLim				<<
		" XoffLim="				<< (int)dcb.XoffLim				<<
		" ByteSize="			<< (int)dcb.ByteSize			<<
		" Parity="				<< (int)dcb.Parity				<<
		" StopBits="			<< (int)dcb.StopBits			);

	return;
}


void TabCanvas::close_serial_device(void)
{
	CloseHandle(comm_handle);

	comm_handle = INVALID_HANDLE_VALUE;

	return;
}


static const std::vector<std::string> sample_data =
{
	"14.8\t14.2\r\n", "15.6\t15.0\r\n", "16.0\t15.1\r\n", "15.4\t14.8\r\n", "14.2\t13.5\r\n", "16.4\t15.6\r\n", "28.0\t21.2\r\n", "15.3\t14.4\r\n", "14.4\t13.6\r\n", "14.8\t13.5\r\n",
	"14.5\t13.3\r\n", "15.5\t14.1\r\n", "16.2\t15.1\r\n", "14.8\t13.7\r\n", "15.2\t14.4\r\n", "16.6\t15.9\r\n", "14.1\t13.4\r\n", "23.6\t16.9\r\n", "14.0\t13.3\r\n", "16.3\t15.2\r\n",
	"15.3\t14.7\r\n", "14.8\t14.2\r\n", "15.3\t13.7\r\n", "14.1\t13.4\r\n", "15.5\t14.7\r\n", "14.4\t13.6\r\n", "15.0\t14.1\r\n", "16.3\t15.2\r\n", "15.4\t14.6\r\n", "14.3\t13.8\r\n",
	"14.7\t14.1\r\n", "16.1\t14.9\r\n", "15.4\t14.5\r\n", "16.2\t15.2\r\n", "15.2\t14.5\r\n", "14.4\t13.6\r\n", "15.4\t14.2\r\n", "14.7\t13.9\r\n", "14.6\t13.8\r\n", "14.0\t13.4\r\n",
	"14.1\t13.3\r\n", "15.8\t14.6\r\n", "14.4\t13.6\r\n", "14.1\t13.5\r\n", "14.3\t13.6\r\n", "13.9\t13.2\r\n", "16.4\t15.6\r\n", "14.8\t14.2\r\n", "14.6\t14.0\r\n", "13.6\t12.9\r\n",
	"15.8\t14.9\r\n", "14.4\t13.8\r\n", "14.6\t13.4\r\n", "15.9\t15.0\r\n", "15.6\t14.7\r\n", "14.7\t13.9\r\n", "15.8\t14.8\r\n", "15.6\t14.4\r\n", "13.6\t12.9\r\n", "13.9\t13.4\r\n",
	"14.2\t13.6\r\n", "14.4\t13.5\r\n", "14.2\t13.6\r\n", "15.5\t14.7\r\n", "15.2\t14.6\r\n", "17.9\t16.4\r\n", "16.1\t15.3\r\n", "14.6\t14.1\r\n", "15.2\t14.4\r\n", "15.1\t14.4\r\n",
	"14.4\t13.8\r\n", "16.7\t14.8\r\n", "14.6\t14.0\r\n", "14.8\t14.3\r\n", "13.6\t13.0\r\n", "15.5\t13.9\r\n", "14.3\t13.8\r\n", "16.5\t15.4\r\n", "15.2\t14.2\r\n", "14.6\t13.8\r\n",
	"14.8\t14.2\r\n", "13.6\t13.2\r\n", "13.4\t12.5\r\n", "14.4\t13.6\r\n", "15.4\t14.6\r\n", "17.6\t15.5\r\n", "15.1\t14.4\r\n", "14.9\t14.0\r\n", "14.5\t13.8\r\n", "15.3\t14.5\r\n",
	"15.4\t14.7\r\n", "14.2\t13.5\r\n", "14.5\t13.9\r\n", "14.6\t13.7\r\n", "14.5\t13.9\r\n", "15.4\t14.1\r\n", "15.6\t14.8\r\n", "15.3\t14.6\r\n", "18.3\t16.2\r\n", "14.4\t13.8\r\n",
	"13.9\t13.2\r\n", "14.2\t13.5\r\n", "15.8\t14.9\r\n", "15.4\t14.2\r\n", "17.1\t16.0\r\n", "15.3\t14.4\r\n", "16.7\t15.8\r\n", "14.9\t14.3\r\n", "13.9\t13.3\r\n", "14.8\t14.0\r\n",
	"14.5\t13.9\r\n", "15.9\t15.1\r\n", "13.8\t12.9\r\n", "16.6\t15.6\r\n", "16.0\t15.1\r\n", "14.6\t13.9\r\n", "14.9\t14.1\r\n", "15.6\t14.5\r\n", "13.6\t12.7\r\n", "16.6\t15.2\r\n",
	"14.5\t13.7\r\n", "15.3\t14.8\r\n", "13.8\t13.2\r\n", "13.9\t13.4\r\n", "14.8\t13.9\r\n", "14.9\t14.1\r\n", "15.6\t14.9\r\n", "14.5\t13.7\r\n", "15.4\t14.1\r\n", "16.5\t15.2\r\n",
	"15.0\t14.4\r\n", "15.2\t14.2\r\n", "14.3\t13.5\r\n", "13.9\t13.0\r\n", "14.1\t13.4\r\n", "14.6\t13.7\r\n", "14.8\t14.1\r\n", "17.0\t15.2\r\n", "14.8\t14.2\r\n", "14.8\t14.2\r\n",
	"15.4\t14.7\r\n", "16.2\t15.2\r\n", "17.2\t16.0\r\n", "14.2\t13.5\r\n", "15.9\t14.3\r\n", "15.3\t14.1\r\n", "16.3\t14.4\r\n", "16.1\t14.6\r\n", "14.8\t14.0\r\n", "16.6\t14.8\r\n",
	"16.0\t15.4\r\n", "16.7\t15.9\r\n", "15.0\t14.4\r\n", "14.3\t13.5\r\n", "14.3\t13.8\r\n", "16.2\t15.5\r\n", "14.5\t13.9\r\n", "16.7\t15.6\r\n", "16.8\t15.9\r\n", "14.9\t14.2\r\n",
	"15.0\t14.3\r\n", "15.9\t14.5\r\n", "15.3\t14.6\r\n", "15.2\t14.5\r\n", "15.8\t14.9\r\n", "15.0\t14.0\r\n", "14.4\t13.5\r\n", "14.9\t14.3\r\n", "14.1\t13.5\r\n", "15.2\t14.4\r\n",
	"14.5\t13.8\r\n", "14.8\t14.1\r\n", "14.2\t13.2\r\n", "15.8\t15.2\r\n", "16.5\t15.7\r\n", "17.1\t16.1\r\n", "16.5\t15.6\r\n", "14.8\t14.1\r\n", "15.4\t14.6\r\n", "15.7\t14.8\r\n",
	"14.0\t12.7\r\n", "14.5\t13.7\r\n", "14.7\t14.1\r\n", "15.9\t15.2\r\n", "15.4\t14.7\r\n", "14.6\t14.1\r\n", "14.7\t14.2\r\n", "14.2\t13.5\r\n", "15.8\t15.2\r\n", "14.6\t14.1\r\n",
	"15.0\t14.4\r\n", "16.2\t15.2\r\n", "14.5\t13.8\r\n", "14.9\t14.3\r\n", "15.3\t14.5\r\n", "15.3\t14.4\r\n", "15.9\t15.1\r\n", "14.6\t14.1\r\n", "14.2\t13.6\r\n", "15.2\t14.4\r\n",
	"15.5\t14.0\r\n", "14.8\t14.3\r\n", "14.9\t14.1\r\n", "16.1\t15.3\r\n", "15.7\t14.9\r\n", "14.4\t13.9\r\n", "15.8\t15.1\r\n", "14.9\t14.1\r\n", "14.5\t13.9\r\n", "15.6\t14.3\r\n",
	"15.7\t14.9\r\n", "14.7\t14.1\r\n", "15.9\t14.8\r\n", "14.4\t13.9\r\n", "15.7\t14.9\r\n", "15.0\t14.4\r\n", "15.8\t15.1\r\n", "15.9\t15.2\r\n", "15.2\t14.4\r\n", "21.2\t17.3\r\n",
	"15.2\t14.2\r\n", "15.2\t14.1\r\n", "14.3\t13.7\r\n", "15.7\t14.9\r\n", "16.0\t14.0\r\n", "28.0\t20.4\r\n", "13.6\t12.8\r\n", "15.4\t14.4\r\n", "15.4\t14.5\r\n", "16.1\t15.3\r\n",
	"17.5\t16.3\r\n", "14.6\t13.6\r\n", "14.5\t13.8\r\n", "15.6\t15.0\r\n", "15.3\t14.7\r\n", "15.1\t14.6\r\n", "14.6\t14.0\r\n", "15.8\t15.0\r\n", "13.8\t13.3\r\n", "14.0\t13.2\r\n",
	"15.0\t14.2\r\n", "14.9\t14.2\r\n", "15.6\t14.5\r\n", "14.7\t14.0\r\n", "15.2\t14.4\r\n", "13.5\t12.9\r\n", "14.3\t13.4\r\n", "14.2\t13.5\r\n", "15.2\t14.2\r\n", "14.0\t13.1\r\n",
	"14.7\t14.0\r\n", "16.5\t14.8\r\n", "27.5\t16.4\r\n", "14.2\t13.7\r\n", "14.8\t14.0\r\n", "13.8\t13.1\r\n", "14.4\t13.4\r\n", "13.8\t13.2\r\n", "15.6\t14.7\r\n", "14.6\t13.9\r\n",
	"13.2\t12.7\r\n", "13.9\t13.0\r\n", "13.9\t12.9\r\n", "14.3\t13.6\r\n", "13.8\t13.2\r\n", "13.5\t12.6\r\n", "14.4\t13.7\r\n", "13.8\t13.1\r\n", "15.0\t14.2\r\n", "14.2\t13.4\r\n",
	"14.4\t13.6\r\n", "13.6\t12.9\r\n", "13.5\t12.8\r\n", "13.8\t13.1\r\n", "13.9\t13.3\r\n", "14.1\t13.6\r\n", "15.0\t13.7\r\n", "14.2\t12.6\r\n", "13.0\t12.5\r\n", "13.5\t12.9\r\n",
	"14.7\t13.8\r\n", "14.4\t13.6\r\n", "14.0\t13.3\r\n", "14.0\t13.1\r\n", "13.8\t13.1\r\n", "14.3\t13.3\r\n", "13.3\t12.6\r\n", "15.6\t14.8\r\n", "13.9\t13.1\r\n", "13.9\t13.2\r\n",
	"13.9\t13.0\r\n", "15.1\t14.4\r\n", "13.0\t12.3\r\n", "13.4\t12.5\r\n", "14.0\t13.1\r\n", "13.9\t13.4\r\n", "15.4\t14.7\r\n", "15.5\t14.2\r\n", "14.2\t13.5\r\n", "13.6\t12.6\r\n",
	"14.4\t13.5\r\n", "14.5\t13.6\r\n", "14.0\t13.4\r\n", "14.8\t14.0\r\n", "14.0\t13.4\r\n", "14.0\t13.0\r\n", "13.2\t12.5\r\n", "14.7\t14.1\r\n", "14.5\t13.3\r\n", "14.6\t13.9\r\n",
	"14.5\t13.4\r\n", "13.4\t12.8\r\n", "13.8\t12.9\r\n", "13.7\t13.0\r\n", "12.8\t12.4\r\n", "14.4\t13.7\r\n", "14.9\t14.2\r\n", "13.5\t12.9\r\n", "14.4\t13.9\r\n", "13.6\t13.0\r\n",
	"14.5\t13.6\r\n", "14.6\t13.8\r\n", "14.8\t13.7\r\n", "13.7\t13.0\r\n", "15.1\t14.2\r\n", "15.6\t14.5\r\n", "13.7\t13.0\r\n", "12.8\t12.2\r\n", "14.3\t13.4\r\n", "14.9\t13.6\r\n",
	"14.2\t13.6\r\n", "16.0\t14.4\r\n", "14.8\t13.9\r\n", "15.2\t13.7\r\n", "14.1\t13.4\r\n", "14.3\t13.5\r\n", "13.9\t12.9\r\n", "13.9\t13.4\r\n", "15.6\t14.8\r\n", "13.2\t12.6\r\n",
	"15.6\t14.4\r\n", "14.4\t13.5\r\n", "14.8\t14.1\r\n", "13.1\t12.5\r\n", "14.0\t13.5\r\n", "15.8\t14.6\r\n", "14.5\t12.9\r\n", "16.3\t15.1\r\n", "13.4\t12.8\r\n", "14.9\t13.6\r\n",
	"14.6\t13.8\r\n", "15.7\t13.5\r\n", "15.1\t13.2\r\n", "14.0\t13.5\r\n", "14.2\t13.4\r\n", "14.4\t13.7\r\n", "15.0\t13.9\r\n", "14.1\t13.2\r\n", "14.1\t13.6\r\n", "14.4\t13.3\r\n",
	"14.0\t13.3\r\n", "14.0\t13.3\r\n", "14.8\t14.1\r\n", "14.2\t13.2\r\n", "14.2\t13.6\r\n", "13.5\t12.5\r\n", "14.5\t13.8\r\n", "14.1\t13.3\r\n", "17.0\t15.8\r\n", "15.0\t13.9\r\n",
	"14.1\t13.5\r\n", "14.6\t13.7\r\n", "14.3\t13.5\r\n", "13.8\t13.2\r\n", "15.4\t14.7\r\n", "14.5\t13.7\r\n", "14.0\t13.1\r\n", "15.4\t14.5\r\n", "13.7\t13.2\r\n", "13.6\t13.0\r\n",
	"15.6\t14.8\r\n", "14.4\t13.6\r\n", "15.2\t13.7\r\n", "15.2\t14.3\r\n", "14.8\t13.9\r\n", "13.9\t13.2\r\n", "14.6\t13.4\r\n", "14.5\t13.8\r\n", "14.7\t13.7\r\n", "14.0\t13.4\r\n",
	"14.1\t13.3\r\n", "13.4\t12.8\r\n", "13.3\t12.6\r\n", "14.8\t14.0\r\n", "13.7\t12.9\r\n", "13.9\t13.1\r\n", "14.7\t13.8\r\n", "13.3\t12.6\r\n", "14.1\t13.4\r\n", "14.8\t13.9\r\n",
	"14.5\t13.7\r\n", "13.9\t13.2\r\n", "14.3\t13.7\r\n", "13.6\t13.0\r\n", "14.2\t13.6\r\n", "14.7\t14.1\r\n", "13.8\t13.3\r\n", "13.8\t13.2\r\n", "14.6\t13.9\r\n", "13.5\t12.6\r\n",
	"15.5\t14.5\r\n", "14.6\t13.6\r\n", "14.8\t13.8\r\n", "14.4\t13.5\r\n", "14.0\t13.4\r\n", "13.9\t13.2\r\n", "14.0\t12.9\r\n", "14.0\t13.1\r\n", "14.9\t14.1\r\n", "14.9\t14.2\r\n",
	"14.0\t13.4\r\n", "15.4\t14.4\r\n", "13.8\t12.9\r\n", "13.4\t12.6\r\n", "14.4\t13.2\r\n", "15.7\t14.6\r\n", "15.6\t14.5\r\n", "14.4\t13.6\r\n", "16.9\t15.3\r\n", "14.0\t13.3\r\n",
	"14.2\t13.3\r\n", "15.8\t14.7\r\n", "14.0\t13.1\r\n", "13.7\t12.9\r\n", "13.7\t12.9\r\n", "13.8\t13.0\r\n", "14.5\t14.0\r\n", "14.0\t13.4\r\n", "13.4\t12.7\r\n", "14.0\t13.3\r\n",
	"15.0\t14.2\r\n", "15.1\t13.7\r\n", "13.8\t13.1\r\n", "13.9\t12.8\r\n", "16.1\t15.0\r\n", "13.6\t12.7\r\n", "15.0\t13.8\r\n", "14.3\t13.5\r\n", "14.1\t13.3\r\n", "14.4\t13.8\r\n",
	"13.8\t12.8\r\n", "14.0\t13.1\r\n", "15.7\t14.9\r\n", "13.6\t12.8\r\n", "13.7\t13.1\r\n", "15.1\t14.1\r\n", "13.2\t12.6\r\n", "15.8\t14.2\r\n", "14.4\t13.7\r\n", "13.5\t12.5\r\n",
	"13.3\t12.5\r\n", "13.4\t12.7\r\n", "14.3\t13.2\r\n", "15.4\t14.2\r\n", "14.5\t13.8\r\n", "14.2\t13.5\r\n", "14.2\t13.3\r\n", "13.1\t12.1\r\n", "13.7\t13.1\r\n", "13.2\t12.2\r\n",
	"13.8\t13.0\r\n", "14.7\t12.7\r\n", "12.9\t12.1\r\n", "12.6\t12.0\r\n", "13.4\t12.6\r\n", "15.5\t13.8\r\n", "12.8\t12.2\r\n", "15.5\t14.7\r\n", "12.9\t12.0\r\n", "15.7\t14.9\r\n",
	"12.6\t12.1\r\n", "15.5\t14.2\r\n", "14.5\t13.8\r\n", "15.5\t14.6\r\n", "14.0\t13.3\r\n", "13.9\t12.8\r\n", "15.8\t14.9\r\n", "14.4\t13.5\r\n", "15.4\t14.6\r\n", "13.4\t12.8\r\n",
	"14.4\t13.5\r\n", "14.8\t13.8\r\n", "12.3\t11.5\r\n", "27.4\t15.0\r\n", "13.7\t12.9\r\n", "14.9\t14.1\r\n", "16.3\t15.3\r\n", "14.6\t13.5\r\n", "13.9\t13.0\r\n", "15.4\t14.6\r\n",
	"15.2\t14.4\r\n", "13.7\t13.0\r\n", "15.4\t14.1\r\n", "14.2\t13.4\r\n", "14.0\t13.3\r\n", "15.2\t14.0\r\n", "17.7\t16.5\r\n", "20.2\t14.9\r\n", "14.8\t13.7\r\n", "15.0\t14.1\r\n",
	"13.6\t12.6\r\n", "13.7\t13.2\r\n", "13.6\t12.8\r\n", "13.8\t12.9\r\n", "13.6\t12.9\r\n", "14.3\t13.7\r\n", "16.3\t15.3\r\n", "13.6\t12.8\r\n", "14.2\t13.4\r\n", "14.9\t14.0\r\n",
	"14.6\t13.5\r\n", "13.7\t12.7\r\n", "13.9\t13.0\r\n", "14.0\t12.8\r\n", "14.8\t14.0\r\n", "13.5\t12.7\r\n", "14.4\t13.7\r\n", "15.4\t14.8\r\n", "15.3\t14.5\r\n", "13.9\t13.2\r\n",
	"13.7\t13.2\r\n", "15.0\t14.2\r\n", "16.0\t15.0\r\n", "14.8\t13.9\r\n", "15.2\t14.0\r\n", "14.3\t13.7\r\n", "15.6\t14.5\r\n", "13.8\t12.9\r\n", "13.3\t12.6\r\n", "14.3\t13.4\r\n",
	"14.9\t14.0\r\n", "13.6\t12.9\r\n", "14.4\t13.5\r\n", "14.0\t13.3\r\n", "14.2\t12.8\r\n", "13.1\t12.3\r\n", "14.4\t13.5\r\n", "15.5\t14.4\r\n", "14.6\t13.3\r\n", "16.4\t15.4\r\n",
	"13.0\t12.0\r\n", "13.8\t12.8\r\n", "16.0\t14.9\r\n", "13.6\t12.3\r\n", "15.3\t14.1\r\n", "14.5\t13.5\r\n", "16.3\t15.4\r\n", "13.3\t12.6\r\n", "14.4\t13.7\r\n", "13.6\t12.9\r\n",
	"14.3\t13.2\r\n", "14.2\t13.6\r\n", "15.0\t14.1\r\n", "13.8\t13.3\r\n", "16.0\t14.9\r\n", "14.4\t13.5\r\n", "14.1\t12.9\r\n", "13.7\t13.0\r\n", "15.7\t14.3\r\n", "14.1\t13.3\r\n",
	"13.4\t12.2\r\n", "13.9\t13.0\r\n", "16.5\t15.3\r\n", "14.7\t13.9\r\n", "14.4\t13.5\r\n", "15.3\t14.4\r\n", "13.9\t13.3\r\n", "13.4\t12.6\r\n", "17.3\t16.4\r\n", "14.2\t13.4\r\n",
	"15.3\t14.4\r\n", "15.0\t14.0\r\n", "16.1\t15.0\r\n", "28.0\t21.3\r\n", "15.3\t13.8\r\n", "13.5\t12.8\r\n", "15.2\t13.9\r\n", "13.6\t12.6\r\n", "14.4\t13.3\r\n", "14.9\t14.0\r\n",
	"15.1\t14.2\r\n", "13.8\t13.0\r\n", "14.0\t13.4\r\n", "13.4\t12.3\r\n", "27.8\t14.8\r\n", "14.8\t13.9\r\n", "15.7\t14.6\r\n", "16.5\t15.4\r\n", "13.9\t13.0\r\n", "14.4\t13.1\r\n",
	"15.3\t14.0\r\n", "17.6\t16.2\r\n", "15.6\t14.4\r\n", "14.9\t13.9\r\n", "12.8\t12.1\r\n", "15.6\t14.8\r\n", "15.7\t14.8\r\n", "14.1\t13.3\r\n", "15.3\t13.8\r\n", "15.2\t14.7\r\n",
	"14.1\t13.4\r\n", "14.3\t13.5\r\n", "14.2\t13.3\r\n", "13.9\t13.1\r\n", "12.8\t12.1\r\n", "14.0\t13.3\r\n", "15.2\t14.0\r\n", "16.1\t15.1\r\n", "15.4\t14.2\r\n", "15.0\t14.2\r\n",
	"14.3\t13.7\r\n", "13.8\t12.8\r\n", "13.7\t12.9\r\n", "16.5\t15.2\r\n", "13.8\t13.2\r\n", "14.2\t13.5\r\n", "15.4\t13.9\r\n", "14.6\t13.6\r\n", "14.5\t13.7\r\n", "15.0\t13.9\r\n",
	"13.8\t13.1\r\n", "13.2\t12.5\r\n", "14.1\t13.4\r\n", "14.2\t13.5\r\n", "15.9\t14.8\r\n", "14.7\t13.9\r\n", "13.8\t13.2\r\n", "14.2\t13.6\r\n", "15.2\t13.8\r\n", "14.4\t13.6\r\n",
	"14.9\t14.1\r\n", "14.7\t13.9\r\n", "13.7\t12.8\r\n", "14.4\t13.4\r\n", "14.4\t13.2\r\n", "15.2\t13.9\r\n", "14.0\t13.3\r\n", "14.1\t13.3\r\n", "15.1\t13.9\r\n", "16.3\t14.9\r\n",
	"15.1\t14.1\r\n", "13.9\t13.0\r\n", "13.9\t13.1\r\n", "14.4\t13.8\r\n", "14.8\t13.8\r\n", "14.2\t13.4\r\n", "13.5\t12.8\r\n", "14.7\t13.9\r\n", "15.4\t14.6\r\n", "15.4\t14.2\r\n",
	"17.5\t15.7\r\n", "15.2\t14.2\r\n", "13.2\t12.6\r\n", "14.9\t13.2\r\n", "14.0\t13.2\r\n", "14.6\t13.7\r\n", "13.9\t13.0\r\n", "13.8\t12.9\r\n", "16.9\t16.1\r\n", "15.3\t13.1\r\n",
	"14.1\t13.1\r\n", "13.4\t12.8\r\n", "14.0\t13.3\r\n", "15.3\t14.4\r\n", "13.3\t12.5\r\n", "14.3\t13.4\r\n", "15.2\t14.5\r\n", "13.8\t12.7\r\n", "13.9\t13.2\r\n", "14.7\t13.2\r\n",
	"13.9\t13.0\r\n", "14.8\t13.9\r\n", "16.4\t15.1\r\n", "12.8\t12.1\r\n", "17.5\t16.6\r\n", "13.2\t12.4\r\n", "17.0\t15.9\r\n", "13.5\t12.8\r\n", "14.3\t13.1\r\n", "13.9\t13.2\r\n",
	"14.0\t13.2\r\n", "14.4\t13.7\r\n", "15.2\t14.3\r\n", "14.1\t13.3\r\n", "17.1\t16.0\r\n", "14.5\t13.7\r\n", "14.7\t13.9\r\n", "14.1\t13.1\r\n", "13.7\t13.0\r\n", "15.0\t14.2\r\n",
	"17.5\t16.4\r\n", "14.7\t14.0\r\n", "13.4\t12.4\r\n", "15.3\t14.4\r\n", "16.3\t15.2\r\n", "13.3\t12.6\r\n", "14.8\t13.9\r\n", "14.8\t13.8\r\n", "15.2\t14.3\r\n", "13.9\t12.8\r\n",
	"13.5\t12.9\r\n", "13.5\t12.8\r\n", "13.7\t13.0\r\n", "16.1\t15.2\r\n", "13.7\t13.0\r\n", "14.2\t12.8\r\n", "14.3\t13.7\r\n", "13.5\t12.8\r\n", "14.3\t13.5\r\n", "14.3\t13.3\r\n",
	"14.2\t13.0\r\n", "13.5\t12.8\r\n", "17.0\t15.3\r\n", "15.8\t14.8\r\n", "14.3\t13.4\r\n", "15.0\t14.1\r\n", "16.2\t15.3\r\n", "13.6\t13.0\r\n", "13.9\t12.9\r\n", "14.2\t13.4\r\n",
	"14.3\t13.4\r\n", "13.6\t12.9\r\n", "13.8\t13.1\r\n", "14.8\t14.2\r\n", "13.6\t12.6\r\n", "14.3\t13.6\r\n", "14.7\t13.9\r\n", "12.9\t12.0\r\n", "14.3\t13.3\r\n", "16.3\t15.2\r\n",
	"15.7\t14.3\r\n", "14.0\t13.4\r\n", "14.6\t13.3\r\n", "14.7\t13.5\r\n", "16.9\t14.8\r\n", "13.3\t12.6\r\n", "14.5\t13.8\r\n", "14.6\t13.9\r\n", "14.9\t14.3\r\n", "12.8\t12.1\r\n",
	"14.8\t14.0\r\n", "14.4\t13.6\r\n", "14.5\t13.7\r\n", "17.4\t15.4\r\n", "15.1\t13.9\r\n", "15.2\t14.1\r\n", "15.5\t14.6\r\n", "13.9\t13.1\r\n", "17.1\t15.7\r\n", "15.1\t14.2\r\n",
	"15.4\t14.4\r\n", "14.5\t13.6\r\n", "15.9\t14.9\r\n", "13.4\t12.7\r\n", "15.6\t15.0\r\n", "13.2\t12.4\r\n", "14.1\t13.4\r\n", "14.2\t13.2\r\n", "13.6\t12.8\r\n", "16.6\t14.9\r\n",
	"16.2\t14.9\r\n", "24.5\t15.7\r\n", "14.4\t13.4\r\n", "14.6\t13.8\r\n", "13.3\t12.4\r\n", "13.9\t13.1\r\n", "13.6\t12.9\r\n", "13.8\t13.0\r\n", "15.4\t14.6\r\n", "13.7\t12.9\r\n",
	"14.2\t13.5\r\n", "15.6\t14.6\r\n", "15.1\t14.0\r\n", "16.1\t14.3\r\n", "16.3\t15.1\r\n", "16.0\t15.0\r\n", "15.4\t14.4\r\n", "14.6\t13.6\r\n", "14.7\t13.5\r\n", "13.6\t12.9\r\n",
	"14.8\t13.6\r\n", "16.7\t15.9\r\n", "15.1\t14.0\r\n", "15.5\t14.3\r\n", "15.0\t13.8\r\n", "14.4\t13.9\r\n", "13.9\t13.1\r\n", "14.1\t13.4\r\n", "15.0\t13.3\r\n", "16.1\t15.2\r\n",
	"13.7\t12.9\r\n", "14.2\t13.5\r\n", "14.6\t14.0\r\n", "15.9\t13.5\r\n", "15.2\t14.5\r\n", "13.8\t13.0\r\n", "13.1\t12.4\r\n", "16.9\t14.0\r\n", "15.8\t14.8\r\n", "14.3\t13.6\r\n",
	"13.9\t13.3\r\n", "14.9\t14.3\r\n", "13.7\t13.2\r\n", "14.6\t14.0\r\n", "13.1\t12.6\r\n", "14.0\t13.2\r\n", "13.8\t13.1\r\n", "13.7\t12.7\r\n", "16.0\t15.1\r\n", "14.6\t13.8\r\n",
	"13.2\t12.6\r\n", "14.9\t13.7\r\n", "13.8\t13.1\r\n", "13.8\t13.1\r\n", "13.3\t12.5\r\n", "14.2\t12.5\r\n", "17.9\t16.2\r\n", "14.0\t13.1\r\n", "13.4\t12.7\r\n", "16.8\t15.7\r\n",
	"13.7\t13.0\r\n", "16.3\t14.8\r\n", "13.5\t12.7\r\n", "16.4\t14.8\r\n", "14.8\t14.0\r\n", "17.4\t15.5\r\n", "16.7\t15.5\r\n", "14.7\t13.9\r\n", "14.4\t13.6\r\n", "13.6\t12.8\r\n",
	"14.1\t13.2\r\n", "15.4\t14.6\r\n", "16.0\t15.1\r\n", "14.6\t13.7\r\n", "13.5\t12.7\r\n", "14.7\t13.7\r\n", "13.0\t12.4\r\n", "14.2\t13.4\r\n", "14.3\t13.3\r\n", "14.4\t13.3\r\n",
	"14.9\t14.0\r\n", "13.6\t13.0\r\n", "15.0\t14.3\r\n", "14.9\t13.8\r\n", "15.7\t14.1\r\n", "13.9\t13.3\r\n", "14.4\t13.7\r\n", "14.4\t13.7\r\n", "13.8\t13.1\r\n", "15.0\t13.7\r\n",
	"13.5\t12.9\r\n", "14.0\t13.2\r\n", "15.3\t14.2\r\n", "15.7\t14.7\r\n", "14.5\t13.2\r\n", "15.1\t13.9\r\n", "13.6\t12.9\r\n", "15.0\t14.0\r\n", "14.0\t13.3\r\n", "15.0\t13.8\r\n",
	"14.2\t13.4\r\n", "13.2\t12.4\r\n", "14.0\t13.4\r\n", "14.4\t13.6\r\n", "15.5\t14.6\r\n", "14.8\t14.1\r\n", "13.2\t12.5\r\n", "13.7\t13.1\r\n", "14.0\t13.0\r\n", "14.9\t14.0\r\n",
	"18.6\t17.2\r\n", "16.8\t15.2\r\n", "14.1\t13.0\r\n", "15.6\t14.5\r\n", "13.9\t13.0\r\n", "13.8\t13.0\r\n", "15.5\t14.6\r\n", "13.9\t13.1\r\n", "14.0\t13.5\r\n", "14.6\t14.0\r\n",
	"14.4\t13.4\r\n", "13.7\t12.8\r\n", "13.6\t13.0\r\n", "14.2\t13.2\r\n", "15.2\t14.2\r\n", "15.6\t14.6\r\n", "13.9\t13.0\r\n", "14.6\t13.5\r\n", "14.8\t13.9\r\n", "15.8\t14.6\r\n",
	"13.4\t12.6\r\n", "13.9\t12.7\r\n", "15.5\t14.7\r\n", "14.0\t13.2\r\n", "14.7\t13.9\r\n", "14.8\t14.0\r\n", "14.9\t14.3\r\n", "14.2\t13.5\r\n", "13.2\t12.7\r\n", "13.9\t13.3\r\n",
	"15.2\t14.1\r\n", "14.6\t13.9\r\n", "13.9\t13.3\r\n", "14.4\t13.7\r\n", "14.8\t13.5\r\n", "15.6\t14.6\r\n", "14.4\t13.3\r\n", "13.9\t13.2\r\n", "14.3\t13.6\r\n", "16.1\t14.3\r\n",
	"13.6\t12.8\r\n", "15.7\t14.8\r\n", "13.8\t13.0\r\n", "14.8\t14.2\r\n", "15.0\t14.2\r\n", "16.2\t15.0\r\n", "15.9\t14.4\r\n", "14.7\t13.6\r\n", "13.9\t13.3\r\n", "13.9\t13.0\r\n",
	"14.5\t13.7\r\n", "14.3\t13.6\r\n", "16.4\t15.3\r\n", "15.1\t14.2\r\n", "13.9\t13.2\r\n", "14.2\t13.4\r\n", "14.6\t13.8\r\n", "14.8\t14.0\r\n", "15.9\t15.0\r\n", "14.2\t13.4\r\n",
	"15.1\t14.3\r\n", "14.8\t14.1\r\n", "14.4\t13.6\r\n", "13.3\t12.8\r\n", "13.7\t13.0\r\n", "14.6\t13.7\r\n", "14.1\t13.4\r\n", "14.3\t13.5\r\n", "14.8\t13.8\r\n", "14.8\t14.0\r\n",
	"13.6\t13.0\r\n", "14.2\t13.4\r\n", "14.8\t13.2\r\n", "15.0\t14.2\r\n", "13.5\t12.9\r\n", "13.5\t12.9\r\n", "15.2\t14.5\r\n", "13.9\t13.2\r\n", "13.4\t12.8\r\n", "14.9\t14.1\r\n",
	"15.2\t14.0\r\n", "14.6\t14.0\r\n", "14.9\t14.0\r\n", "14.6\t13.5\r\n", "14.0\t13.5\r\n", "12.8\t12.2\r\n", "14.2\t13.4\r\n", "14.3\t13.6\r\n", "17.4\t15.7\r\n", "15.1\t14.3\r\n",
	"14.9\t14.1\r\n", "13.8\t13.1\r\n", "13.4\t12.8\r\n", "14.6\t13.7\r\n", "14.2\t13.4\r\n", "13.5\t12.9\r\n", "15.2\t14.1\r\n", "13.5\t12.8\r\n", "13.9\t13.1\r\n", "13.2\t12.6\r\n",
	"13.1\t12.5\r\n", "14.2\t13.7\r\n", "13.8\t13.0\r\n", "13.9\t13.3\r\n", "12.9\t12.4\r\n", "14.0\t13.4\r\n", "14.2\t13.4\r\n", "13.8\t13.3\r\n", "27.9\t20.2\r\n", "13.4\t12.9\r\n",
	"13.6\t12.9\r\n", "14.2\t13.5\r\n", "14.6\t13.9\r\n", "14.0\t13.1\r\n", "15.7\t14.9\r\n", "13.7\t13.1\r\n", "13.6\t12.7\r\n", "14.1\t13.3\r\n", "13.8\t13.2\r\n", "12.9\t12.2\r\n",
	"13.7\t13.0\r\n", "13.7\t13.1\r\n", "14.4\t13.7\r\n", "13.8\t12.4\r\n", "13.5\t12.9\r\n", "13.3\t12.6\r\n", "13.4\t12.6\r\n", "13.4\t13.0\r\n", "15.3\t14.3\r\n", "13.5\t12.9\r\n",
	"13.2\t12.6\r\n", "14.4\t13.8\r\n", "13.5\t12.7\r\n", "13.8\t13.2\r\n", "15.5\t14.4\r\n", "13.2\t12.5\r\n", "14.7\t14.1\r\n", "13.4\t12.8\r\n", "13.4\t12.6\r\n", "13.1\t12.4\r\n",
	"14.2\t13.5\r\n", "15.8\t14.8\r\n", "14.0\t12.9\r\n", "14.2\t13.3\r\n", "14.2\t13.6\r\n", "14.0\t13.3\r\n", "14.2\t13.7\r\n", "18.8\t15.4\r\n", "13.9\t12.9\r\n", "14.7\t13.7\r\n",
	"15.1\t14.5\r\n", "13.6\t12.8\r\n", "14.7\t14.1\r\n", "14.0\t13.4\r\n", "16.8\t15.9\r\n", "16.6\t15.4\r\n", "14.3\t12.9\r\n", "14.9\t14.0\r\n", "14.0\t13.3\r\n", "13.2\t12.3\r\n",
	"14.3\t13.7\r\n", "14.7\t13.9\r\n", "14.8\t13.9\r\n", "13.7\t12.9\r\n", "13.9\t13.1\r\n", "13.9\t12.8\r\n", "16.3\t14.4\r\n", "13.8\t13.2\r\n", "13.6\t13.1\r\n", "14.0\t13.2\r\n",
	"14.1\t13.5\r\n", "14.7\t13.9\r\n", "13.6\t13.0\r\n", "15.2\t14.6\r\n", "14.6\t14.0\r\n", "14.8\t14.2\r\n", "13.9\t13.3\r\n", "14.4\t13.7\r\n", "13.8\t13.3\r\n", "14.0\t13.2\r\n",
	"15.5\t14.6\r\n", "13.8\t13.0\r\n", "13.5\t12.7\r\n", "14.1\t13.5\r\n", "13.9\t13.1\r\n", "15.8\t14.7\r\n", "13.2\t12.4\r\n", "13.5\t12.8\r\n", "14.8\t13.7\r\n", "14.4\t13.5\r\n",
	"14.1\t13.4\r\n", "13.6\t12.7\r\n", "14.4\t13.7\r\n", "13.7\t13.1\r\n", "14.4\t13.7\r\n", "14.1\t13.5\r\n", "14.7\t13.8\r\n", "14.0\t13.1\r\n", "14.0\t13.2\r\n", "13.7\t13.0\r\n",
	"28.0\t21.3\r\n", "14.2\t13.7\r\n", "13.3\t12.6\r\n", "14.1\t13.4\r\n", "15.6\t14.3\r\n", "14.7\t14.0\r\n", "15.3\t14.4\r\n", "13.9\t13.1\r\n", "16.1\t15.2\r\n", "13.4\t12.6\r\n",
	"13.6\t12.8\r\n", "14.1\t13.3\r\n", "14.0\t13.5\r\n", "13.4\t12.5\r\n", "13.8\t13.2\r\n", "13.4\t12.6\r\n", "13.8\t13.2\r\n", "14.2\t13.7\r\n", "13.2\t12.5\r\n", "14.3\t13.5\r\n",
	"15.0\t14.0\r\n", "13.3\t12.7\r\n", "14.9\t13.7\r\n", "14.6\t13.9\r\n", "14.2\t13.0\r\n", "14.0\t13.4\r\n", "14.3\t13.7\r\n", "14.0\t13.1\r\n", "15.2\t14.5\r\n", "14.4\t13.1\r\n",
	"13.8\t13.3\r\n", "14.3\t13.5\r\n", "14.1\t13.4\r\n", "13.8\t13.3\r\n", "13.2\t12.5\r\n", "12.8\t12.0\r\n", "13.0\t12.4\r\n", "16.9\t15.7\r\n", "13.1\t12.5\r\n", "14.0\t13.1\r\n",
	"14.9\t14.1\r\n", "13.3\t12.8\r\n", "15.3\t14.6\r\n", "13.2\t12.5\r\n", "14.2\t13.6\r\n", "13.9\t13.4\r\n", "14.4\t13.6\r\n", "13.1\t12.4\r\n", "13.9\t13.3\r\n", "14.5\t13.8\r\n",
	"13.8\t13.3\r\n", "15.0\t13.9\r\n", "23.7\t17.2\r\n", "14.4\t13.6\r\n", "13.9\t12.9\r\n", "14.0\t13.2\r\n", "13.9\t13.2\r\n", "15.2\t13.8\r\n", "14.7\t13.5\r\n", "15.6\t14.7\r\n",
	"13.8\t13.2\r\n", "14.7\t14.1\r\n", "16.3\t15.4\r\n", "14.2\t13.4\r\n", "15.4\t14.2\r\n", "15.1\t14.3\r\n", "13.2\t12.6\r\n", "14.3\t13.7\r\n", "13.5\t12.9\r\n", "14.4\t13.4\r\n",
	"13.6\t13.0\r\n", "14.5\t13.5\r\n", "13.2\t12.5\r\n", "15.0\t14.2\r\n", "13.8\t13.2\r\n", "13.9\t13.2\r\n", "14.8\t14.0\r\n", "14.5\t13.8\r\n", "14.2\t12.7\r\n", "13.8\t12.5\r\n",
	"14.6\t13.9\r\n", "14.6\t13.9\r\n", "16.1\t15.2\r\n", "15.6\t14.3\r\n", "14.6\t13.9\r\n", "14.4\t13.8\r\n", "14.2\t13.5\r\n", "13.5\t12.8\r\n", "14.9\t14.2\r\n", "14.3\t13.7\r\n",
	"14.0\t13.1\r\n", "13.1\t12.6\r\n", "24.6\t13.3\r\n", "14.6\t13.2\r\n", "14.4\t13.2\r\n", "14.1\t13.4\r\n", "14.8\t13.9\r\n", "13.3\t12.8\r\n", "13.9\t13.2\r\n", "14.1\t13.4\r\n",
	"13.8\t13.3\r\n", "14.6\t13.5\r\n", "13.2\t12.4\r\n", "14.7\t14.0\r\n", "13.8\t13.2\r\n", "14.5\t13.8\r\n", "13.7\t13.0\r\n", "13.6\t13.0\r\n", "15.0\t14.2\r\n", "14.9\t13.9\r\n",
	"13.0\t12.5\r\n", "14.5\t13.7\r\n", "15.2\t14.2\r\n", "13.7\t12.9\r\n", "14.3\t13.5\r\n", "13.0\t12.4\r\n", "15.0\t14.1\r\n", "14.2\t13.3\r\n", "14.2\t13.4\r\n", "15.1\t14.4\r\n",
	"14.1\t13.6\r\n", "13.2\t12.6\r\n", "13.3\t12.6\r\n", "13.2\t12.4\r\n", "14.9\t14.3\r\n", "14.1\t13.6\r\n", "14.2\t13.4\r\n", "14.1\t13.3\r\n", "14.1\t13.2\r\n", "13.0\t12.5\r\n",
	"13.8\t13.2\r\n", "14.0\t13.4\r\n", "14.0\t13.2\r\n", "13.2\t12.6\r\n", "16.2\t15.3\r\n", "14.1\t12.7\r\n", "13.7\t13.0\r\n", "13.2\t12.6\r\n", "13.2\t12.7\r\n", "16.4\t15.6\r\n",
	"14.3\t13.2\r\n", "13.6\t13.0\r\n", "14.6\t13.9\r\n", "14.1\t13.5\r\n", "13.9\t13.4\r\n", "14.1\t13.4\r\n", "14.5\t13.9\r\n", "13.4\t12.7\r\n", "13.3\t12.7\r\n", "13.9\t13.2\r\n",
	"13.7\t13.1\r\n", "13.3\t12.6\r\n", "13.6\t12.9\r\n", "13.6\t12.8\r\n", "14.1\t13.5\r\n", "13.7\t13.0\r\n", "14.4\t13.7\r\n", "15.0\t14.5\r\n", "14.4\t13.8\r\n", "12.4\t11.8\r\n",
	"13.5\t12.9\r\n", "13.6\t13.0\r\n", "15.7\t14.6\r\n", "14.0\t13.2\r\n", "13.6\t12.9\r\n", "15.0\t14.1\r\n", "13.6\t12.7\r\n", "13.8\t12.9\r\n", "13.3\t12.3\r\n", "14.4\t13.2\r\n",
	"14.0\t13.4\r\n", "13.4\t12.8\r\n", "13.3\t12.7\r\n", "13.0\t12.5\r\n", "15.8\t14.8\r\n", "13.5\t12.9\r\n", "15.4\t13.9\r\n", "13.2\t12.5\r\n", "14.7\t13.9\r\n", "14.0\t13.3\r\n",
	"14.1\t13.4\r\n", "15.7\t14.9\r\n", "13.5\t12.9\r\n", "13.6\t12.5\r\n", "14.3\t13.6\r\n", "13.4\t12.8\r\n", "13.5\t12.8\r\n", "13.6\t12.8\r\n", "14.0\t13.4\r\n", "14.3\t13.5\r\n",
	"13.4\t12.8\r\n", "13.9\t13.1\r\n", "12.6\t12.1\r\n", "13.8\t13.2\r\n", "13.5\t12.4\r\n", "13.4\t12.8\r\n", "15.4\t14.4\r\n", "13.8\t13.0\r\n", "13.5\t12.8\r\n", "14.8\t13.1\r\n"
};


std::string TabCanvas::read_from_device(void)
{
#if 1
	DWORD number_of_bytes_read = 0;
	ReadFile(comm_handle, read_buffer.data(), read_buffer.size(), &number_of_bytes_read, nullptr);
	read_buffer[number_of_bytes_read] = '\0';

	return std::string(reinterpret_cast<char*>(read_buffer.data()), number_of_bytes_read);
#else
	static thread_local size_t idx = 0;
	const std::string str = sample_data[idx];
	idx++;
	if (idx >= sample_data.size())
	{
		idx = 0;
	}

	return str;
#endif
}

