/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_BeagleBone_LEDControl.cpp 1732 2016-04-01 05:22:06Z stephane $
 */

#include "sg_BeagleBone_LEDControl.hpp"
#include "sg_LED_patterns.hpp"

#include <chrono>
#include <sys/stat.h>
#include <fstream>
#include <vector>
#include <algorithm>


SG::BeagleBone::LEDControl::~LEDControl( void )
{
	blink_loop_should_continue = false;
	stop_blink_thread();

	if (valid)
	{
		switch (action_on_destruction)
		{
			case ActionOnDestruction::turn_on_all_leds:			toggle( LED::all, true );										break;
			case ActionOnDestruction::turn_off_all_leds:		toggle( LED::all, false );										break;
			case ActionOnDestruction::restore_default_triggers:	toggle( LED::all, false );	set_default_trigger( LED::all );	break;
			default:																											break;
		}
	}

	return;
}


SG::BeagleBone::LEDControl::LEDControl( void ) :
		blink_loop_should_continue		( true	),
		blink_interval_in_milliseconds	( 250	),
		blink_step_index				( 0		),
		action_on_destruction			( ActionOnDestruction::restore_default_triggers ),
		valid							( false	)
{
	/** The constructor:
	 *
	 * * ...does not start the background blink loop thread.  To do so, you'll need to call @ref start_blink_thread().
	 * * ...does not change the trigger on any of the LEDs.  To do so, you'll need to call @ref set_trigger().
	 * * ...calls @ref is_valid() to automatically detect the %BeagleBone LEDs.  See @ref error_message.
	 * * ...loads (but does not automatically start) the @ref LEDPattern::none blink pattern, which is not an exciting pattern to watch since all it does is turn off the LEDs.  To load a different pattern, you'll need to call @ref load_pattern().
	 */

	is_valid();

	load_pattern( LEDPattern::none );

	return;
}


bool SG::BeagleBone::LEDControl::is_valid( void )
{
	if (valid == true && led_base_path.empty() == false)
	{
		// we already know things are OK
		return true;
	}

	/** Depending on the distro or version of the linux software installed, look for the LEDs in one of several
	 * possible locations.  From past experience, look for something like one of these:
	 *
	 *	* @p "/sys/class/leds/beaglebone:green:usr0/brightness"
	 *	* @p "/sys/devices/leds/leds/beaglebone:green:usr0/brightness"
	 *	* @p "/sys/devices/platform/leds/leds/beaglebone:green:usr0/brightness"
	 *	* @p "/sys/devices/ocp.3/gpio-leds.8/leds/beaglebone:green:usr0/brightness"
	 */

	const std::vector<std::string> v =
	{
		"/sys/class/leds"						,
		"/sys/devices/leds/leds"				,
		"/sys/devices/platform/leds/leds"		,
		"/sys/devices/ocp.3/gpio-leds.8/leds"
	};

	// try each of the paths, one at a time, until we find something that matches
	for (const std::string path : v)
	{
		led_base_path = path;
		const std::string directory_to_check = get_base_path(LED::usr0);

		struct stat buf;
		const int rc = stat( directory_to_check.c_str(), &buf );
		if (rc == 0)
		{
			if (S_ISDIR(buf.st_mode))
			{
				// we found a valid path, stop looking for more
				valid = true;
				break;
			}
		}
	}

	if (! valid)
	{
		led_base_path.clear();

		if (error_message.empty())
		{
			/// @note This method will store a text message in @ref error_message if the LEDs cannot be found.
			error_message = "Cannot find the BeagleBone LEDs in any of the usual places in /sys/...";
		}
	}

	return valid;
}


SG::BeagleBone::LEDControl &SG::BeagleBone::LEDControl::toggle( const SG::BeagleBone::LED led, const bool enabled )
{
	/** @param [in] led If @p led is set to @ref LED::all, then all 4 LEDs will be toggled together.
	 *  @param [in] enabled Set @p enabled to @p true to turn on a LED, or set to @p false to turn it off.
	 */

	if (! is_valid())
	{
		// nothing to do, looks like we don't have BB LEDs
		return *this;
	}

	if (led == LED::all)
	{
		toggle( LED::usr0, enabled );
		toggle( LED::usr1, enabled );
		toggle( LED::usr2, enabled );
		toggle( LED::usr3, enabled );

		return *this;
	}

	if (is_pseudo_LED(led))
	{
		// invalid LED, nothing we can do
		return *this;
	}

	const std::string led_filename = get_base_path(led) + "/brightness";
	std::ofstream of( led_filename, std::ofstream::trunc );
	if (of.good())
	{
		of << (enabled ? "1" : "0");
		of.close();
	}
	else
	{
		if (error_message.empty())
		{
			/** @note This method will store a text message in @ref error_message if the LED @p brightness file cannot
			* be written.  (Verify file or user permissions?)
			*/
			error_message = "Cannot write to " + led_filename + ".";
		}
	}

	return *this;
}


std::string SG::BeagleBone::LEDControl::get_base_path( const SG::BeagleBone::LED led )
{
	if (is_pseudo_LED(led))
	{
		// invalid LED, nothing we can do
		return "";
	}

	/// For example, this could return @p "/sys/devices/platform/leds/leds/beaglebone:green:usr2".

	return led_base_path + "/beaglebone:green:usr" + std::to_string(index_of(led));
}


SG::BeagleBone::LEDControl &SG::BeagleBone::LEDControl::set_trigger( const SG::BeagleBone::LED led, const SG::BeagleBone::Trigger trigger )
{
	/** @param [in] led If @p led is set to @ref LED::all, then the trigger for all 4 LEDs will be set.
	 *  @param [in] trigger Value to use for the trigger.  To prevent the LEDs from automatically changing, use @ref Trigger::none.
	 */

	if (! is_valid())
	{
		// nothing to do, looks like we don't have BB LEDs
		return *this;
	}

	if (led == LED::all)
	{
		set_trigger( LED::usr0, trigger );
		set_trigger( LED::usr1, trigger );
		set_trigger( LED::usr2, trigger );
		set_trigger( LED::usr3, trigger );
		
		return *this;
	}

	if (is_pseudo_LED(led))
	{
		// invalid LED, nothing we can do
		return *this;
	}

	const std::string trigger_filename = get_base_path(led) + "/trigger";
	std::ofstream of( trigger_filename, std::ofstream::trunc );
	if (of.good())
	{
		of << to_string(trigger) << std::endl;
		of.close();
	}
	else
	{
		if (error_message.empty())
		{
			/** @note This method will store a text message in @ref error_message if the LED @p trigger file cannot
			 * be written.  (Verify file or user permissions?)
			 */
			error_message = "Cannot write to " + trigger_filename + ".";
		}
	}

	return *this;
}


SG::BeagleBone::LEDControl &SG::BeagleBone::LEDControl::set_default_trigger( const SG::BeagleBone::LED led )
{
	if (! is_valid())
	{
		// nothing to do, looks like we don't have BB LEDs
		return *this;
	}
	
	if (led == LED::all)
	{
		/// @param [in] led If @p led is set to @ref LED::all, then all 4 LEDs will be reset back to their default values.
		set_default_trigger( LED::usr0 );
		set_default_trigger( LED::usr1 );
		set_default_trigger( LED::usr2 );
		set_default_trigger( LED::usr3 );
		
		return *this;
	}
	
	if (is_pseudo_LED(led))
	{
		// invalid LED, nothing we can do
		return *this;
	}

	/// @see @ref SG::BeagleBone::default_trigger()

	return set_trigger( led, default_trigger(led) );
}


SG::BeagleBone::LEDControl &SG::BeagleBone::LEDControl::load_pattern( const BlinkPattern &pattern )
{
	// don't reset the pattern or the index if we don't have to, cause it disrupts the pattern
	if (blink_pattern != pattern)
	{
		// make a full copy of the pattern, and restart at the very first step
		blink_pattern		= pattern;
		blink_step_index	= 0;
	}

	return *this;
}


SG::BeagleBone::LEDPattern SG::BeagleBone::LEDControl::load_random_pattern( const size_t minimum_number_of_steps )
{
	size_t number_of_steps = minimum_number_of_steps;

	if (vector_of_random_pattern_enum_values.empty())
	{
		// since this is our first time through, we're going to create a vector of all the possible patterns

		// determine the range of enum values used to describe patterns, and insert each one into the static vector
		const size_t min_pattern = static_cast<size_t>( LEDPattern::min );
		const size_t max_pattern = static_cast<size_t>( LEDPattern::max );
		for ( size_t idx = min_pattern; idx < max_pattern; idx ++ )
		{
			vector_of_random_pattern_enum_values.push_back(idx);
		}

		// note that we only need to randomize the vector once, not every time this method is called
		std::random_shuffle( vector_of_random_pattern_enum_values.begin(), vector_of_random_pattern_enum_values.end() );

		/* If the minimum_number_of_steps is set too high, we'll never find a pattern that matches, and we'll end
		 * up emptying the vector without finding anything.  To prevent this from happening forever, now that we know
		 * we've re-populated the 'random' vector, set the number of steps to use for next time to '2'.  If we do
		 * end up emptying the vector immediately, then the next time we recurse, we can ensure the mininum is set
		 * to a more appropriate value.
		 */
		number_of_steps = 2;
	}

	// start picking items off the end of the 'random' vector until we find one that satisfies our needs
	while (! vector_of_random_pattern_enum_values.empty())
	{
		const size_t value = vector_of_random_pattern_enum_values.back();
		vector_of_random_pattern_enum_values.pop_back();

		const LEDPattern pattern_enum = static_cast<LEDPattern>(value);
		const BlinkPattern pattern = LED_get_pattern( pattern_enum );

		// pattern must have a minimum number of steps
		if (pattern.size() >= minimum_number_of_steps)
		{
			// pattern must not be the one we're already using
			if (pattern != blink_pattern)
			{
				// we found a new "random" pattern we can use!

				load_pattern( pattern );

				/// @returns This method returns the enum of the pattern that was loaded.
				return pattern_enum;
			}
		}
	}

	/* If we get here, it means we've completely emptied the vector.  Use recursion to start again at the top at
	 * which time we'll notice the empty vector and re-polualte it with all of the possible pattern enum values.
	 */
	return load_random_pattern( number_of_steps );
}


SG::BeagleBone::LEDControl &SG::BeagleBone::LEDControl::invert_pattern( void )
{
	const BlinkPattern old_pattern = blink_pattern;
	const BlinkPattern new_pattern = LED_invert_pattern( old_pattern );

	return load_pattern( new_pattern );
}


SG::BeagleBone::LEDControl &SG::BeagleBone::LEDControl::start_blink_thread( void )
{
	if (! is_valid())
	{
		// nothing to do, looks like we don't have BB LEDs
		return *this;
	}

	if (blink_thread.joinable() == false)
	{
		blink_loop_should_continue = true;
		blink_thread = std::thread( &SG::BeagleBone::LEDControl::blink_loop, this );
	}

	return *this;
}


SG::BeagleBone::LEDControl &SG::BeagleBone::LEDControl::stop_blink_thread( void )
{
	blink_loop_should_continue = false;

	if (blink_thread.joinable())
	{
		/// This method doesn't return until the blink thread has stopped.
		blink_thread.join();
	}

	return *this;
}


void SG::BeagleBone::LEDControl::blink_loop( void )
{
	// this is normally called on a secondary thread; see start_blink_thread()

	/// @note All triggers are set to @ref Trigger::none by this method to ensure that the LEDs are under manual control.
	set_trigger( LED::all, Trigger::none );
	
	while (valid && blink_loop_should_continue)
	{
		const auto next_step_time = std::chrono::high_resolution_clock::now() + std::chrono::milliseconds(blink_interval_in_milliseconds);

		/* We don't want to sleep for long periods of time since we want to react quickly if the the blink thread loop
		 * needs to exit.  So break the sleep periods into smaller chunks, and check the "continue" flag after each
		 * small sleep period.
		 */
		while (blink_loop_should_continue)
		{
			const auto now = std::chrono::high_resolution_clock::now();
			if (now >= next_step_time)
			{
				// it is time to move to the next step in the blink pattern
				break;
			}

			// if we get here, then we haven't slept for long enough, so figure out in milliseconds how much time remains

			const auto time_remaining			= next_step_time - now;
			const size_t milliseconds_remaining	= std::chrono::duration_cast<std::chrono::milliseconds>(time_remaining).count();
			const size_t milliseconds_to_sleep	= std::min( milliseconds_remaining, static_cast<size_t>(100) );

			std::this_thread::sleep_for( std::chrono::milliseconds(milliseconds_to_sleep) );
		}

		if (blink_loop_should_continue)
		{
			// if we get here, then it is time to move to the next "blink" step in the LED pattern

			BlinkPattern pattern = blink_pattern;
			blink_step_index ++;
			if (blink_step_index >= pattern.size())
			{
				// start back at the beginning of the pattern
				blink_step_index = 0;
			}
			const size_t step_index = blink_step_index;

			const PatternStep &step = pattern[step_index];

			for ( const auto iter : step )
			{
				const LED led		= iter.first;
				const bool enabled	= iter.second;
				toggle( led, enabled );
			}
		}
	}

	return;
}
