/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_103020013_I2CADC.hpp 1731 2016-03-31 13:10:51Z stephane $
 */

#pragma once

#include "sg_GroveI2C.hpp"


namespace SG
{
	/** Analog-to-digital converter for analog I2C sensors.
	 *
	 * Apparently, the default I2C address for this twig used to be @p 0x55.  This conflicts with the BeagleBone Green
	 * EEPROM which also uses @p i2c-1 addresses @p 0x54 through @p 0x57.
	 *
	 * For example, do not plug any I2C Groves and run the @p i2cdetect command:
	 *
	 * ~~~~
	 * sudo i2cdetect -r -y 1
	 *     0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f
	 * 00:          -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 10: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 20: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 30: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 40: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 50: -- -- -- -- UU UU UU UU -- -- -- -- -- -- -- -- 
	 * 60: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 70: -- -- -- -- -- -- -- --                         
	 * ~~~~
	 *
	 * If your ADC indicates it is using address @p 0x55, then locate the labels for @p ADDR0 and @p ADDR1,
	 * along with @p H and @p L at the bottom of the ADC.  You'll need to cut the trace to use a different address.
	 * See http://www.seeedstudio.com/wiki/Grove_-_I2C_ADC#With_Beaglebone_Green for details.
	 *
	 * Connect the ADC to the left-hand @p J4 I2C grove interface, and run the @p i2cdetect command to find the address
	 * of your ADC:
	 *
	 * ~~~~
	 * sudo i2cdetect -r -y 1
	 *     0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f
	 * 00:          -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 10: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 20: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 30: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 40: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 50: 50 -- -- -- UU UU UU UU -- -- -- -- -- -- -- -- 
	 * 60: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- 
	 * 70: -- -- -- -- -- -- -- --                         
	 * ~~~~
	 *
	 * In the above example, the ADC is at address @p 0x50.
	 *
	 * Description | Image
	 * ------------|------
	 * I2C ADC with a default address of @p 0x50. | @image html sg_103020013_I2CADC.jpg
	 * I2C ADC with a temperature sensor.<br/>@p J1 connects to the BeagleBone Green, and @p J2 connects to the analog I2C twig. | @image html sg_101020015_TemperatureSensor.jpg
	 *
	 * @see http://www.seeedstudio.com/wiki/Grove_-_I2C_ADC
	 * @see http://www.seeedstudio.com/depot/Grove-I2C-ADC-p-1580.html
	 */
	class I2CADC : public GroveI2C
	{
		public:

			/// I2CADC registers.  @see ADC121C021 datasheet page 17
			enum class ERegister
			{
				kConversionResult	= 0,	///< 0 (this register is read-only)
				kAlertStatus		= 1,	///< 1
				kConfiguration		= 2,	///< 2
				kLowLimit			= 3,	///< 3
				kHighLimit			= 4,	///< 4
				kHysteresis			= 5,	///< 5
				kLowestConversion	= 6,	///< 6
				kHighestConversion	= 7		///< 7
			};

			/// Destructor.
			virtual ~I2CADC( void );

			/// Constructor requires the I2C address to use.
			I2CADC( const uint8_t addr, const std::string &n="" );

			/// Test object equality. @{
			bool operator==( const I2CADC &rhs ) const;
			bool operator!=( const I2CADC &rhs ) const { return ! operator==(rhs); }
			/// @}

			/** Reset the I2CADC configuration.  This stores a default value into every register, similar to when it is
			 * first powered on.
			 */
			I2CADC &reset(void);

			/** The I2CADC can be configured to perform auto conversions.  When automatic mode is disabled, the I2CADC
			 * can enter low-power mode, and a conversion is only started whenever the result register (register #0) is
			 * read.
			 *
			 * Disabling automatic mode is equivalent to calling @ref enable_automatic_mode() with a value of zero.
			 *
			 * @note The Vmin and Vmax conversion registers (#6 and #7) which store the lowest and highest values seen
			 * by the I2CADC are <em>not</em> updated when automatic mode has been disabled.
			 *
			 * @see @ref enable_automatic_mode()
			 */
			I2CADC &disable_automatic_mode(void) { return enable_automatic_mode(0); }

			/** The I2CADC can be configured to perform auto conversions.  When enabled, the I2CADC will not go into
			 * low-power mode, but instead will periodically perform analog-to-digital conversions, set alerts if
			 * necessary, remember high and low values, and store the most recent result into the conversion result
			 * register.
			 *
			 * @param [in] value Determines the conversion interval.  Must be between 0 and 7 (total of 3 bits).  When
			 * auto-conversion is enabled, the I2CADC can be configured to perform between 0.4 and 27 thousands samples
			 * per second (ksps).
			 *
			 * Value	| Binary	| Meaning
			 * ---------|-----------|--------
			 * @p 0		| @p 0b0000	| automatic conversion mode is disabled.
			 * @p 1		| @p 0b0001	| cycle time x 32 (27 ksps)
			 * @p 2		| @p 0b0010	| cycle time x 64 (13.5 ksps)
			 * @p 3		| @p 0b0011	| cycle time x 128 (6.7 ksps)
			 * @p 4		| @p 0b0100	| cycle time x 256 (3.4 ksps)
			 * @p 5		| @p 0b0101	| cycle time x 512 (1.7 ksps)
			 * @p 6		| @p 0b0110	| cycle time x 1024 (0.9 ksps)
			 * @p 7		| @p 0b0111	| cycle time x 2048 (0.4 ksps)
			 *
			 * @see ADC121C021 datasheet, page 19 ("configuration register")
			 * @see ADC121C021 datasheet, page 27 ("automatic conversion mode")
			 */
			I2CADC &enable_automatic_mode( const uint8_t value = 1 );

			/** Read the most recent 12-bit analog-to-digital conversion result register (0x00).  If automatic mode
			 * is disabled, this will force the I2CADC to immediatelly perform a conversion.
			 *
			 * When you have an analog Grove twig connected to an I2CADC, this is the method to call to read the value
			 * from the connected Grove.  The value returned will only have 12 bits, even though the C++ type is
			 * @p uint16_t.  This is because the I2CADC specifically stores only 12 bits.  This means the result will
			 * be between 0 and 4095 (0x0fff).
			 *
			 * @note For advanced users, if you have enabled alerts on the I2CADC and need to inspect the alert flag,
			 * call @ref read16(0x00) instead to get access to bit D15.  Calling @ref get_result() will mask out the
			 * upper 4 bits, including the alert flag.
			 */
			uint16_t get_result(void) { return read12( ERegister::kConversionResult ); }

			/** Return the lowest conversion result seen since the I2CADC was last reset.  This is the 12-bit value
			 * stored in register Vmin (0x06).
			 *
			 * @note This register is only updated when automatic mode has been enabled.
			 */
			uint16_t get_minimum_result(void) { return read12( ERegister::kLowestConversion ); }

			/** Return the highest conversion result seen since the I2CADC was last reset.  This is the 12-bit value
			 * stored in register Vmin (0x06).
			 *
			 * @note This register is only updated when automatic mode has been enabled.
			 */
			uint16_t get_maximum_result(void) { return read12( ERegister::kHighestConversion ); }

			/// Reset the minimum conversion value stored in register 0x06 when automatic conversion mode is enabled.
			I2CADC &reset_minimum_result(void) { write16( ERegister::kLowestConversion, 0x0fff ); return *this; }

			/// Reset the maximum conversion value stored in register 0x07 when automatic conversion mode is enabled.
			I2CADC &reset_maximum_result(void) { write16( ERegister::kHighestConversion, 0x0000 ); return *this; }

			/// Get the minimum conversion result and reset the register.
			uint16_t get_and_reset_minimum_result(void) { const uint16_t value = get_minimum_result(); reset_minimum_result(); return value; }

			/// Get the maximum conversion result and reset the register.
			uint16_t get_and_reset_maximum_result(void) { const uint16_t value = get_maximum_result(); reset_maximum_result(); return value; }

			/// Read a byte (8 bits) from the specificed register. @{
			uint8_t read8( const uint8_t register_address );
			uint8_t read8( const ERegister reg ) { return read8( static_cast<uint8_t>(reg) ); }
			/// @}

			/// Read a 16-bit word from the specified register. @{
			uint16_t read16( const uint8_t register_address );
			uint16_t read16( const ERegister reg ) { return read16( static_cast<uint8_t>(reg) ); }
			/// @}

			/** Read a 16-bit word from the specified register, but only keep the bottom 12 bits.  Since the I2CADC
			 * is 12-bit, the top 4 bits can serve a different purpose or be reserved for future use.
			 *
			 * For example, when reading the conversion result register (0x00), the D15 bit is used as the alert flag.
			 * By calling @ref read12(), those top 4 bits are masked out.
			 * @{
			 */
			uint16_t read12( const uint8_t register_address )	{ return 0x0fff & read16(register_address);		}
			uint16_t read12( const ERegister reg )				{ return read12( static_cast<uint8_t>(reg) );	}
			/// @}

			/// Write the given value to the specified register address. @{
			I2CADC &write8	( const uint8_t register_address, const  uint8_t value );
			I2CADC &write16	( const uint8_t register_address, const uint16_t value );
			I2CADC &write8	( const ERegister reg			, const  uint8_t value )	{ return write8	( static_cast<uint8_t>(reg), value ); }
			I2CADC &write16	( const ERegister reg			, const uint16_t value )	{ return write16( static_cast<uint8_t>(reg), value ); }
			/// @}

			uint8_t	i2c_address;		///< I2C address.
			int		i2c_file_handle;	///< File handle to @p /dev/i2c-1.
	};
}
