/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: seeedgrove.cpp 1732 2016-04-01 05:22:06Z stephane $
 */

#include "sg++.hpp"
#include <iostream>
#include <iomanip>
#include <thread>
#include <typeinfo>
#include <csignal>
#include <cstring>
#include <system_error>


typedef std::vector<std::string> VStr;


volatile sig_atomic_t exit_signal_detected = 0;


void list( const SG::VGroveTypes &v )
{
	const SG::MGroveTypeNames m = SG::get_grove_type_name_map();

	for ( SG::EGroveType grove_type : v )
	{
		// printing the type and sku is simple
		std::cout << "-> 0x"	<< std::hex << std::setw(4) << std::setfill('0') << static_cast<int>(grove_type);
		std::cout << ": SKU #"	<< std::dec << std::setw(9) << std::setfill('0') << SG::get_sku_from_grove_type( grove_type );

		// if something is seriously wrong, then trying to get the name from the map may throw an exception (but this should never happen)
		const std::string grove_name = m.at(grove_type);
		std::cout << ": " << grove_name;

		// see if we know something about the I2C address this type uses
		size_t count = 0;
		for ( const SG::I2CAddress address : SG::get_grove_type_addresses( grove_type ) )
		{
			std::cout << (count == 0 ? " [I2C address:" : ",") << " 0x" << std::hex << std::setw(2) << std::setfill('0') << (int)address;
			count ++;
		}
		std::cout << (count == 0 ? "" : "]") << std::endl;
	}

	std::cout << std::endl << "Number of Groves displayed: " << std::dec << v.size() << std::endl;

	return;
}


void list_all( void )
{
	SG::VGroveTypes v;

	// convert the full map to a vector of types
	for ( auto iter : SG::get_grove_type_name_map() )
	{
		const SG::EGroveType grove_type = iter.first;

		v.push_back( grove_type );
	}

	list( v );

	return;
}


void list_detect( void )
{
	list( SG::detect() );

	return;
}


void list_addresses( void )
{
	// get a unique set of all the types for which we have an address
	SG::SGroveTypes s;
	for ( const auto iter : SG::get_grove_type_addresses() )
	{
		s.insert( iter.first );
	}

	// convert the set to a vector
	SG::VGroveTypes v( s.begin(), s.end() );

	// display all the types that have one or more address
	list( v );

	return;
}


void sg_fatal_signal_handler( int sig_num, siginfo_t *sig_info, void *context )
{
	exit_signal_detected = sig_num;

	SG::SGpp::get().cleanup();

	// terminate -- this will call the default C signal handler which will terminate the application
	raise( sig_num );

	return;
}


void sg_normal_signal_handler( int sig_num, siginfo_t *sig_info, void *context )
{
	exit_signal_detected = sig_num;

	return;
}


void setup_signal_handling( void )
{
	struct sigaction action;
	memset( &action, '\0', sizeof(action) );
	action.sa_sigaction	= sg_normal_signal_handler;
	action.sa_flags		= SA_SIGINFO | SA_RESTART;

	// these signals cause the application to shutdown normally
	sigaction( SIGTERM	, &action, nullptr );	// terminate
	sigaction( SIGQUIT	, &action, nullptr );	// CTRL+BREAK (CTRL+D?)
	sigaction( SIGINT	, &action, nullptr );	// CTRL+C
	sigaction( SIGHUP	, &action, nullptr );	// CTRL+BACKSLASH
	sigaction( SIGHUP	, &action, nullptr );	// hangup
	sigaction( SIGUSR1	, &action, nullptr );	// user
	sigaction( SIGUSR2	, &action, nullptr );	// user

	// these signals cause an immediate semi-orderly shutdown
	action.sa_sigaction = sg_fatal_signal_handler;
	sigaction( SIGSEGV	, &action, nullptr );	// segfault
	sigaction( SIGILL	, &action, nullptr );	// illegal instruction
	sigaction( SIGABRT	, &action, nullptr );	// abort

	return;
}


void chainable_rgb_led( const VStr &args )
{
	SG::ChainableRGBLED rgb_led;

	std::cout << rgb_led.get_description() << std::endl;

	if (args.size() == 1)
	{
		rgb_led.set_RGB( args[0] );
	}
	else if (args.size() == 3)
	{
		const int r = atoi(args[0].c_str());
		const int g = atoi(args[1].c_str());
		const int b = atoi(args[2].c_str());

		std::cout	<< "Red:   " << r << std::endl
					<< "Green: " << g << std::endl
					<< "Blue:  " << b << std::endl;

		rgb_led.set_RGB( r, g, b );
	}
	else
	{
		std::cout << "Random colours..." << std::endl;

		while ( exit_signal_detected == 0 )
		{
			rgb_led.set_RGB( rand() % 255, rand() % 255, rand() % 255 );
			std::this_thread::sleep_for( std::chrono::milliseconds(100) );
		}
	}

	return;
}


void buzzer( const VStr &args )
{
	SG::Buzzer buzzer;

	std::cout << buzzer.get_description() << std::endl;

	while ( exit_signal_detected == 0 )
	{
		buzzer.turn_on( std::chrono::milliseconds(15) );
		std::this_thread::sleep_for( std::chrono::milliseconds(1000) );
	}
	buzzer.turn_off();

	return;
}


void temperature( const VStr &args )
{
	SG::I2CADC adc( 0x50 );
	adc.enable_automatic_mode();
	SG::TemperatureSensor temp(adc);

	std::cout << temp.get_description() << std::endl;

	// i2cget -y 1 0x50 0x00 w
	// 0xf903

	std::cout << "celsius    = " << temp.get_celsius() << std::endl;
	
//	std::cout << "fahrenheit =" << temp.get_fahrenheit()	<< std::endl;

	return;
}


void bb_led( const VStr &args )
{
	SG::BeagleBone::LEDControl led_control;
	led_control.start_blink_thread();
	led_control.load_random_pattern();

	size_t counter = 0;
	while ( exit_signal_detected == 0 )
	{
		std::this_thread::sleep_for( std::chrono::milliseconds(100) );

		counter ++;
		if (counter > 100)
		{
			counter = 0;
			led_control.load_random_pattern();
		}
	}

	return;
}


void show_help( void )
{
	std::cout	<< "Command line usage:"						<< std::endl
				<< "  list_all"									<< std::endl
				<< "  list_detect"								<< std::endl
				<< "  list_address"								<< std::endl
				<< "  buzzer"									<< std::endl
				<< "  chainable_rgb_led [r g b | html_colour]"	<< std::endl
				<< "  temperature"								<< std::endl
				<< "  bb_led"									<< std::endl
				;

	return;
}


int seeedgrove_example_application( int argc, char **argv )
{
	int result = 0;

	std::cout << "-> initializing..." << std::endl;
	SG::SGpp::get().initialize();

	std::string cmd;
	VStr args;

	for (int idx = 1; idx < argc; idx ++)
	{
		if (idx == 1)
		{
			cmd = argv[idx];
			std::cout << "-> processing \"" << cmd << "\"..." << std::endl;
		}
		else
		{
			args.push_back( argv[idx] );
		}
	}

	if		(cmd == "list"				)	list_all();
	else if (cmd == "list_all"			)	list_all();
	else if (cmd == "list_address"		)	list_addresses();
	else if (cmd == "list_addresses"	)	list_addresses();
	else if (cmd == "addresses"			)	list_addresses();
	else if (cmd == "address"			)	list_addresses();
	else if (cmd == "list_detect"		)	list_detect();
	else if (cmd == "detect"			)	list_detect();
	else if (cmd == "chainable_rgb_led"	)	chainable_rgb_led	(args);
	else if (cmd == "buzzer"			)	buzzer				(args);
	else if (cmd == "temperature"		)	temperature			(args);
	else if (cmd == "bb_led"			)	bb_led				(args);
	else if (cmd == "help"				)	show_help();
	else if (cmd == "--help"			)	show_help();
	else if (cmd == ""					)	show_help();
	else
	{
		std::cout << "Unrecognized command \"" << cmd << "\"." << std::endl;
		result = 1;
	}

	return result;
}


int main( int argc, char **argv )
{
	int result = 1;

	try
	{
		setup_signal_handling();

		srand( std::time(nullptr) );

		std::cout	<< "Seeed Grove ++ command line tool"						<< std::endl
					<< "-> version "			<< SG::get_version()			<< std::endl
					<< "-> build timestamp "	<< SG::get_build_timestamp()	<< std::endl;

		result = seeedgrove_example_application( argc, argv );
	}
	catch ( const std::system_error &e )
	{
		std::cout << "Caught " << SG::demangle(typeid(e).name()) << ": ";

		const int value = e.code().value();
		if (value)
		{
			std::cout << "errno " << value << ": ";
		}
		std::cout << e.what() << std::endl;
		result = 2;
	}
	catch ( const std::exception &e )
	{
		std::cout << "Caught " << SG::demangle(typeid(e).name()) << ": " << e.what() << std::endl;
		result = 3;
	}

	if (exit_signal_detected)
	{
		std::cout	<< std::endl
					<< "Signal detected:  " << strsignal(exit_signal_detected) << std::endl
					<< "Stopping " << argv[0] << std::endl;
	}

	return result;
}
