/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_104030006_ChainableRGBLED.cpp 1729 2016-03-30 07:04:48Z stephane $
 */

#include "sg_104030006_ChainableRGBLED.hpp"
#include "sg++.hpp"
#include <cmath>


SG::ChainableRGBLED::~ChainableRGBLED( void )
{
	if (turn_off_LED_when_destructing)
	{
		try
		{
			turn_off();
		}
		catch (...)
		{
			// ignore exceptions; don't throw from the destructor
		}
	}

	SG::SGpp::get().gpio.gpio_unexport( 2 );	// clock
	SG::SGpp::get().gpio.gpio_unexport( 3 );	// data

	return;
}


SG::ChainableRGBLED::ChainableRGBLED( const size_t i, const std::string &n, const bool turn_off_at_destruction ) :
		GroveGPIO( SG::EGroveType::kChainableRGBLED, n ),
		idx		( i ),
		red		( 0 ),
		green	( 0 ),
		blue	( 0 ),
		turn_off_LED_when_destructing( turn_off_at_destruction )
{
	SG::SGpp::get().gpio.gpio_export( 2, GpioManagement::EDirection::kOutput );	// clock
	SG::SGpp::get().gpio.gpio_export( 3, GpioManagement::EDirection::kOutput );	// data

	set_RGB( 0, 0, 0 );

	return;
}


SG::ChainableRGBLED::ChainableRGBLED( const SG::ChainableRGBLED &rhs ) :
		GroveGPIO( rhs.get_type(), rhs.get_name() ),
		idx		( rhs.idx	),
		red		( rhs.red	),
		green	( rhs.green	),
		blue	( rhs.blue	),
		turn_off_LED_when_destructing( rhs.turn_off_LED_when_destructing )
{
	return;
}


SG::ChainableRGBLED &SG::ChainableRGBLED::operator=( const SG::ChainableRGBLED &rhs )
{
	if (this != &rhs)
	{
		GroveGPIO::operator=( rhs );
		idx		= rhs.idx;
		red		= rhs.red;
		green	= rhs.green;
		blue	= rhs.blue;
		turn_off_LED_when_destructing = rhs.turn_off_LED_when_destructing;
	}

	return *this;
}


bool SG::ChainableRGBLED::operator==( const SG::ChainableRGBLED &rhs ) const
{
	return	GroveGPIO::operator==(rhs)						&&
			idx								== rhs.idx		&&
			red								== rhs.red		&&
			green							== rhs.green	&&
			blue							== rhs.blue		&&
			turn_off_LED_when_destructing	== rhs.turn_off_LED_when_destructing;
}


SG::ChainableRGBLED &SG::ChainableRGBLED::set_RGB( const int r, const int g, const int b )
{
	red		= std::max( 0, std::min( 255, r ) );
	green	= std::max( 0, std::min( 255, g ) );
	blue	= std::max( 0, std::min( 255, b ) );

	return turn_on();
}


SG::ChainableRGBLED &SG::ChainableRGBLED::set_RGB( const size_t r, const size_t g, const size_t b )
{
	red		= (r <= 255 ? r : 255);
	green	= (g <= 255 ? g : 255);
	blue	= (b <= 255 ? b : 255);

	return turn_on();
}


SG::ChainableRGBLED &SG::ChainableRGBLED::set_RGB( const double r, const double g, const double b )
{
	red		= std::max( 0.0, std::min( 255.0, r * 255.0 ) );
	green	= std::max( 0.0, std::min( 255.0, g * 255.0 ) );
	blue	= std::max( 0.0, std::min( 255.0, b * 255.0 ) );

	return turn_on();
}


SG::ChainableRGBLED &SG::ChainableRGBLED::set_RGB( std::string html_colour )
{
	size_t len = html_colour.length();
	if (len > 0)
	{
		if (html_colour[0] == '#')
		{
			html_colour.erase(0, 1);
			len --;
		}
	}

	/* We should now have one of the following formats:
	 * 
	 *		"rgb"			<- len = 3
	 *		"rrggbb"		<- len = 6
	 */

	std::string r = "ff";
	std::string g = "ff";
	std::string b = "ff";

	if (len == 3)
	{
		r = html_colour[0];
		g = html_colour[1];
		b = html_colour[2];

		// double the colours, so "7af" looks like "77aaff"
		r += r;
		g += g;
		b += b;
	}
	else if (len == 6)
	{
		r = html_colour.substr(0, 2);
		g = html_colour.substr(2, 2);
		b = html_colour.substr(4, 2);
	}

	char *p = nullptr;
	red		= strtol( r.c_str(), &p, 16 );	if (p[0] != '\0') red	= 255;
	green	= strtol( g.c_str(), &p, 16 );	if (p[0] != '\0') green	= 255;
	blue	= strtol( b.c_str(), &p, 16 );	if (p[0] != '\0') blue	= 255;

	return turn_on();
}


SG::ChainableRGBLED &SG::ChainableRGBLED::turn_off( void )
{
	return turn_on( 0, 0, 0 );
}


SG::ChainableRGBLED &SG::ChainableRGBLED::turn_on( void )
{
	// ensure the colours are within the valid range
	red		= std::max( 0, std::min( 255, red	) );
	green	= std::max( 0, std::min( 255, green	) );
	blue	= std::max( 0, std::min( 255, blue	) );

	return turn_on( red, green, blue );
}


SG::ChainableRGBLED &SG::ChainableRGBLED::turn_on( int r, int g, int b )
{
	// inspiration for this function came from http://www.seeedstudio.com/wiki/Grove_-_Chainable_RGB_LED#With_Beaglebone_Green

	/// @todo How to set multiple RGB LEDs if they are chained?

	// ensure the colours are within the valid range
	const uint8_t byte_red		= std::max( 0, std::min( 255, r) );
	const uint8_t byte_green	= std::max( 0, std::min( 255, g) );
	const uint8_t byte_blue		= std::max( 0, std::min( 255, b) );

	/* I have no idea what this "prefix" is doing.  It is set using the inverse of the 2 most significant bits of the
	 * 3 colour components.  I can only guess it is used to preset the RGB LED to an approximation of the final colour
	 * until every colour has been received, or as a crude checksum of the 2 most significant bits of each colour.
	 */
	uint8_t prefix = 0xC0;						// b11000000
	if ((blue	& 0x80) == 0)	prefix |= 0x20;	// b00100000
	if ((blue	& 0x40) == 0)	prefix |= 0x10;	// b00010000
	if ((green	& 0x80) == 0)	prefix |= 0x08;	// b00001000
	if ((green	& 0x40) == 0)	prefix |= 0x04;	// b00000100
	if ((red	& 0x80) == 0)	prefix |= 0x02;	// b00000010
	if ((red	& 0x40) == 0)	prefix |= 0x01;	// b00000001

	// data frame starts with 32 zeros
	send_byte( 0 );
	send_byte( 0 );
	send_byte( 0 );
	send_byte( 0 );

	send_byte( prefix );

	// this is intentional, the order in which the colours are sent is blue, green, then red
	send_byte( byte_blue	);
	send_byte( byte_green	);
	send_byte( byte_red		);

	// data frame is terminated with an additional 32 zeros
	send_byte( 0 );
	send_byte( 0 );
	send_byte( 0 );
	send_byte( 0 );

	return *this;
}
