/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_104030011_OLED096.hpp 1776 2016-04-17 04:52:34Z stephane $
 */

#pragma once

#include "sg_common_OLED.hpp"


namespace SG
{
	/** I2C 0.96" x 0.96" square OLED display.  This I2C Grove twig has 128 pixels across and 128 pixels in height,
	 * each of which is exactly 1 bit in depth.  With a typical 8x8 font, this means you can display 16 lines of text,
	 * each one with 16 characters.  This OLED is exactly twice the height of @ref OLED112.
	 *
	 * The following measurements are @b <em>approximate</em>:
	 * Description		| Dimension (Imperial)	| Dimension (Metric)
	 * -----------------|-----------------------|-------------------------------
	 * total width		| 1.5\"					| 3.8 cm
	 * total height		| 1.5\"					| 3.8 cm
	 * display width	| 1\"					| 2.5 cm
	 * display height	| 1\"					| 2.5 cm
	 *
	 * @warning The Seeed web site and wiki has conflicting and/or confusing information on this device, including
	 * some wiki links to the wrong device image.  The SSD1327 driver IC has a 128 x 128 bit memory buffer, for a total
	 * of 2 KiB of memory, and this Grove twig has a display with exactly 16384 pixels (128 x 128).  It is square in
	 * shape, not rectangular.
	 *
	 * Connect the OLED display to the left-hand-side I2C interface.  The I2C address for this I2C device is @p 0x3c.
	 *
	 * ~~~~
	 * i2cdetect -y -r 2
	 *
	 *      0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f
	 * 00:          -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 10: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 20: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 30: -- -- -- -- -- -- -- -- -- -- -- -- 3c -- -- --
	 * 40: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 50: -- -- -- -- UU UU UU UU -- -- -- -- -- -- -- --
	 * 60: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 70: -- -- -- -- -- -- -- --
	 * ~~~~
	 *
	 * @see @ref SG::OLED112
	 * @see http://www.seeedstudio.com/wiki/Grove_-_OLED_Display_96%2A96
	 * @see http://www.seeedstudio.com/depot/Grove-OLED-Display-096-p-824.html
	 * @see http://garden.seeedstudio.com/images/8/82/SSD1327_datasheet.pdf
	 */
	class OLED096 : public CommonOLED
	{
		public:

			/// Destructor.
			virtual ~OLED096( void );

			/// Constructor.  If not specified, the I2C address will be obtained using @ref get_address_from_type().
			OLED096( const std::string &name = "", const SG::GroveI2CDigital::I2CAddress addr = 0 );

			virtual size_t pixel_width	(void) const { return 128;	}	///< Get the width in pixels.  This always returns @p 128 for the 0.96" x 0.96" OLED.
			virtual size_t pixel_height	(void) const { return 128;	}	///< Get the height in pixels.  This always returns @p 128 for the 0.96" x 0.96" OLED.
			virtual size_t pixel_depth	(void) const { return 1;	}	///< Get the number of bits per pixel.  This always returns @p 1 for the 0.96" x 0.96" OLED.
			virtual size_t memory_width	(void) const { return 128;	}	///< Get the number of pixels that can be stored in memory.  This always returns @p 128 for the 0.96" x 0.96" OLED.
			virtual size_t memory_height(void) const { return 128;	}	///< Get the number of pixels that can be stored in memory.  This always returns @p 128 for the 0.96" x 0.96" OLED.
			virtual size_t memory_depth	(void) const { return 4;	}	///< Get the number of bits each pixel occupies in memory.  This always returns @p 1 for the 0.96" x 0.96" OLED.

			/// Toggle the various display modes. @{
			virtual OLED096 &normal_display	(void) { write_byte( 0xa4 ); return *this; }	///< Normal display mode.
			virtual OLED096 &all_on_display	(void) { write_byte( 0xa5 ); return *this; }	///< All pixels set @p ON.
			virtual OLED096 &all_off_display(void) { write_byte( 0xa6 ); return *this; }	///< All pixels set @p OFF.
			virtual OLED096 &invert_display	(void) { write_byte( 0xa7 ); return *this; }	///< Invert display mode.
			/// @}

			virtual OLED096 &reset			(void);	///< Reset the OLED.
			virtual OLED096 &clear_screen	(void);	///< Set all of the GDDRAM to 0x00 to clear the screen.
	};
}
