/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_GpioManagement.hpp 1680 2016-03-09 11:20:02Z stephane $
 */

#pragma once

#include <map>
#include <set>


namespace SG
{
	typedef size_t GPIO;
	typedef std::set<GPIO> SGPIO;

	/** Singleton object used to manage GPIO.  Obtain a copy of the singleton through @ref SG::SGpp::gpio.
	 * @see GPIO in Linux is explained here:  https://www.kernel.org/doc/Documentation/gpio/sysfs.txt
	 */
	class GpioManagement final
	{
		public:

			enum class EDirection
			{
				kUnknown	= 0,
				kInput		,
				kOutput
			};
			typedef std::map<GPIO, EDirection> MGPIODirection;

			enum class EValue
			{
				kInvalid	= -1,
				kLow		= 0,
				kHigh		= 1
			};

			/// Get the singleton.
			static GpioManagement &get( void );

			/** Determine whether all exported GPIO are automatically unexported on destruction.  This only includes
			 * GPIOs that were exported by the SG++ library.  Default is @p true to ensure all GPIOs are released.
			 * The equivalent functionality can be obtained at any time by manually calling @ref cleanup().
			 */
			bool unexport_all_gpio_on_destruction;

			/// Destructor.  @see @ref unexport_all_gpio_on_destruction
			~GpioManagement( void );

			/** Perform any necessary cleanup, such as releasing (unexporting) any GPIOs that were originally exported
			 * by this instance of the SG++ library.  @see @ref unexport_all_gpio_on_destruction
			 */
			GpioManagement &cleanup( void );

			/** Get the set of exported GPIO.  This only includes the GPIOs exported by this instance of the SG++
			 * library.  @see @ref get_direction_map()
			 */
			SGPIO get_exported( void );

			/** Get the map of exported GPIO and the direction for each one.  This only includes the GPIOs exported by
			 * this instance of the SG++ library.  @see @ref get_exported()
			 */
			MGPIODirection get_direction_map( void );

			/// Tell the Linux kernel to export a GPIO.  @see /sys/class/gpio/export
			GpioManagement &gpio_export( const GPIO gpio, const EDirection direction = EDirection::kOutput );

			/// Tell the Linux kernel a GPIO is no longer needed.  @see /sys/class/gpio/unexport
			GpioManagement &gpio_unexport( const GPIO gpio );

			/** Tell the Linux kernel if a GPIO will be used to reading or writing.  A GPIO must first be exported for
			 * this to work.  This normally doesn't need to be called unless the direction is changing, since
			 * @ref gpio_export() allows the direction to be set when the GPIO is originally exported.
			 * @see @ref gpio_export()
			 * @see /sys/class/gpio/gpioN/direction where "N" is the GPIO number.
			 */
			GpioManagement &set_direction( const GPIO gpio, const EDirection direction = EDirection::kOutput );

			/** See if the given GPIO has been exported.  If the GPIO was just exported, it may be necessary to retry
			 * several times, where it pauses briefly between each attempt.  By default, it only checks once.
			 * @return @p true if the GPIO is already exported.
			 * @return @p false if the GPIO has not been exported.
			 * @note This method checks @em all GPIOs, not only the ones originally exported by this library.
			 * @see @p /sys/class/gpio/gpioN where "N" is the GPIO number.
			 */
			bool is_exported( const GPIO gpio, int number_of_times_to_check = 1 );

			/** Set the given GPIO low or high.  The GPIO must be exported,
			 * and the direction must be set to @ref EDirection::kOutput.
			 * @see @ref gpio_export()
			 * @see @ref set_direction()
			 * @see /sys/class/gpio/gpioN/value where "N" is the GPIO number.
			 */
			GpioManagement &set_value( const GPIO gpio, const EValue &value );

			/// Alias for @ref set_value() @{
			GpioManagement &set_low	( const GPIO gpio )	{ return set_value(gpio, EValue::kLow	); }
			GpioManagement &set_high( const GPIO gpio ) { return set_value(gpio, EValue::kHigh	); }
			/// @}

		private:

			/// Private constructor.  @see @ref get()
			GpioManagement( void );

			/// Internal map of exported GPIOs and their direction.  @see @ref get_direction_map()
			MGPIODirection gpio_direction_map;

			/// Normally GpioManagement in created and managed by SGpp.  @see @ref SG::SGpp::gpio;
			friend class SGpp;
	};
}
