/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_BeagleBone_Detect.cpp 1847 2016-05-19 05:28:47Z stephane $
 */

#include "sg_BeagleBone_Detect.hpp"
#include "sg_base.hpp"
#include <algorithm>


std::string SG::BeagleBone::Detect::to_string( const SG::BeagleBone::Detect::ECompatibility type )
{
	switch( type )
	{
		case ECompatibility::invalid:				break;
		case ECompatibility::other:					break;
		case ECompatibility::ti_am33xx:				return "ti,am33xx";
		case ECompatibility::ti_am335x_bone:		return "ti,am335x-bone";
		case ECompatibility::ti_am335x_bone_black:	return "ti,am335x-bone-black";
		case ECompatibility::ti_am335x_bone_green:	return "ti,am335x-bone-green";
	}

	return "?";
}


SG::VStr SG::BeagleBone::Detect::to_string( const SG::BeagleBone::Detect::Compatibilities &compabilities )
{
	VStr v;

	for (const auto type : compabilities)
	{
		v.push_back( to_string(type) );
	}

	return v;
}


SG::BeagleBone::Detect::Compatibilities SG::BeagleBone::Detect::get_compatibilities( void )
{
	/* Only a small handful of possible options interest us for BBB and BBG, so just test for those ones and ignore
	 * everything else that might appear in the text string.  Note that these are from most precise to least precise
	 * so the vector stays in the same order as they are listed in the device-tree compatibility file.
	 */
	const Compatibilities all
	{
		ECompatibility::ti_am335x_bone_green,
		ECompatibility::ti_am335x_bone_black,
		ECompatibility::ti_am335x_bone		,
		ECompatibility::ti_am33xx
	};

	/* The compatibilities are stored with a NULL byte between each string.  For example:
	 *
	 *	> hexdump -C /proc/device-tree/compatible
	 *	00000000  74 69 2c 61 6d 33 33 35  78 2d 62 6f 6e 65 2d 67  |ti,am335x-bone-g|
	 *	00000010  72 65 65 6e 00 74 69 2c  61 6d 33 33 35 78 2d 62  |reen.ti,am335x-b|
	 *	00000020  6f 6e 65 2d 62 6c 61 63  6b 00 74 69 2c 61 6d 33  |one-black.ti,am3|
	 *	00000030  33 35 78 2d 62 6f 6e 65  00 74 69 2c 61 6d 33 33  |35x-bone.ti,am33|
	 *	00000040  78 78 00                                          |xx.             |
	 *
	 * (Note the four '00' bytes delimiting the four strings in the text above.)
	 *
	 * This means when it comes time to search for an exact string, we need to append a NULL character at the
	 * end.  Otherwise, when we search for string such as "ti,am335x-bone", we'll get a false positive with
	 * "ti,am335x-bone-black" and "ti,am335x-bone-green".
	 */
	const std::string file_content = read_file( "/proc/device-tree/compatible" );

	Compatibilities v;

	for (const auto type : all)
	{
		const std::string name = to_string(type) + "\0";
		if (file_content.find(name) != std::string::npos)
		{
			// we found a match -- store it in the vector we'll return to the user
			v.push_back( type );
		}
	}

	return v;
}


std::string SG::BeagleBone::Detect::get_model( void )
{
	const VStr v =
	{
		"/proc/device-tree/model",
		"/sys/devices/virtual/dmi/id/product_name"
	};

	std::string str;

	for (const auto &filename : v)
	{
		if (file_exists(filename))
		{
			str = trim( read_file( filename ) );
		}

		if (str.empty())
		{
			// look at the next file
			continue;
		}

		// if we get here, then we found some kind of text to use

		break;
	}

	return str;
}


bool SG::BeagleBone::Detect::is_beaglebone_compatible( void )
{
	const Compatibilities compatabilities = get_compatibilities();

	if ( std::count( compatabilities.begin(), compatabilities.end(), ECompatibility::ti_am335x_bone ) == 0 )
	{
		// not found
		return false;
	}

	return true;
}


bool SG::BeagleBone::Detect::is_beaglebone_black_compatible( void )
{
	const Compatibilities compatabilities = get_compatibilities();

	if ( std::count( compatabilities.begin(), compatabilities.end(), ECompatibility::ti_am335x_bone_black ) == 0 )
	{
		// not found
		return false;
	}

	return true;
}


bool SG::BeagleBone::Detect::is_beaglebone_green_compatible( void )
{
	const Compatibilities compatabilities = get_compatibilities();

	if ( std::count( compatabilities.begin(), compatabilities.end(), ECompatibility::ti_am335x_bone_green ) == 0 )
	{
		// not found
		return false;
	}

	return true;
}
