/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_BeagleBone_USB.cpp 1855 2016-05-24 00:22:22Z stephane $
 */

#include "sg_BeagleBone_USB.hpp"
#include "sg_FD.hpp"
#include <system_error>
#include <thread>
#include <unistd.h>
#include <sys/mman.h>


static void sg_bb_usb_control( const bool turn_on, const size_t sleep_time_in_milliseconds )
{
	/* This function performs the following tasks:
	 *
	 *		1) turn the USB power either ON or OFF.
	 *		2) Optionally, invert the USB power state from step #1 after a short pause.
	 */

	const std::string filename = "/dev/mem";
	SG::SharedFileDescriptor sg_fd( std::make_shared<SG::FD>(filename) );

	// mmap requires addresses to be on a page boundary

	const size_t page_size_in_bytes = getpagesize();		// should be 4096
	const size_t address_gpio3_13	= 0x47401c60;			// USB is GPIO3_13
	const size_t address_start		= address_gpio3_13 / page_size_in_bytes * page_size_in_bytes;
	const size_t address_offset		= address_gpio3_13 - address_start;

	void *addr = mmap( 0, page_size_in_bytes, PROT_READ | PROT_WRITE, MAP_SHARED, sg_fd->fd, address_start );
	if (addr == MAP_FAILED)
	{
		/// @throw std::system_error if the /dev/mem mapping fails.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, "SG++ failed to open " + filename );
	}

	uint8_t *byte_ptr = reinterpret_cast<uint8_t*>(addr);

	// turn the USB power ON or OFF
	byte_ptr[address_offset] = (turn_on ? '\1' : '\0' );

	if (sleep_time_in_milliseconds)
	{
		// pause for a short time, then reverse whatever we just did to the USB power
		std::this_thread::sleep_for( std::chrono::milliseconds(sleep_time_in_milliseconds) );
		byte_ptr[address_offset] = (turn_on ? '\0' : '\1' );
	}

	// don't bother trying to read that memory location, it doesn't give consistent/reliable results

	munmap( addr, page_size_in_bytes );

	return;
}


void SG::BeagleBone::USB::toggle( const bool turn_on, const size_t sleep_time_in_milliseconds )
{
	sg_bb_usb_control( turn_on, sleep_time_in_milliseconds );

	return;
}


void SG::BeagleBone::USB::toggle( const size_t sleep_time_in_milliseconds )
{
	sg_bb_usb_control( false, sleep_time_in_milliseconds );

	return;
}


void SG::BeagleBone::USB::set( const bool turn_on )
{
	toggle( turn_on, 0 );

	return;
}


void SG::BeagleBone::USB::turn_on( void )
{
	set( true );

	return;
}


void SG::BeagleBone::USB::turn_off( void )
{
	set( false );

	return;
}
