/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_104020006_LEDBar.cpp 1853 2016-05-22 04:00:24Z stephane $
 */

#include "sg_104020006_LEDBar.hpp"
#include "sg++.hpp"


SG::LEDBar::~LEDBar( void )
{
	SG::SGpp::get().gpio.gpio_unexport( 2 );	// clock
	SG::SGpp::get().gpio.gpio_unexport( 3 );	// data

	return;
}


SG::LEDBar::LEDBar( const std::string &n ) :
		SG::GroveGPIO( SG::EGroveType::kLEDBarV2, n )
{
	SG::SGpp::get().gpio.gpio_export( 2, GpioManagement::EDirection::kOutput );	// clock
	SG::SGpp::get().gpio.gpio_export( 3, GpioManagement::EDirection::kOutput );	// data

	return;
}


SG::LEDBar &SG::LEDBar::all_on(void)
{
	/* Data is 16 bits for the command followed by 12x16 bits for the PWM data.
	 * This means 2 bytes for the command, and 24 bytes for the data.
	 *
	 * The command is described on page 13 of the MY9221 datasheet:
	 *
	 * Bits					| Description
	 * ---------------------|------------
	 * 15, 14, 13, 12, 11	| Unused.  Set to zero.
	 * 10					| 0=Iout slow, 1=Iout fast
	 * 9, 8					| 00=8-bit, 01=12-bit, 10=14-bit, 11=16-bit grayscale
	 * 7, 6, 5, 4			| 000=8.6MHz, 001=freq/2, 010=freq/4, 011=freq/8, 100=freq/16, 101=freq/64, 110=freq/128, 111=freq=256
	 * 3					| 0=internal oscillator, 1=external oscillator
	 * 2					| 0=LED driver, 1=MY-PWM
	 * 1					| 0=free running, 1=counter reset
	 * 0					| 0=frame cycle repeat, 1=frame cycle one-shot
	 */
	Data data;
	for (int i=0; i<10; i++)
	{
		data.colour[i] = 0x00ff;
	}
	// set the unused colours to zero
	data.colour[10] = 0;
	data.colour[11] = 0;

	send_data( data );

	return *this;
}


SG::LEDBar &SG::LEDBar::all_off(void)
{
	Data data;
	for (int i=0; i<12; i++)
	{
		data.colour[i] = 0x0000;
	}
	
	send_data( data );

	return *this;
}


SG::LEDBar &SG::LEDBar::toggle( const size_t led_idx, const uint8_t intensity )
{
	return *this;
}


//#include <iostream>
//#include <iomanip>
SG::LEDBar &SG::LEDBar::send_data( const Data &data )
{
	const GPIO out = 3;
	const GPIO clk = 2;

	// see the 3.3V "Switching Characteristics" on page 7, as well as the diagrams on page 15 of the data sheet
	const auto tw_dck	= std::chrono::nanoseconds	(900);	// min=50 ns, max=7200 ns
	const auto tstart	= std::chrono::microseconds	(300);	// min=220 microseconds
	const auto tstop	= std::chrono::nanoseconds	(300);	// min=200 ns
	const auto tw_high	= std::chrono::nanoseconds	(100);	// min=70 ns
	const auto tw_low	= std::chrono::nanoseconds	(300);	// min=230 ns

	GpioManagement &gm = SG::SGpp::get().gpio;

	// data is sampled whenever the clock is toggled (see top of page 12) so we need to know the initial state of clk
	bool dcki = gm.get_bool(2);

	// there are 13 x uint16_t in a data structure
	for (size_t count = 0; count < 13; count ++)
	{
		// get the next 16 bits we need to send -- either the command, or one of the colour values
		uint16_t bits = (count == 0 ? data.command : data.colour[count - 1]);

if (count == 0)
{
	//       fedcba9876543210
	bits = 0b0000010000000000;
	//		 |    || |  |||||
	//		 |    || |  |||| \______ 0:		one-shot select, 0=frame cycle repeat
	//		 |    || |  ||| \_______ 1:		counter reset, 0=free running mode
	//		 |    || |  || \________ 2:		output polarity, 0=LED driver
	//		 |    || |  | \_________ 3:		clock source, 0=internal oscillator
	//		 |    || |   \__________ 4:		output waveform, 0=traditional waveform, 1=APDM waveform
	//		 |    ||  \_____________ 7-5:	internal oscillator frequency, 000=8.6 MHz
	//		 |    | \_______________ 8-9:	grayscale, 00=8-bit
	//		 |     \________________ 10:	Iout, 0=slow mode, 1=fast mode
	//		  \_____________________ 11-15:	reserved

	// Tr/Tf (bit #10): "The length of time that it takes to ramp the voltage of the pulse from V1 to V2."
}

		// send all 16 bits by shifting them progressively to the left and checking the leftmost one each time
		for (size_t idx = 0; idx < 16; idx ++)
		{
			gm.set_value( out, (bits & 0x8000) ? true : false );

			// invert the clock
			dcki = ! dcki;
			gm.set_value( clk, dcki );

			if (count == 12 && idx == 15)
			{
				// we're on the last bit of the last colour, so no need to pause anymore
				break;
			}

			std::this_thread::sleep_for( tw_dck );

			// move to the next bit
			bits <<= 1;
		}
	}

	/* After sending the command and the 12 colour values, we need to tell it to use this data.  See page 15 titled
	 * "Internal-latch control cycle timing diagram" of the MY9221 datasheet.  It consists of holding the data line
	 * high (or low) for some time, followed by 4 pulses, and then another pause.
	 *
	 * Thanks to "Jon Trulson" <jtrulson@ics.com> for replying to my e-mail on 2016-05-20 to explain how this works.
	 * Also see his code at https://github.com/intel-iot-devkit/upm/blob/master/src/my9221/my9221.cxx
	 */

	dcki = ! dcki;
	gm.set_value( clk, dcki );

	std::this_thread::sleep_for( tstart );

	// send 4 pulses
	for (size_t count = 0; count < 4; count ++)
	{
		gm.set_high( out );
		std::this_thread::sleep_for( tw_high );
		gm.set_low( out );

		if (count < 3)
		{
			std::this_thread::sleep_for( tw_low );
		}
	}

	// last is the tstop
	std::this_thread::sleep_for( tstop );
	dcki = ! dcki;
	gm.set_value( clk, dcki );
	std::this_thread::sleep_for( tw_dck );

	return *this;
}
