/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_104030008_OLED112.hpp 1817 2016-05-01 00:18:10Z stephane $
 */

#pragma once

#include "sg_common_OLED.hpp"


namespace SG
{
	/** I2C 1.12" rectangular OLED display.  This I2C Grove twig has 128 pixels across and 64 pixels in height, each of
	 * which is exactly 1 bit in depth.  With a typical 8x8 font, this means you can display 8 lines of text, each one
	 * with 16 characters, for a total of 128 characters of text.  This OLED is exactly half the height of @ref OLED096.
	 *
	 * The following measurements are @b <em>approximate</em>:
	 * Description		| Dimension (Imperial)	| Dimension (Metric)
	 * -----------------|-----------------------|-------------------------------
	 * total width		| 1.5\"					| 3.8 cm
	 * total height		| 0.75\"				| 1.9 cm
	 * display width	| 1\"					| 2.5 cm
	 * display height	| 0.5\"					| 1.3 cm
	 *
	 * @warning The Seeed web site and wiki has conflicting and/or confusing information on this device, including
	 * some wiki links to the wrong device image.  The SSD1308 driver IC has a 128 x 64 bit memory buffer, for a total
	 * of 1 KiB of memory, and this Grove twig has a display with exactly 8192 pixels (128 x 64).  It is rectangular
	 * in shape, not square.
	 *
	 * Description				| Image
	 * -------------------------|------
	 * OLED with sample text.	| @image html sg_104030008_OLED112.jpg
	 *
	 * Connect the OLED display to the left-hand-side I2C interface.  The I2C address for this I2C device is @p 0x3c.
	 *
	 * ~~~~
	 * i2cdetect -y -r 2
	 *
	 *      0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f
	 * 00:          -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 10: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 20: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 30: -- -- -- -- -- -- -- -- -- -- -- -- 3c -- -- --
	 * 40: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 50: -- -- -- -- UU UU UU UU -- -- -- -- -- -- -- --
	 * 60: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 70: -- -- -- -- -- -- -- --
	 * ~~~~
	 *
	 * @see @ref SG::OLED096
	 * @see http://www.seeedstudio.com/wiki/Grove_-_OLED_Display_128%2A64
	 * @see http://www.seeedstudio.com/depot/grove-oled-display-12864-p-781.html
	 * @see http://garden.seeedstudio.com/images/4/46/SSD1308_1.0.pdf
	 * @see http://garden.seeedstudio.com/images/c/c4/LY190-128064.pdf
	 */
	class OLED112 : public CommonOLED
	{
		public:

			/// Destructor.
			virtual ~OLED112( void );

			/// Constructor.  If not specified, the I2C address will be obtained using @ref get_address_from_type().
			OLED112( const std::string &name = "", const SG::GroveI2CDigital::I2CAddress addr = 0 );

			virtual size_t pixel_width	(void) const { return 128;	}	///< Get the width in pixels.  This always returns @p 128 for the 1.12" OLED.
			virtual size_t pixel_height	(void) const { return 64;	}	///< Get the height in pixels.  This always returns @p 64 for the 1.12" OLED.
			virtual size_t pixel_depth	(void) const { return 1;	}	///< Get the number of bits per pixel.  This always returns @p 1 (meaning each pixel is either @p on or @p off) for the 1.12" OLED.
			virtual size_t memory_width	(void) const { return 128;	}	///< Get the number of pixels that can be stored in memory.  This always returns @p 128 for the 1.12" OLED.
			virtual size_t memory_height(void) const { return 64;	}	///< Get the number of pixels that can be stored in memory.  This always returns @p 64 for the 1.12" OLED.
			virtual size_t memory_depth	(void) const { return 1;	}	///< Get the number of bits each pixel occupies in memory.  This always returns @p 1 for the 1.12" OLED.

			/// Toggle the various display modes. @{
			virtual OLED112 &normal_display			(void) { send_command(0xa4);	return *this; }	///< Normal display mode.
			virtual OLED112 &all_on_display			(void) { send_command(0xa5);	return *this; }	///< All pixels set @p ON.  @see @ref normal_display()
			virtual OLED112 &all_off_display		(void) {						return *this; }	///< All pixels set @p OFF. @warning This command is not supported by the 1.12" OLED.
			virtual OLED112 &do_not_invert_display	(void) { send_command(0xa6);	return *this; }	///< Do not invert display.  @see @ref invert_display()
			virtual OLED112 &invert_display			(void) { send_command(0xa7);	return *this; }	///< Invert display mode.  @see @ref do_not_invert_display()
			/// @}

			virtual OLED112 &reset					(void);	///< Reset the OLED.
			virtual OLED112 &clear_screen			(void);	///< Set all of the GDDRAM to 0x00 to clear the screen.
	};
}
