/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: seeedgrove.cpp 1867 2016-05-28 08:09:05Z stephane $
 */

#include "sg++.hpp"
#include <iostream>
#include <iomanip>
#include <thread>
#include <typeinfo>
#include <csignal>
#include <cstring>
#include <system_error>
#include <cmath>


typedef std::vector<std::string> VStr;


volatile sig_atomic_t exit_signal_detected = 0;


void list( const SG::VGroveTypes &v )
{
	const SG::MGroveTypeNames m = SG::get_grove_type_name_map();

	for ( SG::EGroveType grove_type : v )
	{
		// printing the type and sku is simple
		std::cout << "-> 0x"	<< std::hex << std::setw(4) << std::setfill('0') << static_cast<int>(grove_type);
		std::cout << ": SKU #"	<< std::dec << std::setw(9) << std::setfill('0') << SG::get_sku_from_grove_type( grove_type );

		// if something is seriously wrong, then trying to get the name from the map may throw an exception (but this should never happen)
		const std::string grove_name = m.at(grove_type);
		std::cout << ": " << grove_name;

		// see if we know something about the I2C address this type uses
		size_t count = 0;
// boooo BOOOO @todo
//		for ( const SG::GroveI2CDigital::I2CAddress address : SG::get_grove_type_addresses( grove_type ) )
//		{
//			std::cout << (count == 0 ? " [I2C address:" : ",") << " 0x" << std::hex << std::setw(2) << std::setfill('0') << (int)address;
//			count ++;
//		}
		std::cout << (count == 0 ? "" : "]") << std::endl;
	}

	std::cout << std::endl << "Number of Groves displayed: " << std::dec << v.size() << std::endl;

	return;
}


void list_all( void )
{
	SG::VGroveTypes v;

	// convert the full map to a vector of types
	for ( auto iter : SG::get_grove_type_name_map() )
	{
		const SG::EGroveType grove_type = iter.first;

		v.push_back( grove_type );
	}

	list( v );

	return;
}


void list_detect( void )
{
	list( SG::detect() );

	return;
}


void list_addresses( void )
{
#if 0 // TODO BOOOO booooo
	// get a unique set of all the types for which we have an address
	SG::SGroveTypes s;
	for ( const auto iter : SG::get_grove_type_addresses() )
	{
		s.insert( iter.first );
	}

	// convert the set to a vector
	SG::VGroveTypes v( s.begin(), s.end() );

	// display all the types that have one or more address
	list( v );
#endif
	return;
}


void sg_fatal_signal_handler( int sig_num, siginfo_t *sig_info, void *context )
{
	exit_signal_detected = sig_num;

	SG::SGpp::get().cleanup();

	// terminate -- this will call the default C signal handler which will terminate the application
	raise( sig_num );

	return;
}


void sg_normal_signal_handler( int sig_num, siginfo_t *sig_info, void *context )
{
	exit_signal_detected = sig_num;

	return;
}


void setup_signal_handling( void )
{
	struct sigaction action;
	memset( &action, '\0', sizeof(action) );
	action.sa_sigaction	= sg_normal_signal_handler;
	action.sa_flags		= SA_SIGINFO | SA_RESTART;

	// these signals cause the application to shutdown normally
	sigaction( SIGTERM	, &action, nullptr );	// terminate
	sigaction( SIGQUIT	, &action, nullptr );	// CTRL+BREAK (CTRL+D?)
	sigaction( SIGINT	, &action, nullptr );	// CTRL+C
	sigaction( SIGHUP	, &action, nullptr );	// CTRL+BACKSLASH
	sigaction( SIGHUP	, &action, nullptr );	// hangup
	sigaction( SIGUSR1	, &action, nullptr );	// user
	sigaction( SIGUSR2	, &action, nullptr );	// user

	// these signals cause an immediate semi-orderly shutdown
	action.sa_sigaction = sg_fatal_signal_handler;
	sigaction( SIGSEGV	, &action, nullptr );	// segfault
	sigaction( SIGILL	, &action, nullptr );	// illegal instruction
	sigaction( SIGABRT	, &action, nullptr );	// abort

	return;
}


void chainable_rgb_led( const VStr &args )
{
	SG::ChainableRGBLED rgb_led;

	std::cout << rgb_led.get_description() << std::endl;

	if (args.size() == 1)
	{
		rgb_led.set_RGB( args[0] );
	}
	else if (args.size() == 3)
	{
		const int r = atoi(args[0].c_str());
		const int g = atoi(args[1].c_str());
		const int b = atoi(args[2].c_str());

		std::cout	<< "Red:   " << r << std::endl
					<< "Green: " << g << std::endl
					<< "Blue:  " << b << std::endl;

		rgb_led.set_RGB( r, g, b );
	}
	else
	{
		std::cout << "Random colours..." << std::endl;

		while ( exit_signal_detected == 0 )
		{
			rgb_led.set_RGB( rand() % 255, rand() % 255, rand() % 255 );
			std::this_thread::sleep_for( std::chrono::milliseconds(100) );
		}
	}

	return;
}


void buzzer( const VStr &args )
{
	SG::Buzzer buzzer;

	std::cout << buzzer.get_description() << std::endl;

	while ( exit_signal_detected == 0 )
	{
		buzzer.turn_on( std::chrono::milliseconds(15) );
		std::this_thread::sleep_for( std::chrono::milliseconds(1000) );
	}
	buzzer.turn_off();

	return;
}


void temperature( const VStr &args )
{
	SG::I2CADC adc( "test", 0x50 );
	adc.enable_automatic_mode();
	SG::TemperatureSensor temp(adc);

	std::cout << temp.get_description() << std::endl;

	// i2cget -y 1 0x50 0x00 w
	// 0xf903

	std::cout << "celsius    = " << temp.get_celsius() << std::endl;
	
//	std::cout << "fahrenheit =" << temp.get_fahrenheit()	<< std::endl;

	return;
}


void compass( const VStr &args )
{
	SG::ThreeAxisDigitalCompass three_axis_digital_compass;
	std::cout << "3-axis digital compass: resetting configuration..." << std::endl;
	three_axis_digital_compass.reset();

	// MODE
	SG::ThreeAxisDigitalCompass::EMode mode = three_axis_digital_compass.get_mode();
	std::cout << "3-axis digital compass: mode = " << three_axis_digital_compass.to_string(mode) << std::endl;
	std::cout << "Setting the mode to kIdle..." << std::endl;
	three_axis_digital_compass.set_mode( SG::ThreeAxisDigitalCompass::EMode::kIdle );
	mode = three_axis_digital_compass.get_mode();
	std::cout << "3-axis digital compass: mode = " << three_axis_digital_compass.to_string(mode) << std::endl;

	// GAIN
	SG::ThreeAxisDigitalCompass::EGain gain;
	three_axis_digital_compass.get_config( gain );
	std::cout << "3-axis digital compass: gain = " << three_axis_digital_compass.to_string(gain) << std::endl;
	std::cout << "Setting the gain to 4.0Ga..." << std::endl;
	three_axis_digital_compass.set_config( SG::ThreeAxisDigitalCompass::EGain::k4_0Ga );
	three_axis_digital_compass.get_config( gain );
	std::cout << "3-axis digital compass: gain = " << three_axis_digital_compass.to_string(gain) << std::endl;

	// OUTPUT RATE + MEASUREMENTS
	SG::ThreeAxisDigitalCompass::EDataOutputRate	output_rate;
	SG::ThreeAxisDigitalCompass::EMeasurements		measurements;
	three_axis_digital_compass.get_config( output_rate, measurements );
	std::cout << "3-axis digital compass: output rate = " << three_axis_digital_compass.to_string(output_rate)	<< std::endl;
	std::cout << "3-axis digital compass: measurement = " << three_axis_digital_compass.to_string(measurements)	<< std::endl;
	std::cout << "Changing output rate and measurement..." << std::endl;
	three_axis_digital_compass.set_config( SG::ThreeAxisDigitalCompass::EDataOutputRate::k30Hz, SG::ThreeAxisDigitalCompass::EMeasurements::kPositiveBias );
	three_axis_digital_compass.get_config( output_rate, measurements );
	std::cout << "3-axis digital compass: output rate = " << three_axis_digital_compass.to_string(output_rate)	<< std::endl;
	std::cout << "3-axis digital compass: measurement = " << three_axis_digital_compass.to_string(measurements)	<< std::endl;

	bool regulator_enabled	= false;
	bool data_locked		= false;
	bool data_ready			= false;
	three_axis_digital_compass.get_status( regulator_enabled, data_locked, data_ready );
	std::cout << "Status: regulator=" << regulator_enabled << ", locked=" << data_locked << ", ready=" << data_ready << std::endl;

	three_axis_digital_compass.reset();

	int16_t x = 0;
	int16_t y = 0;
	int16_t z = 0;

	while (exit_signal_detected == 0)
	{
		three_axis_digital_compass.wait_for_significant_change( x, y, z );

		std::cout	<< "Data: "
					<< "x=" << x << ", "
					<< "y=" << y << ", "
					<< "z=" << z << ", "
					<< "direction="	<< three_axis_digital_compass.get_direction()	<< " radians, "
					<< "heading="	<< three_axis_digital_compass.get_heading()		<< " degrees"
					<< std::endl;
	}

	return;
}


void bb_detect( const VStr &args )
{
	std::cout << "BeagleBone detection:" << std::endl;

	std::cout << "Device model name: " << SG::BeagleBone::Detect::get_model() << std::endl;

	const SG::BeagleBone::Detect::Compatibilities compatibilities = SG::BeagleBone::Detect::get_compatibilities();
	std::cout << "Number of compatibilities: " << compatibilities.size() << std::endl;
	for ( const auto type : compatibilities )
	{
		std::cout << "- " << SG::BeagleBone::Detect::to_string(type) << std::endl;
	}

	std::cout
			<< "BeagleBone compatible: "		<< SG::BeagleBone::Detect::is_beaglebone_compatible()		<< std::endl
			<< "BeagleBone Black compatible: "	<< SG::BeagleBone::Detect::is_beaglebone_black_compatible()	<< std::endl
			<< "BeagleBone Green compatible: "	<< SG::BeagleBone::Detect::is_beaglebone_green_compatible()	<< std::endl;

	return;
}


void bb_led( const VStr &args )
{
	SG::BeagleBone::LEDControl led_control;
	led_control.start_blink_thread();
	led_control.load_random_pattern();

	size_t counter = 0;
	while ( exit_signal_detected == 0 )
	{
		std::this_thread::sleep_for( std::chrono::milliseconds(100) );

		counter ++;
		if (counter > 100)
		{
			counter = 0;
			led_control.load_random_pattern();
		}
	}

	return;
}


void bb_watchdog( const VStr &args )
{
	SG::BeagleBone::Watchdog watchdog;
	const watchdog_info info = watchdog.get_info();
	std::cout	<< "Watchdog identity:  "														<< info.identity									<< std::endl
				<< "Watchdog firmware:  0x"	<< std::hex << std::setw(4) << std::setfill('0')	<< info.firmware_version							<< std::endl
				<< "Watchdog options:   0x"	<< std::hex << std::setw(4) << std::setfill('0')	<< info.options										<< std::endl
				<< "Watchdog status:    0x"	<< std::hex << std::setw(4) << std::setfill('0')	<< watchdog.get_status()							<< std::endl
				<< "Watchdog boot:      0x"	<< std::hex << std::setw(4) << std::setfill('0')	<< watchdog.get_boot_status()						<< std::endl
				<< "Timeout (secs):     "	<< std::dec << std::setw(1) << std::setfill(' ')	<< watchdog.get_timeout()							<< std::endl
				<< "Remaining (secs):   "	<< std::dec << std::setw(1) << std::setfill(' ')	<< watchdog.get_seconds_remaining_before_reboot()	<< std::endl;

	std::cout << "Changing watchdog timeout to 15 seconds..." << std::endl;
	watchdog.set_timeout( 15 );
	std::cout	<< "New timeout:        "	<< std::dec << std::setw(1) << std::setfill(' ')	<< watchdog.get_timeout()							<< std::endl;
	watchdog.kick();

	size_t counter = 0;
	while ( exit_signal_detected == 0 )
	{
		std::this_thread::sleep_for( std::chrono::milliseconds(100) );

		watchdog.get_seconds_remaining_before_reboot();

		counter ++;
		if (counter > 20)
		{
			counter = 0;
			std::cout << "." << std::flush;
			watchdog.kick();
		}
	}
	std::cout << std::endl;

	for (counter = 0; counter < 20; counter ++ )
	{
		std::cout << counter << ": pausing..." << std::endl;
		std::this_thread::sleep_for( std::chrono::seconds(1) );
		std::cout << "Remaining (secs):   " << std::dec << std::setw(1) << std::setfill(' ') << watchdog.get_seconds_remaining_before_reboot() << std::endl;
	}

	// presumably we wont ever get here, cause the box will have rebooted during the "pause" in the loop above

	std::cout << "Exiting..." << std::endl;

	return;
}


void bb_usb( const VStr &args )
{
	if (args.size() == 1 && args[0] == "on")
	{
		std::cout << "Turning ON power to the USB port." << std::endl;
		SG::BeagleBone::USB::turn_on();
	}
	else if (args.size() == 1 && args[0] == "off")
	{
		std::cout << "Turning OFF power to the USB port." << std::endl;
		SG::BeagleBone::USB::turn_off();
	}
	else
	{
		std::cout << "Toggling USB power." << std::endl;
		SG::BeagleBone::USB::toggle( 500 );
		std::cout << "...pausing..." << std::endl;
		std::this_thread::sleep_for( std::chrono::seconds(5) );

		SG::BeagleBone::USB::turn_off();
		std::cout << "USB power should now be off." << std::endl;
		std::cout << "...pausing..." << std::endl;
		std::this_thread::sleep_for( std::chrono::seconds(5) );

		SG::BeagleBone::USB::turn_on();
		std::cout << "USB power should now be on." << std::endl;
	}

	return;
}


void common_oled( SG::CommonOLED &oled, const VStr &args )
{
	std::cout << "working with " << oled.get_description() << std::endl;

	for (size_t idx = 0; idx < args.size(); idx ++)
	{
		const std::string &cmd = args[idx];
		std::cout << "cmd: " << cmd << std::endl;

		if		(cmd == "on"			) oled.turn_on();
		else if	(cmd == "off"			) oled.turn_off();
		else if (cmd == "normal"		) oled.normal_display();
		else if (cmd == "invert"		) oled.invert_display();
		else if (cmd == "all_on"		) oled.all_on_display();
		else if (cmd == "all_off"		) oled.all_off_display();
		else if (cmd == "reset"			) oled.reset();
		else if (cmd == "lock"			) oled.lock();
		else if (cmd == "unlock"		) oled.unlock();
		else if (cmd == "scroll_start"	) oled.start_scrolling();
		else if (cmd == "scroll_stop"	) oled.stop_scrolling();
//		else if (cmd == "left"			) oled.setup_scrolling( SG::OLED096::EScrollingDirection::kLeftHorizontal	, static_cast<SG::OLED096::EScrollingSpeed>( std::atoi(args[++idx].c_str())) );
//		else if (cmd == "right"			) oled.setup_scrolling( SG::OLED096::EScrollingDirection::kRightHorizontal	, static_cast<SG::OLED096::EScrollingSpeed>( std::atoi(args[++idx].c_str())) );
		else if (cmd == "contrast"		) oled.set_contrast( std::atoi(args[++idx].c_str()) );
		else  std::cout << "ignoring unknown command \"" << cmd << "\"" << std::endl;
	}

	return;
}


void oled112( const VStr &args )
{
	SG::OLED112 oled;
	common_oled( oled, args );

	return;
}


void oled096( const VStr &args )
{
	SG::OLED096 oled;
	common_oled( oled, args );

	return;
}


void ledbar( const VStr &args )
{
	SG::LEDBar led_bar;

	if (args.size() == 0 || (args.size() == 1 && args[0] == "on"))
	{
		std::cout << "Turning ON LED bar." << std::endl;
		led_bar.all_on();
	}
	else if (args.size() == 1 && args[0] == "off")
	{
		std::cout << "Turning OFF LED bar." << std::endl;
		led_bar.all_off();
	}
	else
	{
		std::cout << "Unknown command." << std::endl;
	}

	return;
}


void button( const VStr &args )
{
	SG::Button button;

	std::cout << "Waiting up to 5 seconds for button to be pushed down..." << std::endl;
	bool result = button.wait_for_button_down(5000);
	if (result == false)
	{
		std::cout << "Timeout!" << std::endl;
	}

	std::cout << "Waiting up to 5 seconds for button to be released..." << std::endl;
	result = button.wait_for_button_up(5000);
	if (result == false)
	{
		std::cout << "Timeout!" << std::endl;
	}

	bool was_up = false;
	while ( exit_signal_detected == 0 )
	{
		std::this_thread::sleep_for( std::chrono::milliseconds(100) );

		const bool is_up = button.is_up();

		if (was_up != is_up)
		{
			std::cout << "Button is " << (is_up ? "up" : "down") << std::endl;
			was_up = is_up;
		}
	}

	return;
}


void show_help( void )
{
	std::cout	<< "Command line usage:"								<< std::endl
				<< "  list_all"											<< std::endl
				<< "  list_detect"										<< std::endl
				<< "  list_address"										<< std::endl
				<< "  buzzer"											<< std::endl
				<< "  chainable_rgb_led [r g b | html_colour]"			<< std::endl
				<< "  temperature"										<< std::endl
				<< "  compass"											<< std::endl
				<< "  bb_detect"										<< std::endl
				<< "  bb_led"											<< std::endl
				<< "  bb_usb [on|off]"									<< std::endl
				<< "  bb_watchdog"										<< std::endl
				<< "  oled096 [on|off|normal|invert|all_on|all_off|contrast #|lock|unlock|scroll_start|scroll_stop|left #|right #]" << std::endl
				<< "  oled112 [on|off|normal|invert|all_on|all_off|contrast #|lock|unlock|scroll_start|scroll_stop|left #|right #]" << std::endl
				<< "  ledbar [on|off]"									<< std::endl
				<< "  button"											<< std::endl
				;

	return;
}


int seeedgrove_example_application( int argc, char **argv )
{
	int result = 0;

	std::cout << "-> initializing..." << std::endl;
	SG::SGpp::get().initialize();

	std::string cmd;
	VStr args;

	for (int idx = 1; idx < argc; idx ++)
	{
		if (idx == 1)
		{
			cmd = argv[idx];
			std::cout << "-> processing \"" << cmd << "\"..." << std::endl;
		}
		else
		{
			args.push_back( argv[idx] );
		}
	}

	if		(cmd == "list"				)	list_all();
	else if (cmd == "list_all"			)	list_all();
	else if (cmd == "list_address"		)	list_addresses();
	else if (cmd == "list_addresses"	)	list_addresses();
	else if (cmd == "addresses"			)	list_addresses();
	else if (cmd == "address"			)	list_addresses();
	else if (cmd == "list_detect"		)	list_detect();
	else if (cmd == "detect"			)	list_detect();
	else if (cmd == "chainable_rgb_led"	)	chainable_rgb_led	(args);
	else if (cmd == "buzzer"			)	buzzer				(args);
	else if (cmd == "temperature"		)	temperature			(args);
	else if (cmd == "compass"			)	compass				(args);
	else if (cmd == "bb_detect"			)	bb_detect			(args);
	else if (cmd == "bb_led"			)	bb_led				(args);
	else if (cmd == "bb_watchdog"		)	bb_watchdog			(args);
	else if (cmd == "bb_usb"			)	bb_usb				(args);
	else if (cmd == "oled096"			)	oled096				(args);
	else if (cmd == "oled112"			)	oled112				(args);
	else if (cmd == "ledbar"			)	ledbar				(args);
	else if (cmd == "button"			)	button				(args);
	else if (cmd == "help"				)	show_help();
	else if (cmd == "--help"			)	show_help();
	else if (cmd == ""					)	show_help();
	else
	{
		std::cout << "Unrecognized command \"" << cmd << "\"." << std::endl;
		result = 1;
	}

	return result;
}


int main( int argc, char **argv )
{
	int result = 1;

	try
	{
		setup_signal_handling();

		srand( std::time(nullptr) );

		std::cout	<< "Seeed Grove ++ command line tool"						<< std::endl
					<< "-> version "			<< SG::get_version()			<< std::endl
					<< "-> build timestamp "	<< SG::get_build_timestamp()	<< std::endl;

		result = seeedgrove_example_application( argc, argv );
	}
	catch ( const std::system_error &e )
	{
		std::cout << "Caught " << SG::demangle(typeid(e).name()) << ": ";

		const int value = e.code().value();
		if (value)
		{
			std::cout << "errno " << value << ": ";
		}
		std::cout << e.what() << std::endl;
		result = 2;
	}
	catch ( const std::exception &e )
	{
		std::cout << "Caught " << SG::demangle(typeid(e).name()) << ": " << e.what() << std::endl;
		result = 3;
	}

	if (exit_signal_detected)
	{
		std::cout	<< std::endl
					<< "Signal detected:  " << strsignal(exit_signal_detected) << std::endl
					<< "Stopping " << argv[0] << std::endl;
	}

	return result;
}
