/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_BeagleBone_USB.hpp 1862 2016-05-25 00:06:15Z stephane $
 */

#pragma once

#include <cstddef>


namespace SG
{
	namespace BeagleBone
	{
		/** Manipulate the %USB Type-A port on a %BeagleBone.  Power to the %USB port can be turned on or turned off.
		 * Functions such as @ref SG::BeagleBone::USB::toggle() can be used to turn power off, pause for a short time,
		 * and turn power back on.  This will normally completely reset whatever type of %USB device is connected.
		 *
		 * @warning Note that @p %SG::BeagleBone::USB functions such as @ref toggle(), @ref set(), @ref turn_on(), and
		 * @ref turn_off() blindly toggles a specific memory location.  On a non-BeagleBone computer, this likely has
		 * the ability to break things or cause corruption, depending on what happens to reside at that memory location.
		 * It is strongly suggested that a check be performed to verify that the computer is indeed a %BeagleBone.  For
		 * example, see @ref SG::BeagleBone::Detect::is_beaglebone_compatible().
		 */
		namespace USB
		{
			/** Turn @p ON or turn @p OFF the power to the %USB Type-A port.
			 *
			 * @param [in] turn_on Set to @p TRUE to turn the power on, @p FALSE to turn the power off.
			 * @param [in] sleep_time_in_milliseconds Determines the length of time between when the power is toggled
			 * @p on / @p off.  If set to zero, the power will be set as described by @p turn_on and wont be toggled
			 * further.  With values greater than zero, the power will first be set as described by @p turn_on, then
			 * the thread will pause for the given duration, followed by setting the power to the opposite of what
			 * @p turn_on indicates.
			 *
			 * For example, regardless of the initial state of %USB power, this code will turn the power @p off, pause
			 * for 1 second (1000 milliseconds), and turn the power back @p on:
			 * ~~~~
			 * if (SG::BeagleBone::Detect::is_beaglebone_compatible())
			 * {
			 *     SG::BeagleBone::USB::toggle( false, 1000 );
			 * }
			 * ~~~~
			 *
			 * @see @ref SG::BeagleBone::USB::set()
			 * @see http://stackoverflow.com/questions/24039420/turn-usb-power-off-on-with-beaglebone-black-kernel-3-8
			 * @see http://stackoverflow.com/questions/37086057/turn-off-power-to-a-usb-port
			 */
			void toggle( const bool turn_on, const size_t sleep_time_in_milliseconds );

			/** Turn the power to the %USB Type-A port @p OFF then back @p ON after a short pause.
			 * 
			 * @param [in] sleep_time_in_milliseconds Determines the length of
			 * time between when the power is turned @p off and back @p on.
			 *
			 * @see @ref SG::BeagleBone::USB::set()
			 */
			void toggle( const size_t sleep_time_in_milliseconds = 500 );

			/** Set the power to the %USB Type-A port @p ON or @p OFF.
			 *
			 * @see @ref SG::BeagleBone::USB::turn_off()
			 * @see @ref SG::BeagleBone::USB::turn_on()
			 */
			void set( const bool turn_on );

			/** Turn @p ON the power to the %USB Type-A port.
			 *
			 * @see @ref SG::BeagleBone::USB::set()
			 * @see @ref SG::BeagleBone::USB::turn_off()
			 */
			void turn_on( void );

			/** Turn @p OFF the power to the %USB Type-A port.
			 *
			 * @see @ref SG::BeagleBone::USB::set()
			 * @see @ref SG::BeagleBone::USB::turn_on()
			 */
			void turn_off( void );
		}
	}
}
