/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_BeagleBone_Watchdog.hpp 1855 2016-05-24 00:22:22Z stephane $
 */

#pragma once

#include <string>
#include <thread>
#include <atomic>
#include <linux/watchdog.h>
#include "sg_FD.hpp"


namespace SG
{
	namespace BeagleBone
	{
		/** %BeagleBone hardware watchdog.  Once the watchdog on %BeagleBone Green (BBG) or %BeagleBone Black (BBB) has
		 * been activated, it cannot be stopped.  The default watchdog timeout is 60 seconds, but it can be changed to
		 * a different value with @ref set_timeout().
		 *
		 * Creating one of these objects does @em not automatically activate the watchdog.  A call must be made to
		 * either @ref kick() or @ref start_simple_thread() to activate it.
		 *
		 * @see https://www.kernel.org/doc/Documentation/watchdog/watchdog-api.txt
		 */
		class Watchdog
		{
			public:

				/** Destructor.  If @ref start_simple_thread() has been called, the destructor will ensure that the
				 * secondary thread is stopped when the object goes out of scope.
				 *
				 * @note If the watchdog has been activated, the destruction of the Watchdog object will result in the
				 * device rebooting after the timeout period has been reached.  The only way to prevent a reboot is to
				 * instantiate a new Watchdog object and call @ref start_simple_thread() or resume calling @ref kick().
				 */
				virtual ~Watchdog( void );

				/** Constructor.  This <em>does not</em> activate the watchdog.  The only way to activate the watchdog
				 * is to call @ref start_simple_thread(), or periodically call @ref kick().  Once activated, if the
				 * watchdog isn't updated regularly, the %BeagleBone will reboot.
				 *
				 * There are two ways to use the Watchdog class:
				 *
				 * 1. The method @ref kick() can be regularly called.
				 * 2. A secondary thread can be started by @ref start_simple_thread() which will then automatically
				 * call @ref kick() until the watchdog object goes out of scope.
				 *
				 * First method:
				 * ~~~~
				 * SG::BeagleBone::Watchdog watchdog;
				 * while (true)
				 * {
				 *     do_some_quick_work();	// must return in less than 60 seconds
				 *     watchdog.kick();		// otherwise the device will reboot
				 * }
				 * ~~~~
				 *
				 * Second method:
				 * ~~~~
				 * SG::BeagleBone::Watchdog watchdog;
				 * watchdog.start_simple_thread();	// watchdog is now kicked from a new thread
				 * while (true)
				 * {
				 *     do_some_long_work();	// is OK even if it runs for hours or days
				 * }
				 * ~~~~
				 */
				Watchdog( const size_t timeout_in_seconds = 60 );

				/** Kick the watchdog to prevent it from triggering a reboot.  This needs to be called relatively
				 * often to prevent the watchdog from triggering a reboot.  For example, if the timeout is set to
				 * 60 seconds (the default timeout value) then @ref kick() must be called at intervals of 59 seconds
				 * or less to prevent a reboot.  @see @p WDIOC_KEEPALIVE
				 */
				virtual Watchdog &kick( void );

				/** Attempt to stop kicking the watchdog.  Normally, once the watchdog has been activated, the device
				 * will reboot if too much time elapses between calls to @ref kick().  But if the watchdog driver was
				 * built without @p CONFIG_WATCHDOG_NOWAYOUT, then there may be a way to close the watchdog without
				 * triggering a reboot.
				 */
				virtual Watchdog &magic_close(void);

				/** Get the watchdog timeout (in seconds).  By default, the watchdog timeout is set to 60 seconds.
				 * This determines the maximum length of time that can elapse between calls to @ref kick() without
				 * triggering a reboot.  @see @ref set_timeout()  @see @p WDIOC_GETTIMEOUT
				 */
				virtual size_t get_timeout( void );

				/** Set the watchdog timeout (in seconds).  By default, the watchdog timeout is set to 60 seconds.
				 * This determines the length of time that can elapse between calls to @ref kick() without triggering
				 * a reboot.  @see @ref get_timeout()  @see @p WDIOC_SETTIMEOUT
				 */
				virtual Watchdog &set_timeout( const size_t timeout_in_seconds = 60 );

				/// Get the amount of time left before the watchdog triggers a reboot.  @see @p WDIOC_GETTIMELEFT
				virtual size_t get_seconds_remaining_before_reboot(void);

				/** Get some simple information from the OMAP watchdog driver.  The options are a combination of the
				 * @p WDIOF_... flags defined in watchdog.h.  For example, 0x8180:
				 *
				 * * 0x8000: <tt>WDIOF_KEEPALIVEPING	0x8000	// Keep alive ping reply</tt>
				 * * 0x0100: <tt>WDIOF_MAGICCLOSE		0x0100	// Supports magic close char</tt>
				 * * 0x0080: <tt>WDIOF_SETTIMEOUT		0x0080  // Set timeout (in seconds)</tt>
				 *
				 * @see /usr/include/linux/watchdog.h
				 * @see @p WDIOC_GETSUPPORT
				 */
				virtual watchdog_info get_info(void);

				/** Get some status information on the watchdog.
				 * @see /usr/include/linux/watchdog.h
				 * @see @p WDIOC_GETSTATUS
				 */
				virtual int get_status(void);

				/** Get some information on the cause of the last reboot.
				 * @see /usr/include/linux/watchdog.h
				 * @see @p WDIOC_GETBOOTSTATUS
				 */
				virtual int get_boot_status(void);

				/** Calling this will start @ref run_simple_loop() on a secondary thread.
				 *
				 * Instead of periodically calling @ref kick(), you can have a loop running on a secondary thread to
				 * regularly update the watchdog.  Because this runs on a secondary thread, it wont detect some kinds
				 * of problems such as deadlocks that would be detected by directly calling @ref kick().  But this
				 * greatly simplifies the use of watchdog in some applications, and will trigger a reboot if the
				 * application that instantiated the Watchdog object aborts, or if the device completely stops
				 * responding.
				 *
				 * It is safe to call this multiple times.
				 *
				 * @param [in] sleep_time_in_seconds Determines how long the secondary thread will sleep between calls
				 * to @ref kick().
				 */
				virtual Watchdog &start_simple_thread( const size_t sleep_time_in_seconds = 45 );

				/** Stop the secondary watchdog thread started by @ref start_simple_thread().  If the watchdog is
				 * active, this will eventually cause a reboot once the timeout period is reached.  (Default is 60
				 * seconds.)
				 * 
				 * It is safe to call this multiple times.
				 */
				virtual Watchdog &stop_simple_thread( void );

				/** Never-ending loop that will periodically call @ref kick() to update the watchdog.  This is
				 * typically started by calling @ref start_simple_thread(), and wont return until
				 * @ref stop_simple_thread() is called.
				 *
				 * @param [in] sleep_time_in_seconds Determines how long to sleep between calls to @ref kick().
				 */
				virtual Watchdog &run_simple_loop( const size_t sleep_time_in_seconds );
				/// @}

				/// Determine if the watchdog handle seems to be valid.
				bool is_valid(void) const { return shared_fd->is_valid(); }

			protected:

				/// Path to the watchdog.  Typically will be @p /dev/watchdog.
				std::string path;

				/// File descriptor for the watchdog device.
				SharedFileDescriptor shared_fd;

				/// Secondary watchdog thread.  Only used when @ref start_simple_thread() has been called. @{
				std::thread simple_watchdog_thread;
				std::atomic<bool> watchdog_can_continue;
				/// @}
		};
	}
}
