/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_101020034_3AxisCompass.cpp 1867 2016-05-28 08:09:05Z stephane $
 */

#include "sg_101020034_3AxisCompass.hpp"
#include <thread>
#include <cmath>


SG::ThreeAxisDigitalCompass::~ThreeAxisDigitalCompass( void )
{
	return;
}


SG::ThreeAxisDigitalCompass::ThreeAxisDigitalCompass( const std::string &n, const SG::GroveI2CDigital::I2CAddress addr ) :
		GroveI2CDigital( SG::EGroveType::k3AxisDigitalCompass, n, addr ),
		declination_degrees( 0.0 ),
		declination_radians( 0.0 ),
		most_recent_x( 0 ),
		most_recent_y( 0 ),
		most_recent_z( 0 ),
		most_recent_gain( SG::ThreeAxisDigitalCompass::EGain::k1_2Ga )
{
	return;
}


SG::ThreeAxisDigitalCompass &SG::ThreeAxisDigitalCompass::reset( void )
{
	/** The default configuration is:
	 * * 15 Hz data output rate
	 * * normal bias
	 * * single measurement mode
	 */

	set_config	( EDataOutputRate::k15Hz, EMeasurements::kNormal	);
	set_config	( EGain::k1_2Ga										);
	set_mode	( EMode::kSingleMeasurement							);

	return *this;
}


SG::ThreeAxisDigitalCompass &SG::ThreeAxisDigitalCompass::set_mode( const EMode mode )
{
	switch(mode)
	{
		case EMode::kContinuousMeasurement:
		case EMode::kSingleMeasurement:
		case EMode::kIdle:
		case EMode::kSleep:
		{
			break;
		}
		default:
		{
			throw std::invalid_argument( "SG++ invalid mode \"" + std::to_string((int)mode) + "\" for 3-axis digital compass" );
		}
		
	}

	const uint8_t		value	= static_cast<uint8_t>(mode);
	const I2CRegister	reg		= static_cast<I2CRegister>(ERegister::kModeReg);
	write8( reg, value );

	return *this;
}


SG::ThreeAxisDigitalCompass::EMode SG::ThreeAxisDigitalCompass::get_mode( void )
{
	const I2CRegister	reg		= static_cast<I2CRegister>(ERegister::kModeReg);
	const uint8_t		mode	= read8( reg );

	return static_cast<EMode>(mode);
}


SG::ThreeAxisDigitalCompass &SG::ThreeAxisDigitalCompass::set_config( const EDataOutputRate output_rate, const EMeasurements measurements )
{
	/* bits 4,3,2 is the data output rate
	 * bits 1,0 is the measurement config
	 *
	 * More recent versions of the datasheet seems to indicate that bits 6,5 are used to select the number of samples
	 * averaged per measurement output, where 00=1 (default), 01=2, 10=4, and 11=8
	 */

	const uint8_t		value	= (static_cast<uint8_t>(output_rate) << 2) | static_cast<uint8_t>(measurements);
	const I2CRegister	reg		= static_cast<I2CRegister>(ERegister::kConfigRegA);
	write8( reg, value );

	return *this;
}


SG::ThreeAxisDigitalCompass &SG::ThreeAxisDigitalCompass::get_config( EDataOutputRate &output_rate, EMeasurements &measurements )
{
	output_rate		= EDataOutputRate::k15Hz;
	measurements	= EMeasurements::kNormal;

	const I2CRegister	reg		= static_cast<I2CRegister>(ERegister::kConfigRegA);
	uint8_t				val1	= read8( reg );
	uint8_t				val2	= val1;

	// bits 4,3,2 is the data output rate
	// bits 1,0 is the measurement config
	val1 >>= 2;
	val2 &= 0x03;

	output_rate		= static_cast<EDataOutputRate>	(val1);
	measurements	= static_cast<EMeasurements>	(val2);

	return *this;
}


SG::ThreeAxisDigitalCompass &SG::ThreeAxisDigitalCompass::set_config( const EGain gain )
{
	// bits 7,6,5 is the gain
	// bits 4,3,2,1,0 are reserved

	const uint8_t		value	= static_cast<uint8_t>(gain) << 5;
	const I2CRegister	reg		= static_cast<I2CRegister>(ERegister::kConfigRegB);
	write8( reg, value );

	most_recent_gain = gain;

	return *this;
}


SG::ThreeAxisDigitalCompass &SG::ThreeAxisDigitalCompass::get_config( EGain &gain )
{
	gain = EGain::k1_2Ga;

	const I2CRegister	reg		= static_cast<I2CRegister>(ERegister::kConfigRegB);
	uint8_t				value	= read8( reg );

	// bits 7,6,5 is the gain
	// bits 4,3,2,1,0 are reserved
	value >>= 5;

	gain = static_cast<EGain>(value);

	return *this;
}


SG::ThreeAxisDigitalCompass &SG::ThreeAxisDigitalCompass::get_data( int16_t &x, int16_t &y, int16_t &z )
{
	x = 0;
	y = 0;
	z = 0;

	I2CRegister reg = static_cast<I2CRegister>(ERegister::kXMSB);

	int16_t x_msb = read8( reg ++ );
	int16_t x_lsb = read8( reg ++ );
	int16_t y_msb = read8( reg ++ );
	int16_t y_lsb = read8( reg ++ );
	int16_t z_msb = read8( reg ++ );
	int16_t z_lsb = read8( reg ++ );

	x_msb <<= 8;
	y_msb <<= 8;
	z_msb <<= 8;

	x = x_msb | x_lsb;
	y = y_msb | y_lsb;
	z = z_msb | z_lsb;

	most_recent_x = x;
	most_recent_y = y;
	most_recent_z = z;

	return *this;
}


bool SG::ThreeAxisDigitalCompass::get_single_measurement( int16_t &x, int16_t &y, int16_t &z, const size_t milliseconds_to_wait )
{
	set_mode( EMode::kSingleMeasurement );

	const size_t pause_time_in_milliseconds	= 10; // newer versions of datasheet indicates only 6 milliseconds is needed
	const size_t number_of_times_to_loop	= milliseconds_to_wait / pause_time_in_milliseconds;

	size_t	counter	= 0;
	bool	ready	= false;

	// now we wait for data to be available before attempting to read
	while (true)
	{
		counter ++;
		std::this_thread::sleep_for( std::chrono::milliseconds( pause_time_in_milliseconds ) );
		ready = is_data_ready();

		if (ready)
		{
			break;
		}

		if (milliseconds_to_wait > 0 && counter >= number_of_times_to_loop)
		{
			break;
		}
	}

	get_data( x, y, z );

	return ready;
}


bool SG::ThreeAxisDigitalCompass::wait_for_significant_change( int16_t &x, int16_t &y, int16_t &z, const size_t milliseconds_to_wait, const int16_t delta )
{
	const int16_t original_x = x;
	const int16_t original_y = y;
	const int16_t original_z = z;

	const auto end_time = std::chrono::high_resolution_clock::now() + std::chrono::milliseconds(milliseconds_to_wait);
	bool result = false;

	while (true)
	{
		result = get_single_measurement( x, y, z, milliseconds_to_wait );
		if (result == false)
		{
			// something went wrong, break out of the loop
			break;
		}

		const auto diff_x = std::abs( original_x - x );
		const auto diff_y = std::abs( original_y - y );
		const auto diff_z = std::abs( original_z - z );

		if (diff_x >= delta	||
			diff_y >= delta	||
			diff_z >= delta	)
		{
			// a significant change was detected
			break;
		}

		if (milliseconds_to_wait > 0)
		{
			const auto now = std::chrono::high_resolution_clock::now();
			if (now > end_time)
			{
				// we've run out of time
				result = false;
				break;
			}
		}
	}

	return result;
}


SG::ThreeAxisDigitalCompass &SG::ThreeAxisDigitalCompass::get_status( bool &regulator_enabled, bool &data_locked, bool &data_ready )
{
	regulator_enabled	= false;
	data_locked			= false;
	data_ready			= false;

	const I2CRegister	reg		= static_cast<I2CRegister>(ERegister::kStatusReg);
	const uint8_t		value	= read8( reg );

	// bits 7,6,5,4,3 are reserved
	// bit 2 is the regulator
	// bit 1 is data lock
	// bit 0 is data ready

	regulator_enabled	= (value & 0x04) ? true : false;
	data_locked			= (value & 0x02) ? true : false;
	data_ready			= (value & 0x01) ? true : false;

	return *this;
}


bool SG::ThreeAxisDigitalCompass::is_regulator_enabled( void )
{
	bool regulator_enabled	= false;
	bool data_locked		= false;
	bool data_ready			= false;

	get_status( regulator_enabled, data_locked, data_ready );

	return regulator_enabled;
}


bool SG::ThreeAxisDigitalCompass::is_data_locked( void )
{
	bool regulator_enabled	= false;
	bool data_locked		= false;
	bool data_ready			= false;

	get_status( regulator_enabled, data_locked, data_ready );

	return data_locked;
}


bool SG::ThreeAxisDigitalCompass::is_data_ready( void )
{
	bool regulator_enabled	= false;
	bool data_locked		= false;
	bool data_ready			= false;

	get_status( regulator_enabled, data_locked, data_ready );

	return data_ready;
}


SG::ThreeAxisDigitalCompass &SG::ThreeAxisDigitalCompass::set_declination( const double declination )
{
	declination_degrees = declination;
	declination_radians = declination / (2.0 * M_PI);

	return *this;
}


double SG::ThreeAxisDigitalCompass::get_direction( void )
{
	/** @todo I don't know how to test if this code is working correctly.  Inspiration came from the method
	 * @p Hmc5883l::direction() in https://github.com/intel-iot-devkit/upm/blob/master/src/hmc5883l/hmc5883l.cxx
	 */

	return get_direction( most_recent_x, most_recent_y, most_recent_gain );
}


double SG::ThreeAxisDigitalCompass::get_direction( const int16_t x, const int16_t y )
{
	/** @todo I don't know how to test if this code is working correctly.  Inspiration came from the method
	 * @p Hmc5883l::direction() in https://github.com/intel-iot-devkit/upm/blob/master/src/hmc5883l/hmc5883l.cxx
	 */

	return get_direction( x, y, most_recent_gain );
}


double SG::ThreeAxisDigitalCompass::get_direction( const int16_t x, const int16_t y, const EGain gain )
{
	const double digital_resolution = gain_to_digital_resolution( gain );

	/** @todo I don't know how to test if this code is working correctly.  Inspiration came from the method
	 * @p Hmc5883l::direction() in https://github.com/intel-iot-devkit/upm/blob/master/src/hmc5883l/hmc5883l.cxx
	 */

	return std::atan2( digital_resolution * y, digital_resolution * x ) + declination_radians;
}


double SG::ThreeAxisDigitalCompass::get_heading( void )
{
	/** @todo I don't know how to test if this code is working correctly.  Inspiration came from the method
	 * @p Hmc5883l::heading() in https://github.com/intel-iot-devkit/upm/blob/master/src/hmc5883l/hmc5883l.cxx
	 */

	return get_heading( most_recent_x, most_recent_y, most_recent_gain );
}


double SG::ThreeAxisDigitalCompass::get_heading( const int16_t x, const int16_t y )
{
	/** @todo I don't know how to test if this code is working correctly.  Inspiration came from the method
	 * @p Hmc5883l::heading() in https://github.com/intel-iot-devkit/upm/blob/master/src/hmc5883l/hmc5883l.cxx
	 */

	return get_heading( x, y, most_recent_gain );
}


double SG::ThreeAxisDigitalCompass::get_heading( const int16_t x, const int16_t y, const EGain gain )
{
	/** @todo I don't know how to test if this code is working correctly.  Inspiration came from the method
	 * @p Hmc5883l::heading() in https://github.com/intel-iot-devkit/upm/blob/master/src/hmc5883l/hmc5883l.cxx
	 */

	double d = get_direction(x, y, gain) * 180.0 / M_PI;
	if (d < 0)
	{
		d += 360.0;
	}

	return d;
}


double SG::ThreeAxisDigitalCompass::gain_to_digital_resolution( const SG::ThreeAxisDigitalCompass::EGain gain )
{
	/* From the newer HMC5883L datasheet, page 13:
	 *
	 * The table below shows nominal gain settings. Use the “Gain” column to convert counts to Gauss. The “Digital
	 * Resolution” column is the theoretical value in term of milli-Gauss per count (LSb) which is the inverse of
	 * the values in the “Gain” column.
	 */
	switch(gain)
	{
		case EGain::k0_9Ga: return 1000.0/1280.0;
		case EGain::k1_2Ga: return 1000.0/1024.0;
		case EGain::k1_9Ga: return 1000.0/768.0;
		case EGain::k2_5Ga: return 1000.0/614.0;
		case EGain::k4_0Ga: return 1000.0/415.0;
		case EGain::k4_6Ga: return 1000.0/361.0;
		case EGain::k5_5Ga: return 1000.0/307.0;
		case EGain::k7_9Ga: return 1000.0/219.0;
	}

	// should never get here
	return 1.0;
}


std::string SG::ThreeAxisDigitalCompass::to_string( const EMode mode )
{
	switch(mode)
	{
		case EMode::kContinuousMeasurement:		return "continuous measurement";
		case EMode::kSingleMeasurement:			return "single measurement";
		case EMode::kIdle:						return "idle";
		case EMode::kSleep:						return "sleep";
	}
	
	return "?";
}


std::string SG::ThreeAxisDigitalCompass::to_string( const EDataOutputRate output_rate )
{
	switch(output_rate)
	{
		case EDataOutputRate::k0_75Hz:	return "0.75 Hz";
		case EDataOutputRate::k1_5Hz:	return "1.5 Hz";
		case EDataOutputRate::k3Hz:		return "3 Hz";
		case EDataOutputRate::k7_5Hz:	return "7.5 Hz";
		case EDataOutputRate::k15Hz:	return "15 Hz";
		case EDataOutputRate::k30Hz:	return "30 Hz";
		case EDataOutputRate::k75Hz:	return "75 Hz";
	}

	return "?";
}


std::string SG::ThreeAxisDigitalCompass::to_string( const EMeasurements measurements )
{
	switch(measurements)
	{
		case EMeasurements::kNormal:		return "normal";
		case EMeasurements::kPositiveBias:	return "positive bias";
		case EMeasurements::kNegativeBias:	return "negative bias";
	}

	return "?";
}


std::string SG::ThreeAxisDigitalCompass::to_string( const SG::ThreeAxisDigitalCompass::EGain gain )
{
	switch(gain)
	{
		case EGain::k0_9Ga:	return "+/- 0.9Ga";
		case EGain::k1_2Ga:	return "+/- 1.2Ga";
		case EGain::k1_9Ga:	return "+/- 1.9Ga";
		case EGain::k2_5Ga:	return "+/- 2.5Ga";
		case EGain::k4_0Ga:	return "+/- 4.0Ga";
		case EGain::k4_6Ga:	return "+/- 4.6Ga";
		case EGain::k5_5Ga:	return "+/- 5.5Ga";
		case EGain::k7_9Ga:	return "+/- 7.9Ga";
	}

	return "?";
}
