/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_101020034_3AxisCompass.hpp 1886 2016-06-01 05:35:45Z stephane $
 */

#pragma once

#include "sg_GroveI2CDigital.hpp"


namespace SG
{
	/** 3-axis digital compass.
	 *
	 * Description								| Image
	 * -----------------------------------------|------
	 * Relative size of 3-axis digital compass. | @image html sg_101020034_3AxisDigitalCompass_2.jpg
	 * 3-axis digital compass connected to BBG.	| @image html sg_101020034_3AxisDigitalCompass_1.jpg
	 *
	 * Connect the digital compass to the left-hand-side digital I2C interface.  The I2C address for this device
	 * is @p 0x1e.
	 *
	 * ~~~~
	 * i2cdetect -y -r 2
	 *
	 *      0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f
	 * 00:          -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 10: -- -- -- -- -- -- -- -- -- -- -- -- -- -- 1e --
	 * 20: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 30: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 40: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 50: -- -- -- -- UU UU UU UU -- -- -- -- -- -- -- --
	 * 60: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 70: -- -- -- -- -- -- -- --
	 * ~~~~
	 *
	 * Example source:
	 * ~~~~
	 * SG::ThreeAxisDigitalCompass compass;
	 * compass.reset(); // use default settings
	 * while (! done)
	 * {
	 *     int16_t x = 0;
	 *     int16_t y = 0;
	 *     int16_t z = 0;
	 *     compass.wait_for_significant_change( x, y, z );
	 *     std::cout << "x=" << x << ", y=" << y << ", z=" << z << std::endl;
	 * }
	 * ~~~~
	 *
	 * Example source:
	 * ~~~~
	 * SG::ThreeAxisDigitalCompass compass;
	 * int16_t x = 0;
	 * int16_t y = 0;
	 * int16_t z = 0;
	 * compass.get_data( x, y, z );
	 * compass.set_declination( -3.5 ); // -3 degrees 30 minutes
	 * double heading = compass.get_heading( x, y );
	 * ~~~~
	 *
	 * @see http://www.seeedstudio.com/wiki/Grove_-_3-Axis_Digital_Compass
	 * @see http://www.seeedstudio.com/depot/grove-3axis-compass-p-759.html
	 * @see http://www.seeedstudio.com/wiki/images/4/42/HMC5883.pdf
	 * @see https://aerospace.honeywell.com/~/media/aerospace/files/datasheet/3-axisdigitalcompassichmc5883l_ds.pdf
	 */
	class ThreeAxisDigitalCompass : public GroveI2CDigital
	{
		public:

			/// I2C registers.  @see HMC5883 datasheet page 11
			enum class ERegister
			{
				kConfigRegA			= 0,	///< 0
				kConfigRegB			= 1,	///< 1
				kModeReg			= 2,	///< 2
				kXMSB				= 3,	///< 3 (this register is read-only)
				kXLSB				= 4,	///< 4 (this register is read-only)
				kZMSB				= 5,	///< 5 (this register is read-only)
				kZLSB				= 6,	///< 6 (this register is read-only)
				kYMSB				= 7,	///< 7 (this register is read-only)
				kYLSB				= 8,	///< 8 (this register is read-only)
				kStatusReg			= 9,	///< 9 (this register is read-only)
				kIDRegA				= 10,	///< 10 (this register is read-only)
				kIDRegB				= 11,	///< 11 (this register is read-only)
				kIDRegC				= 12	///< 12 (this register is read-only)
			};

			/** Mode register (0x02).
			 * @see @ref ERegister::kModeReg
			 * @see @ref set_mode()
			 * @see HMC5883 datasheet page 10, and table 16 on page 14
			 */
			enum class EMode
			{
				kContinuousMeasurement	= 0,	///< The device continuously makes measurements and places measured data in data output registers.
				kSingleMeasurement		= 1,	///< The device makes a single measurement, places the measured data in data output registers, then puts itself in idle mode.  This is the default when first powered on.
				kIdle					= 2,	///< The device is accessible through the I2C bus, but major sources of power consumption are disabled, such as, but not limited to, the ADC, the amplifier, the SVDD pin, and the sensor bias current.
				kSleep					= 3		///< The device functionality is limited to listening to the I2C bus.
			};

			/** The rate at which new data readings are done when running in continuous measurement mode.
			 * @see @ref EMode::kContinuousMeasurement
			 * @see @ref set_mode()
			 * @see @ref set_config()
			 */
			enum class EDataOutputRate
			{
				k0_75Hz					= 0,	///< 1333 milliseconds between data updates
				k1_5Hz					= 1,	///< 667 milliseconds between data updates
				k3Hz					= 2,	///< 333 milliseconds between data updates
				k7_5Hz					= 3,	///< 133 milliseconds between data updates
				k15Hz					= 4,	///< 67 milliseconds between data updates (default value)
				k30Hz					= 5,	///< 33 milliseconds between data updates
				k75Hz					= 6		///< 13 milliseconds between data updates
			};

			/// Measurement bias.  @see @ref get_config() @see @ref set_config()
			enum class EMeasurements
			{
				kNormal					= 0,	///< default value
				kPositiveBias			= 1,
				kNegativeBias			= 2
			};

			/** Device gain.  These are specific to the HMC5883.  The newer HMC5883L has different gain values which
			 * are slightly different, and which will also impact @ref gain_to_digital_resolution().  Unfortunately,
			 * both the HMC5883 and HMC5883L identify themselves identically.
			 * @see @ref get_config()
			 * @see @ref set_config()
			 */
			enum class EGain
			{
				k0_9Ga = 0,	///< gain (counts/Gauss) = 1280, digital resolution = 1/1280*1000 = 0.78 mG
				k1_2Ga = 1,	///< gain (counts/Gauss) = 1024, digital resolution = 1/1024*1000 = 0.98 mG (default value)
				k1_9Ga = 2,	///< gain (counts/Gauss) = 768, digital resolution = 1/768*1000 = 1.30 mG
				k2_5Ga = 3,	///< gain (counts/Gauss) = 614, digital resolution = 1/614*1000 = 1.63 mG
				k4_0Ga = 4,	///< gain (counts/Gauss) = 415, digital resolution = 1/415*1000 = 2.41 mG
				k4_6Ga = 5,	///< gain (counts/Gauss) = 361, digital resolution = 1/361*1000 = 2.77 mG
				k5_5Ga = 6,	///< gain (counts/Gauss) = 307, digital resolution = 1/307*1000 = 3.26 mG
				k7_9Ga = 7	///< gain (counts/Gauss) = 219, digital resolution = 1/219*1000 = 4.57 mG
			};

			/// Destructor.
			virtual ~ThreeAxisDigitalCompass( void );

			/// Constructor.  If not specified, the I2C address will be obtained using @ref get_address_from_type().
			ThreeAxisDigitalCompass( const std::string &name = "", const SG::GroveI2CDigital::I2CAddress addr = 0 );

			/// Reset the configuration to default values.
			virtual ThreeAxisDigitalCompass &reset( void );

			/** Set the operating mode.  The default when first powered on is @ref EMode::kSingleMeasurement which
			 * leads to @ref EMode::kIdle after having stored the data.  To get compass data to continuously update,
			 * set the mode to @ref EMode::kContinuousMeasurement, or call @ref get_single_measurement().
			 *
			 * Example source:
			 * ~~~~
			 * SG::ThreeAxisDigitalCompass compass;
			 * compass.set_mode( SG::ThreeAxisDigitalCompass::EMode::kContinuousMeasurement );
			 * // 3Hz update rate means we'll have a new reading available every 333 milliseconds (1000ms/3=333.3ms)
			 * compass.set_config( SG::ThreeAxisDigitalCompass::EDataOutputRate::k3Hz, SG::ThreeAxisDigitalCompass::EMeasurements::kNormal );
			 * while ( ! done )
			 * {
			 *     std::this_thread::sleep_for( std::chrono::milliseconds( 350 ) );
			 *     int16_t xyz[3];
			 *     compass.get_data( xyz[0], xyz[1], xyz[2] );
			 *     do_something_with_readings( xyz );
			 * }
			 * ~~~~
			 *
			 * @see @ref get_mode()
			 */
			virtual ThreeAxisDigitalCompass &set_mode( const EMode mode = EMode::kSingleMeasurement );

			/** Get the operating mode.  @see @ref set_mode()  @see @ref to_string()  @see @ref EMode
			 *
			 * Example source:
			 * ~~~~
			 * SG::ThreeAxisDigitalCompass compass;
			 * // ...
			 * const SG::ThreeAxisDigitalCompass::EMode mode = compass.get_mode();
			 * std::cout << "digital compass mode: " << SG::ThreeAxisDigitalCompass::to_string(mode) << std::endl;
			 * ~~~~
			 */
			virtual EMode get_mode( void );

			/** Set the output rate and measurement bias.
			 *
			 * Example source:
			 * ~~~~
			 * SG::ThreeAxisDigitalCompass compass;
			 * compass.set_mode( SG::ThreeAxisDigitalCompass::EMode::kContinuousMeasurement );
			 * // 7.5Hz update rate means we'll have a new reading available every 133 milliseconds (1000ms/7.5=133.3ms)
			 * compass.set_config( SG::ThreeAxisDigitalCompass::EDataOutputRate::k7_5Hz, SG::ThreeAxisDigitalCompass::EMeasurements::kNormal );
			 * ~~~~
			 *
			 * @see @ref get_config()
			 * @see @ref EDataOutputRate
			 * @see @ref EMeasurements
			 */
			virtual ThreeAxisDigitalCompass &set_config( const EDataOutputRate output_rate, const EMeasurements measurements );

			/** Get the output rate and measurement bias.
			 *
			 * Example source:
			 * ~~~~
			 * SG::ThreeAxisDigitalCompass compass;
			 * SG::ThreeAxisDigitalCompass::EDataOutputRate rate;
			 * SG::ThreeAxisDigitalCompass::EMeasurements measurements;
			 * compass.get_config( rate, measurements );
			 * std::cout << "rate="         << SG::ThreeAxisDigitalCompass::to_string(rate)         << ", "
			 *           << "measurements=" << SG::ThreeAxisDigitalCompass::to_string(measurements) << std::endl;
			 * ~~~~
			 *
			 * @see @ref set_config()
			 * @see @ref EDataOutputRate
			 * @see @ref EMeasurements
			 */
			virtual ThreeAxisDigitalCompass &get_config( EDataOutputRate &output_rate, EMeasurements &measurements );

			/** Set the device gain.
			 *
			 * @warning The documentation for the more recent HMC5883L states the following:
			 * <tt>Note that the very first measurement after a gain change maintains the same gain as the previous
			 * setting.  The new gain setting is effective from the second measurement and on.</tt>
			 *
			 * @see @ref get_config()
			 * @see @ref EGain
			 */
			virtual ThreeAxisDigitalCompass &set_config( const EGain gain );

			/** Get the device gain.
			 * @see @ref set_config()
			 * @see @ref EGain
			 */
			virtual ThreeAxisDigitalCompass &get_config( EGain &gain );

			/** Get the measurement data stored in the HMC5883.  If the mode has been set to
			 * @ref EMode::kContinuousMeasurement, then this call may obtain a new measurement (but also see
			 * @ref EDataOutputRate).
			 *
			 * @note Calling @ref get_data() does not cause the device to obtain a new reading.  It only returns what
			 * was previously read and stored.
			 *
			 * * If using continuous mode, consider calling @ref is_data_ready() prior to @ref get_data().
			 * * If using single measurement mode, then @ref get_data() is likely the wrong call to make.  Instead,
			 * the device can be told to make a new reading and return it by calling either
			 * @ref get_single_measurement() or @ref wait_for_significant_change().
			 *
			 * Example source:
			 * ~~~~
			 * SG::ThreeAxisDigitalCompass compass;
			 * compass.set_mode( SG::ThreeAxisDigitalCompass::EMode::kContinuousMeasurement );
			 * compass.set_config( SG::ThreeAxisDigitalCompass::EDataOutputRate::k15Hz, SG::ThreeAxisDigitalCompass::EMeasurements::kNormal );
			 * // output rate of 15Hz means a new reading should be available every 66.7 milliseconds (1000ms/15=66.7ms)
			 *
			 * while ( ! done )
			 * {
			 *     bool ready = false;
			 *     while (! done && ! ready)
			 *     {
			 *         std::this_thread::sleep_for( std::chrono::milliseconds( 25 ) );
			 *         ready = compass.is_data_ready();
			 *     }
			 *
			 *     if (ready)
			 *     {
			 *         int16_t xyz[3];
			 *         compass.get_data( xyz[0], xyz[1], xyz[2] );
			 *         do_something_with_readings( xyz );
			 *     }
			 * }
			 * ~~~~
			 *
			 * @see @ref get_single_measurement()
			 * @see @ref wait_for_significant_change()
			 */
			virtual ThreeAxisDigitalCompass &get_data( int16_t &x, int16_t &y, int16_t &z );

			/** This sets the mode to @ref EMode::kSingleMeasurement, waits for a new measurement to have been made,
			 * and returns the new data.
			 *
			 * @note After making a measurement in single measurement mode, the device automatically puts itself
			 * into @ref EMode::kIdle.
			 *
			 * This method may be called multiple times, where each time it is called a new measurement will be made.
			 *
			 * @param [out] x Measurement value from channel @p X.
			 * @param [out] y Measurement value from channel @p Y.
			 * @param [out] z Measurement value from channel @p Z.
			 * @param [in] milliseconds_to_wait The length of time to wait for data to be available.  If set to zero,
			 * then @p get_single_measurement() will wait as long as necessary.  It normally takes less than 10
			 * milliseconds for data to be available.
			 * @returns @p TRUE if a reading was obtained.
			 * @returns @p FALSE if a timeout occurred.  This can only happen when @p milliseconds_to_wait is non-zero.
			 *
			 * Source example:
			 * ~~~~
			 * SG::ThreeAxisDigitalCompass compass;
			 * while ( ! done )
			 * {
			 *     int16_t x, y, z;
			 *     bool valid = compass.get_single_measurement( x, y, z );
			 *     if ( ! valid )
			 *     {
			 *         break;
			 *     }
			 *     std::cout << "x=" << x << ", y=" << y << ", z=" << z << std::endl;
			 * }
			 * ~~~~
			 */
			virtual bool get_single_measurement( int16_t &x, int16_t &y, int16_t &z, const size_t milliseconds_to_wait = 1000 );

			/** Repeatedly call @ref get_single_measurement() until the data values are significantly different than
			 * the initial values for @p x, @p y, and @p z.
			 * @param [in,out] x Measurement value from channel @p X.
			 * @param [in,out] y Measurement value from channel @p Y.
			 * @param [in,out] z Measurement value from channel @p Z.
			 * @param [in] milliseconds_to_wait The length of time to wait for a signficant change.  If set to zero,
			 * then @p wait_for_significant_change() will wait forever.
			 * @param [in] delta Determines how different @p X, @p Y, or @p Z must be from their original values for
			 * the difference to be considered "significant".
			 * @returns @p TRUE if a significantly different reading was obtained.
			 * @returns @p FALSE if a timeout occurred.  This can only happen when @p milliseconds_to_wait is non-zero.
			 */
			virtual bool wait_for_significant_change( int16_t &x, int16_t &y, int16_t &z, const size_t milliseconds_to_wait = 0, const int16_t delta = 10 );

			/** Get status information.
			 * @param [out] regulator_enabled This is set when the internal voltage regulator is enabled. This is
			 * cleared when the internal regulator is disabled.
			 * @param [out] data_locked This is set when some but not all of the data output registers have been
			 * read.  When this is set, the data output is locked and new data will not be stored.
			 * @param [out] data_ready This is set once data has been stored in the data registers, and cleared when
			 * the device initiates a write to the data output registers.
			 *
			 * @see @ref is_regulator_enabled()
			 * @see @ref is_data_locked()
			 * @see @ref is_data_ready()
			 */
			virtual ThreeAxisDigitalCompass &get_status( bool &regulator_enabled, bool &data_locked, bool &data_ready );

			/// Alias for the given field from @ref get_status(). @{
			virtual bool is_regulator_enabled	( void );
			virtual bool is_data_locked			( void );
			virtual bool is_data_ready			( void );
			/// @}

			/** Set the magnetic declination to compensate for differences between magnetic north and geographic north.
			 * This is specific to the geographic location where the device runs, and can be obtained from places such
			 * as http://www.magnetic-declination.com/ .
			 * @param [in] declination Set the declination in @b decimal @b degrees @b format.  For example,
			 * 15 degrees 54 minutes of declination would be specified as @p 15.9 (15 + 54/60).
			 * @see http://www.magnetic-declination.com/
			 */
			virtual ThreeAxisDigitalCompass &set_declination( const double declination );

			/// Get the direction in radians.  This uses the most recent values read by @ref get_data().
			virtual double get_direction( void );

			/// Get the direction in radians. @{
			virtual double get_direction( const int16_t x, const int16_t y );
			virtual double get_direction( const int16_t x, const int16_t y, const EGain gain );
			/// @}

			/** Get the compass heading in degrees, where east is @p 0, north is @p 90, west is @p 180, and
			 * south is @p 270.  This uses the most recent values read by @ref get_data().
			 */
			virtual double get_heading( void );

			/** Get the compass heading in degrees, where east is @p 0, north is @p 90, west is @p 180, and
			 * south is @p 270.  @{
			 */
			virtual double get_heading( const int16_t x, const int16_t y );
			virtual double get_heading( const int16_t x, const int16_t y, const EGain gain );
			/// @}

			/// Convert the enum to a string. @{
			static std::string to_string( const EMode			mode			);
			static std::string to_string( const EDataOutputRate	output_rate		);
			static std::string to_string( const EMeasurements	measurements	);
			static std::string to_string( const EGain			gain			);
			/// @}

			/** Convert the gain to digital resolution.
			 * @see @ref EGain
			 * @see The datasheet for the newer model HMC5883L has a better explanation of the gain and the digital resolution.
			 */
			double gain_to_digital_resolution( const EGain gain );

		protected:

			/// Magnetic declination.  @see @ref set_declination() @see http://www.magnetic-declination.com/ @{
			double declination_degrees;
			double declination_radians;
			/// @}

			/// Most recent value read by @ref get_data(). @{
			int16_t most_recent_x;
			int16_t most_recent_y;
			int16_t most_recent_z;
			/// @}

			/// Most recent gain.
			EGain most_recent_gain;
	};
}
