/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_104030006_ChainableRGBLED.hpp 1904 2016-06-15 04:20:07Z stephane $
 */

#pragma once

#include "sg_GroveGPIO.hpp"


namespace SG
{
	/** RGB LED which can be set programatically to any of 16.7 million colours.  Each component (red, green blue) can
	 * be set in intensity to any value between 0 and 255.  Setting the colour to (0, 0, 0) will turn off the light,
	 * while (255, 255, 255) will result in bright white.
	 *
	 * Description | Image
	 * ------------|------
	 * Chainable RGB LED connected to the Grove UART interface. <br/>The light in this example was enabled from the command-line with @p "sudo seeedgrove chainable_rgb_led 255 128 0". | @image html ChainableRGBLED_1.jpg
	 * GROVE cable connects to the @p 'in' side of the chainable RGB LED. | @image html ChainableRGBLED_2.jpg
	 *
	 * @see http://www.seeedstudio.com/wiki/Grove_-_Chainable_RGB_LED
	 * @see http://www.seeedstudio.com/depot/Grove-Chainable-RGB-LED-p-850.html
	 *
	 * @note This is not the same as the variable colour LED, SKU #104020001.  The colour on the chainable RGB LED can
	 * be set programmatically.  The colour on the variable colour LED can only be set by turning tiny screws on the
	 * back of the twig.
	 */
	class ChainableRGBLED : public GroveGPIO
	{
		public:

			/** The index of the LED into the chain.  The first LED is index zero.  Normally, @p idx
			 * should be zero.  Only set this if you have multiple chainable LEDs connected together.
			 */
			size_t idx;

			int red;	///< Red component.  Must be in the range 0-255.  This is the internal value for @p red stored in the C++ object, and may not be the same as the actual LED's red intensity.
			int green;	///< Green component.  Must be in the range 0-255.  This is the internal value for @p green stored in the C++ object, and may not be the same as the actual LED's green intensity.
			int blue;	///< Blue component.  Must be in the range 0-255.  This is the internal value for @p blue stored in the C++ object, and may not be the same as the actual LED's blue intensity.

			/// Determines whether the destructor attempts to turn off the light.  This can be specified in the constructor.
			bool turn_off_LED_when_destructing;

			/// Destructor.  @see @ref turn_off_LED_when_destructing
			virtual ~ChainableRGBLED( void );

			/** Constructor.
			 * @param [in] i Since the LEDs are chainable, @p i is the index of the LED into the chain.  The first LED is idx zero.
			 * @param [in] n User-supplied name which can be used to identify this LED.  The name can be anything, it isn't parsed by SG++.
			 * @param [in] turn_off_at_destruction Normally when an object of this class is destructed, the LED retains
			 * the very last state ("colour").  If @p turn_off_at_destruction is set to @p true, then the LED will be
			 * turned off by the destructor.
			 */
			ChainableRGBLED( const size_t i=0, const std::string &n="", const bool turn_off_at_destruction=false );

			/** Copy constructor.  Both copies will refer to the exact same chainable RGB LED, but wont know about the
			 * other object.  For example:
			 * ~~~~
			 * ChainableRGBLED rgb_led_foo( 2, "testing third light" );
			 * rgb_led_foo.set_RGB( 128, 160, 192 );
			 * 
			 * ChainableRGBLED rgb_led_bar( rgb_led_foo );
			 * rgb_led_bar.set_RGB( 255, 204, 128 );
			 * ~~~~
			 * After that 4th line executes, the original object @p rgb_led_foo has no way of knowing the LED colours
			 * were changed by another object.  The @p red, @p green, and @p blue values in @p reg_led_foo will still
			 * indicate 128, 160, and 192.
			 */
			ChainableRGBLED( const ChainableRGBLED &rhs );

			/** Assignment.  See the comment in copy constructor in regards to multiple C++ objects pointing to the
			 * same physical ChainableRGBLED object.
			 */
			ChainableRGBLED &operator=( const ChainableRGBLED &rhs );

			/// Test object equality. @{
			bool operator==( const ChainableRGBLED &rhs ) const;
			bool operator!=( const ChainableRGBLED &rhs ) const { return ! operator==(rhs); }
			/// @}

			/** Sets the LED to the specified colour, and updates the @ref red, @ref green, and @ref blue internal
			 * variables.  The range of @p r, @p g, and @p b is 0 to 255.  Values outside that range will automatically
			 * be brought within range.
			 */
			ChainableRGBLED &set_RGB( const int r, const int g, const int b );

			/** Sets the LED to the specified colour, and updates the @ref red, @ref green, and @ref blue internal
			 * variables.  The range of @p r, @p g, and @p b is 0 to 255.
			 */
			ChainableRGBLED &set_RGB( const size_t r, const size_t g, const size_t b );

			/** Sets the LED to the specified colour, and updates the @ref red, @ref green, and @ref blue internal
			 * variables.  The range of @p r, @p g, and @p b is 0.0 to 1.0.
			 */
			ChainableRGBLED &set_RGB( const double r, const double g, const double b );

			/** Sets the LED to the specified colour, and updates the @ref red, @ref green, and @ref blue internal
			 * variables.
			 * @param [in] html_colour The string must be formatted like an HTML colour entry, such as @p #00ff88 or
			 * @p #0f8.  Colour names are not recognized.  Values that cannot be parsed will be set to @p 0xff.
			 */
			ChainableRGBLED &set_RGB( std::string html_colour );

			/** Sets the LED to [0,0,0] effectively turning off the LED, but doesn't alter the @ref red, @ref green,
			 * and @ref blue members of this class.  This way, calling @ref turn_on() will re-activate the LED to the
			 * previous preset colour.
			 */
			ChainableRGBLED &turn_off( void );

			/// Sets the LED to the previously set values @ref red, @ref green, and @ref blue defined above.
			ChainableRGBLED &turn_on( void );

			/** Set the LED to the specified colours, but doesn't alter the values @ref red, @ref green, or @ref blue
			 * defined above.  Normally you'd call @ref set_RGB(), unless you want to temporarily change the LED colour
			 * without altering the original saved colour values.
			 * @param [in] r Red component.  Values below zero will be interpreted as zero.  Values above 255 will be interpreted as 255.
			 * @param [in] g Green component.  Values below zero will be interpreted as zero.  Values above 255 will be interpreted as 255.
			 * @param [in] b Blue component.  Values below zero will be interpreted as zero.  Values above 255 will be interpreted as 255.
			 */
			ChainableRGBLED &turn_on( int r, int g, int b );
	};
}
