/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_common_OLED.cpp 1817 2016-05-01 00:18:10Z stephane $
 */

#include "sg_common_OLED.hpp"
#include <algorithm>
#include <thread>


SG::CommonOLED::~CommonOLED( void )
{
	return;
}


SG::CommonOLED::CommonOLED( const SG::EGroveType t, const std::string &n, const SG::GroveI2CDigital::I2CAddress addr ) :
		GroveI2CDigital( t, n, addr ),
		orientation( EOrientation::kInvalid )
{
	set_font( Font::EType::k8x8CapRouge );

	return;
}


SG::CommonOLED &SG::CommonOLED::send_command( const uint8_t cmd )
{
	I2CBlock block;
	block.push_back( 0x80 );
	block.push_back( cmd );
	write_block( block );

	return *this;
}


SG::CommonOLED &SG::CommonOLED::send_command( const uint8_t cmd, const uint8_t value )
{
	I2CBlock block;
	block.push_back( 0x80	);
	block.push_back( cmd	);
	block.push_back( value	);
	write_block( block );

	return *this;
}


SG::CommonOLED &SG::CommonOLED::set_font( const SG::Font::EType type )
{
	/** Fonts included with SG++:
	 *
	 * @image html CapRouge_8x8.png "8x8 Cap Rouge"
	 * @image html CourierNew_8x8.png "8x8 Courier New"
	 * @image html DejaVuSansMono_8x8.png "8x8 DejaVu Sans"
	 * @image html UbuntuMono_8x8.png "8x8 Ubuntu Mono"
	 *
	 * @see @ref SG::Font::EType
	 * @see @ref SG::Font::get()
	 */

	return set_font( SG::Font::get(type) );
}


SG::CommonOLED &SG::CommonOLED::set_font( std::shared_ptr<SG::Font::Monospace> font_to_use )
{
	/** For example:
	 * ~~~~
	 * SG::OLED112 oled( "test grove" );
	 * oled.reset();
	 * set_font( user_supplied_font );
	 * oled.show( "hello" );
	 * ~~~~
	 */

	font		= font_to_use;
	font_map	= font->get_fontmap();

	return *this;
}


SG::CommonOLED &SG::CommonOLED::set_XY( const uint8_t x, const uint8_t y )
{
	// this only works when page addressing mode is enabled

	const uint8_t page_start_address	= 0xb0 + y;
	const uint8_t lower_column_address	= 0x00 + ((8 * x	 ) & 0x0f);
	const uint8_t higher_column_address	= 0x10 + ((8 * x >> 4) & 0x0f);

	const I2CBlock cmds =
	{
		0x80,
		page_start_address,
		lower_column_address,
		higher_column_address
	};
//	write_block( cmds );

	return *this;
}


SG::CommonOLED &SG::CommonOLED::flood( const I2CBlock &pattern )
{
	if (pattern.empty())
	{
		return clear_screen();
	}

//	stop_scrolling();

	I2CBlock cmds =
	{
		0x80,
		0xe3,
		0x21,			// set column address
				0x00,	// set column address: A: start address
				0x7f,	// set column address: B: end address

		0x22,			// set page address
				0x00,	// set page address: A: start address
				0x07,	// set page address: B: end address
	};
	write_block( cmds );

	set_XY( 0, 0 );

	const size_t total_bytes_needed = pixel_width() * pixel_height() / 8;
	I2CBlock block;
	block.push_back( 0x40 );
	while (block.size() < total_bytes_needed)
	{
		block.insert( block.end(), pattern.begin(), pattern.end() );
	}

	// truncate any extra bytes (if any)
	block.resize( total_bytes_needed + 1 );

	write_block( block );

	set_XY( 0, 0 );

	return *this;
}


SG::CommonOLED &SG::CommonOLED::fade_white_to_black( const size_t milliseconds_between_transitions )
{
	const auto pause_time = std::chrono::milliseconds(milliseconds_between_transitions);

	I2CBlock block;
	// start with a white screen
	block = { 0xff };						flood(block);	std::this_thread::sleep_for( pause_time );
	block = { 0xaa, 0x55 };					flood(block);	std::this_thread::sleep_for( pause_time );
	block = { 0x00, 0x55 };					flood(block);	std::this_thread::sleep_for( pause_time );
	block = { 0x00, 0x01, 0x00, 0x10 };		flood(block);	std::this_thread::sleep_for( pause_time );
	block = { 0x00 };						flood(block);
	// screen is now black

	return *this;
}


SG::CommonOLED &SG::CommonOLED::fade_black_to_white( const size_t milliseconds_between_transitions )
{
	const auto pause_time = std::chrono::milliseconds(milliseconds_between_transitions);
	
	I2CBlock block;
	// start with a black screen
	block = { 0x00 };						flood(block);	std::this_thread::sleep_for( pause_time );
	block = { 0x00, 0x01, 0x00, 0x10 };		flood(block);	std::this_thread::sleep_for( pause_time );
	block = { 0x00, 0x55 };					flood(block);	std::this_thread::sleep_for( pause_time );
	block = { 0xaa, 0x55 };					flood(block);	std::this_thread::sleep_for( pause_time );
	block = { 0xff };						flood(block);
	// screen is now white

	return *this;
}


SG::CommonOLED &SG::CommonOLED::show( const char c )
{
	const Font::Monospace::Bitmap bitmap = font->get(c, font_map);

	const size_t w = font->get_width();
	const size_t h = font->get_height();

	if (bitmap.size() != w * h)
	{
		/// @throw std::runtime_error if the font bitmap is the wrong size.
		throw std::runtime_error( "Expected bitmap for character \"" + std::to_string(c) + "\" to be " + std::to_string(w*h/8) + " bytes, but instead found size is " + std::to_string(bitmap.size()) + " bytes" );
	}

	I2CBlock block;
	block.push_back( 0x40 );

	const bool portrait_mode = is_portrait();

	for (size_t idx = 0; idx < 8; idx ++)
	{
		uint8_t byte = '\0';

		if (portrait_mode)
		{
			const size_t offset = idx * 8;
			byte |= (bitmap[offset + 0] ? 0x80 : 0x00);
			byte |= (bitmap[offset + 1] ? 0x40 : 0x00);
			byte |= (bitmap[offset + 2] ? 0x20 : 0x00);
			byte |= (bitmap[offset + 3] ? 0x10 : 0x00);
			byte |= (bitmap[offset + 4] ? 0x08 : 0x00);
			byte |= (bitmap[offset + 5] ? 0x04 : 0x00);
			byte |= (bitmap[offset + 6] ? 0x02 : 0x00);
			byte |= (bitmap[offset + 7] ? 0x01 : 0x00);
		}
		else
		{
			// horizontal (rotate each character 90 degrees)
			byte |= (bitmap[  0 + idx ] ? 0x01 : 0x00 );
			byte |= (bitmap[  8 + idx ] ? 0x02 : 0x00 );
			byte |= (bitmap[ 16 + idx ] ? 0x04 : 0x00 );
			byte |= (bitmap[ 24 + idx ] ? 0x08 : 0x00 );
			byte |= (bitmap[ 32 + idx ] ? 0x10 : 0x00 );
			byte |= (bitmap[ 40 + idx ] ? 0x20 : 0x00 );
			byte |= (bitmap[ 48 + idx ] ? 0x40 : 0x00 );
			byte |= (bitmap[ 56 + idx ] ? 0x80 : 0x00 );
		}

		block.push_back( byte );
	}

#if 0
	// mirror image the font/text
	std::reverse( block.begin() + 1, block.end() );
#endif

	write_block( block );

	return *this;
}


SG::CommonOLED &SG::CommonOLED::show( const std::string &str )
{
	for (size_t idx = 0; idx < str.size(); idx ++)
	{
		show( str[idx] );
	}

	return *this;
}


SG::CommonOLED &SG::CommonOLED::rotate_90(void)
{
	switch(orientation)
	{
		case EOrientation::kLandscape:		return set_orientation( EOrientation::kPortrait		);
		case EOrientation::kPortrait:		return set_orientation( EOrientation::kLandscape180	);
		case EOrientation::kLandscape180:	return set_orientation( EOrientation::kPortrait180	);
		case EOrientation::kPortrait180:	return set_orientation( EOrientation::kLandscape	);
		default:							return set_orientation( EOrientation::kPortrait		);
	}

	return *this;
}


SG::CommonOLED &SG::CommonOLED::rotate_180(void)
{
	switch(orientation)
	{
		case EOrientation::kLandscape:		return set_orientation( EOrientation::kLandscape180	);
		case EOrientation::kPortrait:		return set_orientation( EOrientation::kPortrait180	);
		case EOrientation::kLandscape180:	return set_orientation( EOrientation::kLandscape	);
		case EOrientation::kPortrait180:	return set_orientation( EOrientation::kPortrait		);
		default:							return set_orientation( EOrientation::kPortrait		);
	}

	return *this;
}


SG::CommonOLED &SG::CommonOLED::rotate_270(void)
{
	switch(orientation)
	{
		case EOrientation::kLandscape:		return set_orientation( EOrientation::kPortrait180	);
		case EOrientation::kPortrait:		return set_orientation( EOrientation::kLandscape	);
		case EOrientation::kLandscape180:	return set_orientation( EOrientation::kPortrait		);
		case EOrientation::kPortrait180:	return set_orientation( EOrientation::kLandscape180	);
		default:							return set_orientation( EOrientation::kPortrait		);
	}

	return *this;
}


SG::CommonOLED &SG::CommonOLED::set_orientation( const EOrientation new_orientation )
{
	I2CBlock cmds =
	{
		0x80,
		0xe3,			// no-op
//		0x40,			// set start line
		0x20,	0x00,	// horizontal mode
//		0x20,	0x01,	// vertical mode

		0x21,			// set column address
				0x00,	// set column address: A: start address
				0x7f,	// set column address: B: end address

		0x22,			// set page address
				0x00,	// set page address: A: start address
				0x07,	// set page address: B: end address
	};

	switch(new_orientation)
	{
		case EOrientation::kLandscape:
		{
			cmds.push_back( 0xa0 );		// do not re-map segments
			cmds.push_back( 0xc0 );		// scan normal (COM0-COMn)

			cmds.push_back( 0xd3 );		// set vertical shift
			cmds.push_back( 0x00 );		// set vertical shift (between 0x00 and 0x3f)
			break;
		}
		case EOrientation::kLandscape180:
		{
			cmds.push_back( 0xa1 );		// re-map segments
			cmds.push_back( 0xc8 );		// scan reverse (COMn-COM0)

			cmds.push_back( 0xd3 );		// set vertical shift (between 0x00 and 0x3f)
			cmds.push_back( 0x12 );		// set vertical shift (0x12 == 18d)

			cmds.push_back( 0x40 );		// set start line

			break;
		}
		case EOrientation::kPortrait:
		{
//			cmds.push_back( 0x00 );		// set low column address
//			cmds.push_back( 0x1f );		// set high column address
//			cmds.push_back( 0xb0 );		// set page start address
			
			cmds.push_back( 0xa0 );		// do not remap segments
//			cmds.push_back( 0xa1 );		// remap segments
//			cmds.push_back( 0xc0 );		// scan normal (COM0-COMn)
			cmds.push_back( 0xc8 );		// scan reverse (COMn-COM0) -- mirror image

			cmds.push_back( 0xd3 );		// set vertical shift (between 0x00 and 0x3f)
			cmds.push_back( 0x00 );		// set vertical shift (none)
//			cmds.push_back( 0x12 );		// set vertical shift (0x12 == 18d)
			break;
		}
		case EOrientation::kPortrait180:
		{
			cmds.push_back( 0xa0 );		// do not remap segments
			cmds.push_back( 0xc0 );		// scan normal (COM0-COMn)
			
			cmds.push_back( 0xd3 );		// set vertical shift (between 0x00 and 0x3f)
			cmds.push_back( 0x00 );		// set vertical shift (none)
			break;
		}
		default:
		{
			/// @throw std::invalid_argument if the orientation is invalid.
			throw std::invalid_argument( get_description() + ": unrecognized orientation (\"" + std::to_string( (int)new_orientation ) + "\")" );
		}
	}

	write_block( cmds );

	orientation = new_orientation;

	return *this;
}


#if 0
SG::CommonOLED &SG::CommonOLED::setup_scrolling(
		const EScrollingDirection	direction	,
		const EScrollingSpeed		speed		,
		const uint8_t				start_row	,
		const uint8_t				end_row		,
		const uint8_t				start_col	,
		const uint8_t				end_col		)
{
	if (start_row > 0x7f || end_row > 0x7f )	throw std::invalid_argument( get_description() + ": scrolling row parameter must be <= 0x7F (127)"		);
	if (start_col > 0x3f || end_col > 0x3f )	throw std::invalid_argument( get_description() + ": scrolling column parameter must be <= 0x3F (63)"	);
	if (start_row > end_row)					throw std::invalid_argument( get_description() + ": scrolling row start must be <= row end"				);
	if (start_col > end_col)					throw std::invalid_argument( get_description() + ": scrolling column start must be <= column end"		);

	/* SSD1327 datasheet, section 10.2.1, page 53, says the following about scrolling setup:
	 * 
	 *		"Before issuing this command the horizontal scroll must be deactivated (2Eh).
	 *		Otherwise, RAM content may be corrupted."
	 */
	stop_scrolling();

	// need to send 8-byte block
	const I2CBlock block =
	{
		static_cast<uint8_t>(direction)	,
		0x00 /* padding */				,
		start_row						,
		static_cast<uint8_t>(speed)		,
		end_row							,
		start_col						,
		end_col							,
		0x00 /* padding */
	};

	write_block( 0, block );

	return *this;
}


SG::CommonOLED &SG::CommonOLED::set_column( const uint8_t start_col, const uint8_t end_col )
{
	if (start_col > 0x3f || end_col > 0x3f )	throw std::invalid_argument( get_description() + ": column start and end address must be <= 0x3F (63)"	);
	if (start_col > end_col)					throw std::invalid_argument( get_description() + ": column start must be <= column end"					);

	const I2CBlock block =
	{
		static_cast<uint8_t>(0x15),
		start_col,
		end_col
	};

	write_block( 0, block );

	return *this;
}


SG::CommonOLED &SG::CommonOLED::set_row( const uint8_t start_row, const uint8_t end_row )
{
	if (start_row > 0x7f || end_row > 0x7f )	throw std::invalid_argument( get_description() + ": row start and end address must be <= 0x7F (127)"	);
	if (start_row > end_row)					throw std::invalid_argument( get_description() + ": row start must be <= row end"						);

	const I2CBlock block =
	{
		static_cast<uint8_t>(0x75),
		start_row,
		end_row
	};

	write_block( 0, block );

	return *this;
}
#endif


SG::GroveI2CDigital::I2CBlock SG::CommonOLED::get_display_bitmap(void)
{
	I2CBlock block( 8192, '\0' );
	/// @todo TODO BOOOO booooo

	return block;
}


SG::CommonOLED &SG::CommonOLED::set_display_bitmap( const SG::GroveI2CDigital::I2CBlock &block )
{
	/* 128 pixels x 128 pixels = 16384 pixels, each of which requires 4 bits
	 * (sixteen levels of grayscale) for a total of 16384/2 = 8192 bytes
	 */
	const size_t number_of_bytes = 8192;

	if (block.size() != number_of_bytes)	throw std::invalid_argument( get_description() + ": display bitmap must be exactly 8192 bytes in length" );

#if 0
	size_t counter = 0;
	for ( const uint8_t byte : block )
	{
		if (true)	write_byte( counter & 0xff );
		else		write_byte( byte );
		counter ++;
	}
#endif
#if 0
	for ( size_t idx = 0; idx < number_of_bytes; idx += 32 )
	{
		const uint8_t *ptr_start	= block.data() + idx + 0x00;
		const uint8_t *ptr_end		= block.data() + idx + 0x20;
		const I2CBlock block32( ptr_start, ptr_end );
		write_block( 0x01, block32 );
	}
#endif
#if 0
	write_block(block);
#endif

	return *this;
}
