/* CamCap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: CamCap_util.cpp 2160 2017-02-19 09:18:04Z stephane $
 */

/** @file
 * Utility and miscellaneous functions.
 */

#include "CamCap.hpp"
#include "CamCap_defines.h"
#include <algorithm>
#include <sstream>
#include <limits>
#include <iomanip>


std::string CC::Device::describe( void )
{
	std::stringstream ss;

	ss	<< "CamCap v"			<< CC::get_version()						<< std::endl
		<< "Filename:\t\t\""	<< fname << "\""							<< std::endl
		<< "Initialized:\t\t"	<< (is_initialized() ? "true" : "false")	<< std::endl;

	if (is_initialized() == false)
	{
		return ss.str();
	}

	ss	<< "Driver name:\t\t"	<< capabilities.driver
		<< " v"					<< int((capabilities.version >> 16) & 0xff)
		<< "."					<< int((capabilities.version >>  8) & 0xff)
		<< "."					<< int((capabilities.version >>  0) & 0xff)				<< std::endl
		<< "Device name:\t\t"	<< capabilities.card									<< std::endl
		<< "Location:\t\t"		<< capabilities.bus_info								<< std::endl
		<< "Flags:\t\t\t0x"		<< std::hex << capabilities.capabilities << std::dec	<< std::endl;

	for ( const auto iter : get_capability_flags() )
	{
		ss << "\t\t\t" << iter.second << std::endl;
	}
	for ( const auto iter : get_inputs() )
	{
		const v4l2_input &input = iter.second;
		ss	<< "Device input #" << input.index << ":" << std::endl
			<< "\tname:\t\t" << input.name << std::endl
			<< "\ttype:\t\t" << (input.type == V4L2_INPUT_TYPE_CAMERA ? "V4L2_INPUT_TYPE_CAMERA" : std::to_string(input.type)) << std::endl;
		if (input.audioset)		ss << "\taudio:\t\t"		<< input.audioset		<< std::endl;
		if (input.tuner)		ss << "\ttuner:\t\t"		<< input.tuner			<< std::endl;
		if (input.status)		ss << "\tstatus:\t\t"		<< input.status			<< std::endl;
		if (input.capabilities)	ss << "\tcapabilities:\t"	<< input.capabilities	<< std::endl;
	}
	for ( const auto iter : get_formats() )
	{
		const v4l2_fmtdesc &format = iter.second;
		ss	<< "Image format #" << format.index << ":" << std::endl
			<< "\tbuffer type:\t" << (format.type == V4L2_BUF_TYPE_VIDEO_CAPTURE	? "V4L2_BUF_TYPE_VIDEO_CAPTURE" : std::to_string(format.type)) << std::endl
			<< "\tflags:\t\t" << (	format.flags == V4L2_FMT_FLAG_COMPRESSED		? "V4L2_FMT_FLAG_COMPRESSED"	:
									format.flags == V4L2_FMT_FLAG_EMULATED			? "V4L2_FMT_FLAG_EMULATED"		:
									std::to_string(format.flags)
								) << std::endl
			<< "\tdescription:\t" << format.description << std::endl
			<< "\tFourcc:\t\t" << fourcc_to_string( format.pixelformat ) << std::endl;
	}
	ss	<< "Currently selected format:" << std::endl
		<< "\ttype:\t\t" << (	selected_format.type == V4L2_BUF_TYPE_VIDEO_CAPTURE			? "V4L2_BUF_TYPE_VIDEO_CAPTURE"			:
								selected_format.type == V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE	? "V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE"	:
								std::to_string(selected_format.type) ) << std::endl;
	if (selected_format.type == V4L2_BUF_TYPE_VIDEO_CAPTURE)
	{
		ss	<< "\twidth:\t\t"			<< selected_format.fmt.pix.width										<< std::endl
			<< "\theight:\t\t"			<< selected_format.fmt.pix.height										<< std::endl
			<< "\tpixel format:\t"		<< fourcc_to_string(selected_format.fmt.pix.pixelformat)				<< std::endl
			<< "\tfield:\t\t"			<< (	selected_format.fmt.pix.field == V4L2_FIELD_NONE				? "V4L2_FIELD_NONE"		:
												selected_format.fmt.pix.field == V4L2_FIELD_TOP					? "V4L2_FIELD_TOP"		:
												selected_format.fmt.pix.field == V4L2_FIELD_BOTTOM				? "V4L2_FIELD_BOTTOM"	:
												selected_format.fmt.pix.field == V4L2_FIELD_INTERLACED			? "V4L2_FIELD_INTERLACED" :
												std::to_string(selected_format.fmt.pix.field)
											)																	<< std::endl
			<< "\tbytes per line:\t"	<< selected_format.fmt.pix.bytesperline		<< " bytes"					<< std::endl
			<< "\timage size:\t"		<< selected_format.fmt.pix.sizeimage		<< " bytes"					<< std::endl
			<< "\tcolour space:\t"		<< (	selected_format.fmt.pix.colorspace == V4L2_COLORSPACE_DEFAULT	? "V4L2_COLORSPACE_DEFAULT"	:
												selected_format.fmt.pix.colorspace == V4L2_COLORSPACE_JPEG		? "V4L2_COLORSPACE_JPEG"	:
												selected_format.fmt.pix.colorspace == V4L2_COLORSPACE_SRGB		? "V4L2_COLORSPACE_SRGB"	:
												selected_format.fmt.pix.colorspace == V4L2_COLORSPACE_RAW		? "V4L2_COLORSPACE_RAW"		:
												std::to_string(selected_format.fmt.pix.colorspace)
											)																	<< std::endl;
	}
	for ( const auto iter : buffer_details )
	{
		const BufferDetail &bd = iter.second;
		ss	<< "Memory mapped buffer #" << bd.buffer.index << ":"		<< std::endl
			<< "\ttype:\t\t" << (	bd.buffer.type == V4L2_BUF_TYPE_VIDEO_CAPTURE			? "V4L2_BUF_TYPE_VIDEO_CAPTURE" :
									bd.buffer.type == V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE	? "V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE" :
									std::to_string(bd.buffer.type) )	<< std::endl
			<< "\tbytes used:\t"	<< bd.buffer.bytesused << " bytes"	<< std::endl
			<< "\tflags:\t\t"		<< bd.buffer.flags					<< std::endl
			<< "\tfield:\t\t"		<< (	bd.buffer.field == V4L2_FIELD_NONE			? "V4L2_FIELD_NONE"		:
											bd.buffer.field == V4L2_FIELD_TOP			? "V4L2_FIELD_TOP"		:
											bd.buffer.field == V4L2_FIELD_BOTTOM		? "V4L2_FIELD_BOTTOM"	:
											bd.buffer.field == V4L2_FIELD_INTERLACED	? "V4L2_FIELD_INTERLACED" :
											std::to_string(bd.buffer.field)
										)								<< std::endl
			<< "\tsequence:\t"		<< bd.buffer.sequence				<< std::endl
			<< "\tlength:\t\t"		<< bd.buffer.length << " bytes"		<< std::endl
			<< "\taddress:\t"		<< bd.address						<< std::endl;
	}

	const Dimensions dimensions = get_dimensions();
	if (dimensions.size() > 0)
	{
		ss << "Image dimensions with the current format (" << dimensions.size() << " detected):" << std::endl;
		for ( const auto &d : dimensions )
		{
			const uint32_t w = d.width;
			const uint32_t h = d.height;
			const uint32_t tmp = gcd( w, h );

			ss << "\tw=" << d.width << "\t\th=" << d.height << "\t\t(" << (w/tmp) << ":" << (h/tmp) << ")" << std::endl;
		}
	}

	ss	<< "Streaming:\t\t"				<< (is_streaming_enabled() ? "enabled" : "disabled")			<< std::endl
		<< "Retry count:\t\t"			<< retry_count													<< std::endl
		<< "Use libv4l interface:\t"	<< (use_libv4l ? "true" : "false")								<< std::endl
		<< "Transfer method:\t"			<< (transfer_method == ETransferMethod::kRead ?	"direct read"	:
											transfer_method == ETransferMethod::kMmap ?	"mmap"			:
																						"unknown" )		<< std::endl
		<< "Number of controls:\t"		<< get_all_controls().size()									<< std::endl;

	ss << describe_controls();

	return ss.str();
}


CC::Dimensions CC::Device::get_dimensions( void )
{
	if (is_not_initialized())
	{
		initialize();
	}

	v4l2_frmsizeenum frame_size;
	ZERO(frame_size);
	frame_size.pixel_format = selected_format.fmt.pix.pixelformat;
	xioctl( VIDIOC_ENUM_FRAMESIZES, &frame_size, "VIDIOC_ENUM_FRAMESIZES" );

	Dimensions dimensions;
	if (frame_size.type == V4L2_FRMSIZE_TYPE_DISCRETE)
	{
		int			rc	= 0;
		uint32_t	idx	= 0;

		while (rc == 0)
		{
			dimensions.insert( Dimension(frame_size.discrete.width, frame_size.discrete.height) );

			idx ++;
			frame_size.index = idx;
			rc = xioctl( VIDIOC_ENUM_FRAMESIZES, &frame_size );
		}
	}

	return dimensions;
}


CC::VCtrls CC::Device::get_all_controls( void )
{
	if (is_not_initialized())
	{
		initialize();
	}

	VCtrls v;

	v4l2_queryctrl queryctrl;
	ZERO(queryctrl);
	queryctrl.id = V4L2_CTRL_FLAG_NEXT_CTRL;
	while ( xioctl(VIDIOC_QUERYCTRL, &queryctrl) == 0)
	{
		v.push_back( queryctrl );
		queryctrl.id |= V4L2_CTRL_FLAG_NEXT_CTRL;
	}

	return v;
}


CC::Device &CC::Device::set_all_controls_to_defaults( void )
{
	if (is_not_initialized())
	{
		initialize();
	}

	const VCtrls v = get_all_controls();
	for ( const auto &ctrl : v )
	{
		try
		{
			set_control( ctrl.id, ctrl.default_value );
		}
		catch (const std::exception &e)
		{
			// ignore the error -- maybe this control cannot be changed right now because it is disabled
		}
	}

	return *this;
}


CC::Device &CC::Device::set_control( const uint32_t index, const int32_t value )
{
	v4l2_control control;
	ZERO(control);
	control.id		= index;
	control.value	= value;
	xioctl( VIDIOC_S_CTRL, &control, "VIDIOC_S_CTRL" );

	return *this;
}


int32_t CC::Device::get_control( const uint32_t index )
{
	v4l2_control control;
	ZERO(control);
	control.id = index;
	xioctl( VIDIOC_G_CTRL, &control, "VIDIOC_G_CTRL" );

	return control.value;
}


uint32_t CC::Device::control_name_to_index( std::string name )
{
	if (name.empty())
	{
		/// @throw std::invalid_argument if the control name is empty
		throw std::invalid_argument( "control name cannot be empty" );
	}

	// convert everything to lowercase
	std::transform( name.begin(), name.end(), name.begin(), ::tolower );

	// create a map of lowercase control names and the corresponding indexes
	MU32Str m;
	for (const auto ctrl : get_all_controls())
	{
		std::string tmp = (char*)ctrl.name;
		std::transform( tmp.begin(), tmp.end(), tmp.begin(), ::tolower );
		m[ ctrl.id ] = tmp;
	}

	// now see if we have a perfect match
	for ( const auto iter : m )
	{
		if (iter.second == name)
		{
			// perfect match -- stop looking!
			return iter.first;
		}
	}

	// no perfect match -- see if we can find a partial word match
	for ( const auto iter : m )
	{
		if (iter.second.find(name) != std::string::npos)
		{
			// control contains "name", so consider this a partial match
			return iter.first;
		}
	}

	// maybe the name we've been given is actually a hex value?

	// trim any leading '0' or 'x' characters, so "0x001ab3f" gets converted to "1ab3f"
	while (name[0] == '0' || name[0] == 'x' || name[0] == 'X')
	{
		name.erase( 0, 1 );
	}

	uint32_t index = 0;
	std::stringstream ss;
	ss << name;
	ss >> std::hex >> index;

	if (index != 0)
	{
		return index;
	}

	/// @throw std::runtime_error if a control with the given name cannot be found
	throw std::runtime_error( "cannot find a control with the name \"" + name + "\"" );
}


CC::Device &CC::Device::set_control( const std::string &name, const int32_t value )
{
	return set_control( control_name_to_index(name), value );
}


int32_t CC::Device::get_control( const std::string &name )
{
	return get_control( control_name_to_index(name) );
}


std::string CC::Device::describe_controls( void )
{
	const size_t f1 = 6;	// id
	const size_t f2 = 5;	// min
	const size_t f3 = 5;	// default
	const size_t f4 = 5;	// max
	const size_t f5 = 6;	// current

	std::stringstream ss;
	ss	<< "\t" << std::setw(f1) << "Index"  << std::setw(f2) << "Min"  << std::setw(f3) << "Def"  << std::setw(f4) << "Max"  << std::setw(f5) << "Value" << " Description" << std::endl
		<< "\t" << std::setw(f1) << "------" << std::setw(f2) << "----" << std::setw(f3) << "----" << std::setw(f4) << "----" << std::setw(f5) << "-----" << " -----------" << std::endl;

	for (const auto &ctrl : get_all_controls())
	{
		ss	<< "\t"
			<< std::setw(f1) << std::hex << ctrl.id << std::dec
			<< std::setw(f2) << std::setw(5) << ctrl.minimum
			<< std::setw(f3) << ctrl.default_value
			<< std::setw(f4) << ctrl.maximum
			<< std::setw(f5) << get_control(ctrl.id)
			<< " " << ctrl.name << std::endl;
	}

	return ss.str();
}
