#!/bin/bash
# C Code Run (C) 2016-2017 Stephane Charette <stephanecharette@gmail.com>
# $Id: ccr_log.sh 2086 2017-01-09 10:15:39Z stephane $
#
# https://www.ccoderun.ca/bbtools/
#
# Set LEDs on and off.


led_base="$(find /sys/devices -type d -name 'beaglebone:green:usr0')"
if [ -z "${led_base}" ]; then
	echo "No LEDs found."
	exit 1
fi
led_base="${led_base::-1}"


function set_all()
{
	if [ "$1" = "" -o "$2" = "" ]; then
		echo "Invalid call to function set_all"
		exit 1
	fi

	for i in {0..3}; do
		echo "$1" > "${led_base}${i}/$2"
	done

	return
}


if [ "$1" = "" -o "$1" = "help" -o "$1" = "--help" -o "$1" = "-h" ]; then
	cat <<-eof
		$0
		Parameters:
		    permissions  Set permissions on "${led_base}*/brightness".
		                 This must be called as root, and happens automatically
		                 on every reboot to ensure the lights can be toggled.
		    on           Turn the given lights on.
		    off          Turn the given lights off.
		    dark|none    Turn all the lights off.
		    bright|all   Turn all the lights on.
		    auto         Set all 4 LEDs back to their automatic settings.
		    0|1|2|3      LEDs 0 through 3.
		    blink        Blinks all lights on and off.
		    alternate    Alternate lights, 0 & 2, then 1 & 3.
		    sweep        Typical sweep pattern.
		    random       Randomly turn lights on and off.

		For example:
		    $0 on 1 3 off 0 2
		    $0 auto
		    $0 bright off 3
		    $0 sweep

		When using blink, alternate, sweep, or random, pattern will keep running until the script is stopped.
		eof
	exit 1
fi


if [ "$1" = "permissions" ]; then
	if (( EUID != 0 )); then
		echo "must be root to set write permissions on ${led_base}*/brightness"
		exit 1
	fi
	for i in {0..3}; do
		for name in brightness trigger; do
			chmod a+w "${led_base}${i}/${name}"
		done
	done
	exit 0
fi


max_brightness="$(cat ${led_base}0/max_brightness)"
val=${max_brightness}
for var in "$@"; do
	case "${var}" in
		on)
			val=${max_brightness}
			;;
		off)
			val=0
			;;
		dark|none)
			set_all "none" "trigger"
			set_all "0" "brightness"
			;;
		bright|all)
			set_all "none" "trigger"
			set_all "${max_brightness}" "brightness"
			;;
		auto)
			set_all "0" "brightness"
			echo "heartbeat"	> ${led_base}0/trigger
			echo "mmc0"			> ${led_base}1/trigger
			echo "none"			> ${led_base}2/trigger
			echo "mmc1"			> ${led_base}3/trigger
			;;
		[0-3])
			echo "none"			> ${led_base}${var}/trigger
			echo ${val}			> ${led_base}${var}/brightness
			;;
		blink)
			set_all "none" "trigger"
			while true; do
				set_all "0" "brightness"
				sleep 0.5
				set_all "${max_brightness}" "brightness"
				sleep 0.5
			done
			;;
		alternate)
			set_all "none" "trigger"
			while true; do
				echo 0					> ${led_base}0/brightness
				echo ${max_brightness}	> ${led_base}1/brightness
				echo 0					> ${led_base}2/brightness
				echo ${max_brightness}	> ${led_base}3/brightness
				sleep 0.5
				echo 0					> ${led_base}3/brightness
				echo ${max_brightness}	> ${led_base}2/brightness
				echo 0					> ${led_base}1/brightness
				echo ${max_brightness}	> ${led_base}0/brightness
				sleep 0.5
			done
			;;
		sweep)
			set_all "none" "trigger"
			j=0
			while true; do
				for loop in {0..7}; do
					i=${loop}
					if [ $i -gt 3 ]; then
						i=$((7 - i))
					fi
					echo ${max_brightness} > "${led_base}${i}/brightness"
					if [ ${i} -ne ${j} ]; then
						echo 0 > "${led_base}${j}/brightness"
						sleep 0.2
					fi
					j=${i}
				done
			done
			;;
		random)
			set_all "none" "trigger"
			while true; do
				idx=$(( RANDOM % 4 ))
				val=$(cat ${led_base}${idx}/brightness)
				if [ ${val} -eq 0 ]; then
					val=${max_brightness}
				else
					val=0
				fi
				echo ${val} > ${led_base}${idx}/brightness
				sleep 0.1
			done
			;;
		*)
			echo "Unknown command '${var}'."
			;;
	esac
done


exit 0
