#!/bin/bash
# C Code Run (C) 2016-2017 Stephane Charette <stephanecharette@gmail.com>
# $Id: ccr_set_timezone.sh 2116 2017-01-18 13:13:47Z stephane $
#
# https://www.ccoderun.ca/bbtools/
#
# Attempt to auto-detect the timezone and set the device accordingly.
# Will retry multiple times if the network or server is unavilable.


# must be (or should be!) root to change the timezone
if (( EUID != 0 )); then
	logger --id --priority user.error --stderr "$0: must be root to change timezone"
	exit 1
fi


# (Initially I was going to try and use one of these public DNS servers to do
# a lookup of geoip.ubuntu.com, but now this script simply looks up one of
# these "public" IP addresses to ensure we have network capability.)
#
# Try several different common (and public) DNS servers:
dns=(
	# Google:
	"8.8.8.8"
	"8.8.4.4"
	"2001:4860:4860::8888"
	"2001:4860:4860::8844"
	# Level3:
	"209.244.0.3"
	"209.244.0.4"
	# Verisign:
	"64.6.64.6"
	"64.6.65.6"
	)


idx=$((RANDOM % ${#dns[@]}))
ip_address="${dns[${idx}]}"
host "${ip_address}"
if [ $? -ne 0 ]; then
	logger --id --priority user.warning --stderr "$0: failed to lookup DNS server ${ip_address}"
else
	# assume we have internet capability if we can get here (we successfully looked up one of the DNS servers)
	current_tz=$(timedatectl status | grep --perl-regex --only-matching "(?<=Time zone: )[^\n]+")
	detected_tz=$(wget --quiet --output-document=- http://geoip.ubuntu.com/lookup | grep --perl-regex --only-matching '(?<=<TimeZone>)[^<]+')
	if [ $? -ne 0 ]; then
		logger --id --priority user.warning --stderr "$0: failed to retrieve timezone from geoip server (current TZ is \"${current_tz}\")"
		if [ -x /usr/bin/smk ]; then
			smk add 1 "Failed to retrieve timezone from geoip."
		fi
	else
		if [ -x /usr/bin/smk ]; then
			smk del "Failed to retrieve timezone from geoip"
		fi
		detected_tz="${detected_tz:-UTC}"
		if [[ "${current_tz}" =~ "${detected_tz}" ]]; then

			# we're done, timezone is correctly set
			logger --id --priority user.notice --stderr "$0: timezone set to ${detected_tz} -- now setting the time"

			# stop the full ntpd with all the overhead, open ports, etc, and instead call sntp ONCE to set the current time
			systemctl stop ntp.service
			systemctl disable ntp.service
			sntp --syslog --settod pool.ntp.org
			if [ $? -eq 0 ]; then
				# we're finally done setting both the timezone and the time
				exit 0
			fi
		else
			logger --id --priority user.notice --stderr "$0: changing timezone from \"${current_tz}\" to \"${detected_tz}\""
			timedatectl set-timezone ${detected_tz}
			timedatectl status
			# sleep and try again to ensure the timezone is set correctly
		fi
	fi
fi	

# wait a short period of time before we try again
sleep 30

# restart the script (in case a new version was installed)
exec $0
