/* JPEGWrap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: JWCompress.cpp 2139 2017-02-02 11:17:46Z stephane $
 */

#include "JWCompress.hpp"
#include <stdexcept>
#include <vector>
#include <cstring>
#define ZERO(x) memset( &x, '\0', sizeof x )

JWCompress::JWCompress( void )
{
	ZERO(cinfo);
	cinfo_ptr = &cinfo;

	cinfo.err = jpeg_std_error( jerr_ptr );
	jpeg_create_compress( cinfo_ptr );

	set_colour();
	set_default_parameters();

	return;
}


JWCompress::~JWCompress( void )
{
	jpeg_destroy_compress( cinfo_ptr );

	return;
}


JWCompress &JWCompress::compress( void )
{
	throw std::logic_error( "JWCompress::compress() not yet defined" );

	return *this;
}


JWCompress &JWCompress::compress_yuv422( const uint8_t * const data, const size_t bytes_per_row  )
{
	if (file == nullptr)
	{
		/// @throw std::logic_error if the output file has not been set
		throw std::logic_error( "output file has not been set" );
	}
	if (cinfo.image_width < 1 ||
		cinfo.image_height < 1 )
	{
		/// @throw std::logic_error if the image width or height is invalid or has not been set
		throw std::logic_error( "image width or height has not been set" );
	}
	if (data == nullptr)
	{
		/// @throw std::invalid_argument if the image data pointer is null
		throw std::invalid_argument( "image data pointer is null" );
	}
	if (bytes_per_row < 1)
	{
		/// @throw std::invalid_argument if the bytes per row is invalid
		throw std::invalid_argument( "image bytes per row is invalid" );
	}

	/** @note JCS_YCbCr in libjpeg needs 24-bit values.  This means 8 bits each for Y, Cb, and Cr.
	 * This method automatically takes 16-bit YUV 422 values and converts them to full 24-bit values
	 * prior to calling libjpeg to do the actual encoding.
	 *
	 * @warning This method automatically calls @ref set_colour() to set the number of components to 3 and the colour
	 * space to JCS_YCbCr.
	 */
	set_colour( 3, JCS_YCbCr );

	jpeg_start_compress( cinfo_ptr, true );

	std::vector<uint8_t> row;
	row.reserve( 3 * cinfo.image_width );

	while (cinfo.next_scanline < cinfo.image_height)
	{
		row.clear();
		const size_t offset = cinfo.next_scanline * bytes_per_row;

		// convert one row at a time from 16-bit YUV 4:2:2 to something that libjpeg understands
		size_t idx = 0;
		while (idx < bytes_per_row)
		{
			/* Read 4 bytes of YUV, write out 6 bytes of YUV.  Format is:  Y0 U0 Y1 V0 Y2 U2 Y3 V2
			 *
			 * To recombine it into full 24-bit colour, duplicate the U and V components like this:
			 *
			 *		Y0 U0 V0 , Y1 U0 V0 , Y2 U0 V0 - ...
			 */
			const uint8_t y1		= data[offset + idx + 0];
			const uint8_t u			= data[offset + idx + 1];
			const uint8_t y2		= data[offset + idx + 2];
			const uint8_t v			= data[offset + idx + 3];
			row.push_back( y1 );
			row.push_back( u );
			row.push_back( v );
			row.push_back( y2 );
			row.push_back( u );
			row.push_back( v );
			idx += 4;

			/* For the record, if we instead had needed it as RGB, the conversion is:
			 *
			 *		R = Y + 1.4075(V-128)
			 *		G = Y - 0.3455(U-128) - 0.7169(V-128)
			 *		B = Y + 1.7790(U-128)
			 */
		}

		JSAMPROW row_pointer[1] = { row.data() };

		jpeg_write_scanlines( cinfo_ptr, row_pointer, 1 );
	}

	jpeg_finish_compress( cinfo_ptr );

	fclose( file );
	file = nullptr;

	return *this;
}


JWCompress &JWCompress::set_size( const size_t width, const size_t height )
{
	cinfo.image_width	= width;
	cinfo.image_height	= height;

	return *this;
}


JWCompress &::JWCompress::set_colour( const int components, const J_COLOR_SPACE colour_space )
{
	cinfo.num_components	= components;
	cinfo.input_components	= components;
	cinfo.in_color_space	= colour_space;

	return *this;
}


JWCompress &JWCompress::set_default_parameters( const J_COLOR_SPACE colour_space )
{
	cinfo.in_color_space = colour_space;
	jpeg_set_defaults( cinfo_ptr );

	return *this;
}


JWCompress &JWCompress::set_quality( const int quality, const bool force_baseline )
{
	jpeg_set_quality( cinfo_ptr, quality, force_baseline );

	return *this;
}
