// TinyAES++ -- AES encryption/decryption for C and C++
// Public domain; see http://unlicense.org/
// Stephane Charette <stephanecharette@gmail.com>
// $Id: test_cbc.cpp 2416 2017-12-04 23:44:48Z stephane $

#include "TinyAES.hpp"
#include <gtest/gtest.h>


TEST(AES_CBC, StringEncryption)
{
	TinyAES::VBytes key;
	TinyAES::VBytes iv;
	TinyAES::generate_random_key_and_iv(key, iv);

	const std::string original_text = "Hello, World";
	std::cout << "original =\"" << original_text << "\"" << std::endl;

	const std::string encrypted = TinyAES::cbc_encrypt(original_text, key, iv);
	std::cout << "encrypted=\"" << std::string(encrypted.size(), '*') << "\" (" << encrypted.size() << " bytes)" << std::endl;

	// with padding, the text should now be 16 bytes long
	ASSERT_EQ(encrypted.size(), 16);

	const std::string decrypted = TinyAES::cbc_decrypt(encrypted, key, iv);
	std::cout << "decrypted=\"" << decrypted << "\"" << std::endl;

	ASSERT_EQ(decrypted.size(), 12);
	ASSERT_EQ(original_text, decrypted);
}


TEST(AES_CBC, EmptyEncryption)
{
	TinyAES::VBytes key;
	TinyAES::VBytes iv;
	TinyAES::generate_random_key_and_iv(key, iv);

	const std::string encrypted = TinyAES::cbc_encrypt("", key, iv);

	// 16 bytes of padding should have been added
	ASSERT_EQ(encrypted.size(), 16);

	const std::string decrypted = TinyAES::cbc_decrypt(encrypted, key, iv);

	ASSERT_EQ(decrypted.size(), 0);
}


TEST(AES_CBC, DuplicateEncryption)
{
	TinyAES::VBytes key1;
	TinyAES::VBytes key2;

	TinyAES::VBytes iv1;
	TinyAES::VBytes iv2;

	TinyAES::generate_random_key_and_iv(key1, iv1);
	TinyAES::generate_random_key_and_iv(key2, iv2);

	ASSERT_NE(key1, key2);
	ASSERT_NE(iv1, iv2);

	const std::string encrypted1 = TinyAES::cbc_encrypt("testing", key1, iv1);	// 1, 1
	const std::string encrypted2 = TinyAES::cbc_encrypt("testing", key2, iv2);	// 2, 2
	const std::string encrypted3 = TinyAES::cbc_encrypt("testing", key1, iv2);	// 1, 2
	const std::string encrypted4 = TinyAES::cbc_encrypt("testing", key2, iv1);	// 2, 1

	ASSERT_NE(encrypted1, encrypted2);
	ASSERT_NE(encrypted1, encrypted3);
	ASSERT_NE(encrypted1, encrypted4);
	ASSERT_NE(encrypted2, encrypted3);
	ASSERT_NE(encrypted2, encrypted4);
	ASSERT_NE(encrypted3, encrypted4);
}


TEST(AES_CBC, Appendix_F_2_5_block1)
{
	// see http://nvlpubs.nist.gov/nistpubs/Legacy/SP/nistspecialpublication800-38a.pdf
	// 256-bit CBC starts at F2.5, starts on the bottom of page 28
	//
	// Note:  The document assumes zero padding, while TinyAES++ assumes
	// and automatically adds the necessary PKCS padding at the end of the
	// block.  Thus, the encrypted vectors will definitely not match.

	const TinyAES::VBytes key =
	{
		0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
		0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4
	};
	const TinyAES::VBytes iv =
	{
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
	};

	const TinyAES::VBytes block1_decrypted	= { 0x6b, 0xc0, 0xbc, 0xe1, 0x2a, 0x45, 0x99, 0x91, 0xe1, 0x34, 0x74, 0x1a, 0x7f, 0x9e, 0x19, 0x25 };
	const TinyAES::VBytes block1_encrypted	= { 0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba, 0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6 };

	const TinyAES::VBytes v1 = TinyAES::cbc_encrypt(block1_decrypted, key, iv);
	const TinyAES::VBytes v2 = TinyAES::cbc_decrypt(v1, key, iv);

//	ASSERT_EQ(v1, block1_encrypted);
	ASSERT_EQ(v2, block1_decrypted);
}
