/* CamCap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: CamCap_capture.cpp 2157 2017-02-18 20:05:58Z stephane $
 */

/** @file
 * Methods related capturing images.
 */

#include "CamCap.hpp"
#include "CamCap_defines.h"
#include <limits>
#include <JWCompress.hpp>


CC::Device &CC::Device::stream_start( void )
{
	if (is_not_initialized())
	{
		initialize();
	}

	if (streaming_enabled == false)
	{
		if (buffer_details.empty() && transfer_method == CC::ETransferMethod::kMmap)
		{
			request_memory_mapped_buffers();
		}

		int type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		xioctl( VIDIOC_STREAMON, &type, "VIDIOC_STREAMON" );
		streaming_enabled = true;
	}

	return *this;
}


CC::Device &CC::Device::stream_stop( void )
{
	if (is_not_initialized())
	{
		initialize();
	}

	if (streaming_enabled)
	{
		int type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		xioctl( VIDIOC_STREAMOFF, &type, "VIDIOC_STREAMOFF" );
		streaming_enabled = false;
	}

	return *this;
}


CC::Bytes CC::Device::capture( void )
{
	Bytes bytes;
	capture( bytes );

	return bytes;
}


v4l2_buffer CC::Device::capture( Bytes &bytes )
{
	if (is_not_initialized())
	{
		initialize();
	}

	v4l2_buffer buffer;

	switch (transfer_method)
	{
		case CC::ETransferMethod::kMmap:
		{
			buffer = capture_using_mmap_buffers( bytes );
			break;
		}
		case CC::ETransferMethod::kRead:
		{
			buffer = capture_using_read( bytes );
			break;
		}
	}

	return buffer;
}


v4l2_buffer CC::Device::capture_to_jpeg( const std::string &output_filename )
{
	if (output_filename.empty())
	{
		/// @throw std::invalid_argument if the filename is empty
		throw std::invalid_argument( "output jpeg filename cannot be empty" );
	}

	if (is_not_initialized())
	{
		initialize();
	}

	// see if the format we're currently using can be easily converted to JPEG -- if not, fix what we can
	bool must_fix_format = false;
	v4l2_format format = get_format();
	if (format.fmt.pix.pixelformat != V4L2_PIX_FMT_YUYV )
	{
		format.fmt.pix.pixelformat = V4L2_PIX_FMT_YUYV;
		must_fix_format = true;
	}

	if (format.fmt.pix.width > std::numeric_limits<uint16_t>::max())
	{
		// JPEG has a size limit of 65535 (uint16)
		format.fmt.pix.width = std::numeric_limits<uint16_t>::max();
		must_fix_format = true;
	}
	if (format.fmt.pix.height > std::numeric_limits<uint16_t>::max())
	{
		format.fmt.pix.width = std::numeric_limits<uint16_t>::max();
		must_fix_format = true;
	}

	if (must_fix_format)
	{
		// we've had to change something in the format
		set_format(format);
	}

	Bytes bytes;
	v4l2_buffer info = capture( bytes );

	JWCompress jpeg;
	jpeg
		.set_output_file( output_filename )
		.set_size( format.fmt.pix.width, format.fmt.pix.height )
		.set_default_parameters( JCS_YCbCr )
		.use_huffman_coding()
		.use_float_dct()
		.set_smoothing( 1 )
		.set_quality( 85 )
		.compress_yuv422( bytes.data(), format.fmt.pix.bytesperline );

	return info;
}


CC::Device &CC::Device::capture_many( const size_t count, Images &images )
{
	images.clear();
	images.resize( count );

	stream_start();

	for (size_t i = 0; i < count; i++)
	{
		CaptureAndBuffer &capture_and_buffer = images[i];
		capture_and_buffer.info = capture( capture_and_buffer.bytes );
	}

	return *this;
}


CC::VStr CC::Device::capture_many_to_jpeg( const size_t count, const std::string &directory )
{
	VStr v;

	stream_start();

	for (size_t i = 0; i < count; i++)
	{
		const std::string fname = directory + "/" + std::to_string(i) + ".jpg";
		capture_to_jpeg( fname );
		v.push_back( fname );
	}

	return v;
}


v4l2_buffer CC::Device::capture_using_mmap_buffers( Bytes &bytes )
{
	if (buffer_details.empty())
	{
		request_memory_mapped_buffers();
	}

	stream_start();

	uint32_t idx = 0;

	for (size_t retry = retry_count; retry > 0; retry --)
	{
		// VIDIOC_DQBUF docs say:  "By default VIDIOC_DQBUF blocks when no buffer is in the outgoing queue."
		v4l2_buffer buffer;
		ZERO( buffer );
		buffer.type		= V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buffer.field	= selected_format.fmt.pix.field;
		buffer.memory	= V4L2_MEMORY_MMAP;

		xioctl( VIDIOC_DQBUF, &buffer, "VIDIOC_DQBUF" );

		// is this a partial transfer?  should we be trying again?
		if (buffer.bytesused < buffer.length && retry > 1)
		{
			// we have a partial buffer, and at least 1 more retry attempt left, so forget this buffer and try again
			xioctl( VIDIOC_QBUF, &buffer, "VIDIOC_QBUF" );
			continue;
		}

		// if we get here we either have a full buffer, or we've used up our retry attempts

		// figure out the address where the camera stored the picture
		idx = buffer.index;
		BufferDetail &bd = buffer_details.at( idx );
		const uint8_t *ptr = reinterpret_cast<const uint8_t *>( bd.address );

		// make a copy of the buffer information as well as the image that was captured
		bd.buffer = buffer;
		bytes.assign( ptr, ptr + buffer.bytesused );

		// re-queue the buffer so the camera can start filling it up again
		xioctl( VIDIOC_QBUF, &buffer, "VIDIOC_QBUF" );

		/* if the buffer we received wasn't "full" (should something have gone wrong) then we need to add some
		 * bytes at the end of the vector, otherwise we could end up dereferencing invalid memory addresses when
		 * the vector is processed
		 */
		if (bd.buffer.bytesused != bd.buffer.length)
		{
			bytes.resize( bd.buffer.length, '\xFF' );
		}

		break;
	}

	/// @returns Returns a V4L2 structure with meta information on the captured image.
	return buffer_details[idx].buffer;
}


v4l2_buffer CC::Device::capture_using_read( Bytes &bytes )
{
	release_memory_mapped_buffers();

	// emulate a v4l2_buffer reply to be similar to the mmap read method
	v4l2_buffer buffer;
	ZERO(buffer);
	buffer.type		= selected_format.type;
	buffer.field	= selected_format.fmt.pix.field;
	buffer.length	= selected_format.fmt.pix.sizeimage;

	bytes.clear();
	bytes.resize( buffer.length, '\0' );

	for (size_t retry = retry_count; retry > 0; retry --)
	{
		const int error_no = xread( bytes );
		if (error_no == 0 && bytes.size() > 0)
		{
			break;
		}
	}

	buffer.bytesused = bytes.size();

	return buffer;
}


CC::Device &CC::Device::retry_incomplete_captures( const size_t retries )
{
	if (is_not_initialized())
	{
		initialize();
	}

	retry_count = std::min( (size_t)99, std::max( (size_t)1, retries ) );

	return *this;
}
