/* CamCap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: CamCap_init.cpp 2155 2017-02-14 07:45:55Z stephane $
 */

/** @file
 * Methods related to initializing camera devices.
 */

#include "CamCap.hpp"
#include "CamCap_defines.h"
#include <dirent.h>


CC::Device::Device( const std::string &fn ) :
	fname( fn ),
	fd( -1 ),
	initialized( false ),
	v4l2_message_ptr( nullptr ),
	v4l2_message_size( 0 ),
	retry_count( 5 ),
	use_libv4l( true ),
	transfer_method( CC::ETransferMethod::kMmap )
{
	set_v4l2_msg_handling();

	reset();

	return;
}


CC::Device::~Device( void )
{
	reset();
	set_v4l2_msg_handling( EV4L2_msg_handling::kStdErr );

	return;
}


CC::Device &CC::Device::reset( void )
{
	if (is_initialized())
	{
		try
		{
			release_memory_mapped_buffers();
		}
		catch (...)
		{
			// do nothing -- ignore whatever error was caught and continue to "reset" the class
		}
	}

	xclose();

	// do not reset "fname" or v4l2 error message handling in case we want to re-initialize the device

	transfer_method			= CC::ETransferMethod::kMmap;
	initialized				= false;
	streaming_enabled		= false;
	use_libv4l				= true;
	retry_count				= 5;
	ZERO( capabilities		);
	ZERO( selected_format	);
	inputs		.clear();
	formats		.clear();

	return *this;
}


CC::Device &CC::Device::initialize( const bool use_libv4l_interface )
{
	reset();

	// calling reset() wipes out the value, so set it again
	use_libv4l = use_libv4l_interface;

	if (fname.empty())
	{
		// create a filter function that will tell us if a directory entry contains the word "video", such as /dev/video0
		const auto filter = [](const struct dirent * entry)
		{
			if (std::string(entry->d_name).find("video") == std::string::npos)
			{
				return 0;
			}
			return 1;
		};

		const std::string base_path = "/dev/";
		dirent **namelist = nullptr;

		const int result = scandir( base_path.c_str(), &namelist, filter, alphasort );
		for ( int idx = 0; idx < result; idx ++)
		{
			if ( is_initialized() == false )
			{
				try
				{
					initialize( base_path + namelist[idx]->d_name, use_libv4l_interface );
				}
				catch (...)
				{
					// do nothing, we'll try again if there is another device
				}
			}
			free( namelist[idx] );
		}
		free( namelist );
	}
	else
	{
		// we've been given a very specific name to use
		initialize(fname, use_libv4l_interface);
	}

	if (is_initialized() == false)
	{
		/// @throw std::runtime_error if a camera device cannot be initialized
		throw std::runtime_error( "failed to initialize camera device" );
	}

	return *this;
}


CC::Device &CC::Device::initialize( const std::string &fn, const bool use_libv4l_interface )
{
	reset();

	// calling reset() wipes out the value, so set it again
	use_libv4l = use_libv4l_interface;

	if (fn.empty())
	{
		/// @throw std::invalid_argument if the camera device name is empty
		throw std::invalid_argument( "camera device name cannot be empty" );
	}

	xopen( fn );
	fname		= fn;
	initialized	= true;

	xioctl( VIDIOC_QUERYCAP, &capabilities, "VIDIOC_QUERYCAP" );

	for (uint32_t idx = 0; true; idx ++)
	{
		v4l2_input input;
		ZERO( input );
		input.index = idx;

		const int rc = xioctl( VIDIOC_ENUMINPUT, &input );
		if (rc)
		{
			// no more inputs to enumerate
			break;
		}

		// remember this input
		inputs[idx] = input;
	}

	// automatically select the very first input index
	set_input_index();

	// get all of the image formats
	// https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-enum-fmt.html
	for (uint32_t idx = 0; true; idx ++)
	{
		v4l2_fmtdesc fmtdesc;
		ZERO(fmtdesc);
		fmtdesc.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		fmtdesc.index = idx;
		const int rc = xioctl( VIDIOC_ENUM_FMT, &fmtdesc );
		if (rc)	break;

		formats[idx] = fmtdesc;
	}

	if (formats.empty())
	{
		/// @throw std::runtime_error if enumerating the supported image formats fails to return anything
		throw std::runtime_error( "device " + fname + " has not returned anything when asked to list the image formats it supports" );
	}

	// assume the first format is a useful one, and use it to set the image format
	set_format( formats[0] );

	return *this;
}


CC::MU32Str CC::Device::get_capability_flags( void ) const
{
	if (is_not_initialized())
	{
		/// @throw std::runtime_error if the device is unitialized.
		throw std::runtime_error( "uninitialized device does not have capabilities" );
	}

	// flags defined at https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-querycap.html
	const MU32Str full_map =
	{
		{ V4L2_CAP_VIDEO_CAPTURE		, "The device supports the single-planar API through the Video Capture interface."	},
		{ V4L2_CAP_VIDEO_CAPTURE_MPLANE	, "The device supports the multi-planar API through the Video Capture interface."	},
		{ V4L2_CAP_VIDEO_OUTPUT			, "The device supports the single-planar API through the Video Output interface."	},
		{ V4L2_CAP_VIDEO_OUTPUT_MPLANE	, "The device supports the multi-planar API through the Video Output interface."	},
		{ V4L2_CAP_VIDEO_M2M			, "The device supports the single-planar API through the Video Memory-To-Memory interface."	},
		{ V4L2_CAP_VIDEO_M2M_MPLANE		, "The device supports the multi-planar API through the Video Memory-To-Memory interface."	},
		{ V4L2_CAP_VIDEO_OVERLAY		, "The device supports the Video Overlay interface." },
		{ V4L2_CAP_VBI_CAPTURE			, "The device supports the Raw VBI Capture interface, providing Teletext and Closed Caption data." },
		{ V4L2_CAP_VBI_OUTPUT			, "The device supports the Raw VBI Output interface." },
		{ V4L2_CAP_SLICED_VBI_CAPTURE	, "The device supports the Sliced VBI Capture interface." },
		{ V4L2_CAP_SLICED_VBI_OUTPUT	, "The device supports the Sliced VBI Output interface." },
		{ V4L2_CAP_RDS_CAPTURE			, "The device supports the RDS capture interface." },
		{ V4L2_CAP_VIDEO_OUTPUT_OVERLAY	, "The device supports the Video Output Overlay (OSD) interface." },
		{ V4L2_CAP_HW_FREQ_SEEK			, "The device supports the ioctl VIDIOC_S_HW_FREQ_SEEK ioctl for hardware frequency seeking." },
		{ V4L2_CAP_RDS_OUTPUT			, "The device supports the RDS output interface." },
		{ V4L2_CAP_TUNER				, "The device has some sort of tuner to receive RF-modulated video signals." },
		{ V4L2_CAP_AUDIO				, "The device has audio inputs or outputs." },
		{ V4L2_CAP_RADIO				, "This is a radio receiver." },
		{ V4L2_CAP_MODULATOR			, "The device has some sort of modulator to emit RF-modulated video/audio signals." },
		{ V4L2_CAP_SDR_CAPTURE			, "The device supports the SDR Capture interface." },
		{ V4L2_CAP_EXT_PIX_FORMAT		, "The device supports the struct v4l2_pix_format extended fields." },
		{ V4L2_CAP_SDR_OUTPUT			, "The device supports the SDR Output interface." },
		{ V4L2_CAP_READWRITE			, "The device supports the read() and/or write() I/O methods." },
		{ V4L2_CAP_ASYNCIO				, "The device supports the asynchronous I/O methods." },
		{ V4L2_CAP_STREAMING			, "The device supports the streaming I/O method." },
		{ V4L2_CAP_TOUCH				, "This is a touch device." },
		{ V4L2_CAP_DEVICE_CAPS			, "The driver fills the device_caps field." }
	};

	// from the full map of possible capabilities, see which ones this device advertises
	MU32Str m;
	for ( const auto iter : full_map )
	{
		if (capabilities.capabilities & iter.first)
		{
			m[iter.first] = iter.second;
		}
	}

	return m;
}
