/* CamCap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: CamCap_v4l2.cpp 2161 2017-02-19 18:46:24Z stephane $
 */

/** @file
 * Wrappers for some libv4l2 functions, such as @p v4l2_open(), @p v4l2_close(), and @p v4l2_ioctl().
 */

#include "CamCap.hpp"
#include <cstring>
#include <libv4l2.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>


CC::Device &CC::Device::set_v4l2_msg_handling( const EV4L2_msg_handling type )
{
	if (v4l2_log_file)
	{
		fclose( v4l2_log_file );
		v4l2_log_file = nullptr;

		if (v4l2_message_ptr)
		{
			free( v4l2_message_ptr );
			v4l2_message_ptr	= nullptr;
			v4l2_message_size	= 0;
		}
	}

	switch( type )
	{
		case EV4L2_msg_handling::kStdErr:
		{
			v4l2_log_file = nullptr;
			break;
		}
		case EV4L2_msg_handling::kDiscard:
		{
			v4l2_log_file = std::fopen( "/dev/null", "rw+" );
			break;
		}
		case EV4L2_msg_handling::kKeep:
		{
			v4l2_log_file = open_memstream( &v4l2_message_ptr, &v4l2_message_size );
			break;
		}
	}

	return *this;
}


std::string CC::Device::get_v4l2_messages( void )
{
	std::string msg;

	if (v4l2_log_file)
	{
		std::fflush( v4l2_log_file );
		if (v4l2_message_ptr && v4l2_message_size)
		{
			msg = std::string( v4l2_message_ptr, v4l2_message_size );
			std::fseek( v4l2_log_file, 0, SEEK_SET );
		}
	}

	return msg;
}


CC::Device &CC::Device::xioctl( const int request, void *parm, const std::string &name )
{
	/**
	 * @param [in] request V4L2 ioctl request, such as @p VIDIOC_QUERYCAP.
	 * @param [in] parm Pointer to V4L2 structure to pass as the 3rd parameter to @p ioctl().
	 * @param [in] name Text name corresponding to the ioctl request.  Will be used to generate an error message if
	 * the call to @p ioctl() fails.
	 */

	if (fd < 0)
	{
		/// @throw std::runtime_error if the file descriptor is invalid (device not initialized?)
		throw std::runtime_error( "cannot perform ioctl (" + name + ") on an invalid file descriptor" );
	}

	if (is_not_initialized())
	{
		/// @throw std::runtime_error if the camera device has not yet been fully initialized
		throw std::runtime_error( "cannot perform ioctl (" + name + ") on an uninitialized camera device" );
	}

	const int rc = xioctl( request, parm );
	if (rc)
	{
		const int error_no = errno;
		const std::string msg = strerror(error_no);

		/// @throw std::runtime_error if the requested ioctl failed
		throw std::runtime_error( "ioctl request for " + name + " on device " + fname + " failed with rc=" + std::to_string(rc) + " (errno " + std::to_string(error_no) + ": " + msg + ")" );
	}

	/// @return only if the ioctl call successed, otherwise throws an exception.
	return *this;
}


int CC::Device::xioctl( const int request, void *parm )
{
	/// @return Returns the result of calling v4l2_ioctl().  This version of @p xioctl() does not throw.

	errno = 0;

	if (fd < 0)
	{
		return -1;
	}

	errno	= 0;
	int rc	= 0;
	do
	{
		if (use_libv4l)
		{
			rc = v4l2_ioctl( fd, request, parm );
		}
		else
		{
			rc = ioctl( fd, request, parm );
		}
	}
	while ( rc == -1 && errno == EINTR);

	return rc;
}


CC::Device &CC::Device::xopen( const std::string &fn )
{
	xclose();

	errno = 0;
	if (use_libv4l)
	{
		fd = v4l2_open( fn.c_str(), O_RDWR );
	}
	else
	{
		fd = open( fn.c_str(), O_RDWR );
	}

	if (fd < 0)
	{
		const int error_no = errno;
		const std::string msg = strerror(error_no);

		/// @throw std::runtime_error if the device named fails to open
		throw std::runtime_error( "camera device " + fn + " cannot be opened (errno " + std::to_string(error_no) + ": " + msg + ")" );
	}

	return *this;
}


CC::Device &CC::Device::xclose( void )
{
	if (fd >= 0)
	{
		if (use_libv4l)
		{
			v4l2_close(fd);
		}
		else
		{
			close(fd);
		}
		fd = -1;
	}

	return *this;
}


void *CC::Device::xmmap( size_t length, off_t offset)
{
	void * address = nullptr;

	errno = 0;
	if (use_libv4l)
	{
		address = v4l2_mmap(
						nullptr, length,
						PROT_READ | PROT_WRITE,
						MAP_SHARED,
						fd,
						offset );
	}
	else
	{
		address = mmap(
						nullptr, length,
						PROT_READ | PROT_WRITE,
						MAP_SHARED,
						fd,
						offset );
	}

	if (address == MAP_FAILED)
	{
		const int error_no = errno;
		const std::string msg = strerror(error_no);

		/// @throw std::runtime_error if @p mmap() fails to return an address to use for a buffer
		throw std::runtime_error( "failed to mmap a " + std::to_string(length) + "-byte block (errno " + std::to_string(error_no) + ": " + msg + ")" );
	}

	return address;
}


CC::Device &CC::Device::xmunmap( void *addr, size_t length )
{
	if (use_libv4l)
	{
		v4l2_munmap( addr, length );
	}
	else
	{
		munmap( addr, length );
	}

	return *this;
}


int CC::Device::xread( Bytes &bytes )
{
	ssize_t count	= 0;
	int error_no	= 0;

	while (true)
	{
		errno = 0;
		if (use_libv4l)
		{
			count = v4l2_read( fd, bytes.data(), bytes.size() );
		}
		else
		{
			count = read( fd, bytes.data(), bytes.size() );
		}
		error_no = errno;

		if (count <= 0 && error_no == EAGAIN)
		{
			continue;
		}

		if ((size_t)count != bytes.size())
		{
			// truncate the bytes vector so it contains the exact number of bytes

			if (count <= 0)
			{
				bytes.clear();
			}
			else
			{
				bytes.resize( count );
			}
		}

		break;
	}

	if (count < 0)
	{
		const std::string msg = strerror(error_no);

		/// @throw std::runtime_error if the read request failed
		throw std::runtime_error( "read request on device " + fname + " failed (errno " + std::to_string(error_no) + ": " + msg + ")" );
	}

	return error_no;
}
