/* CamCap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: TestUtil.cpp 2785 2019-07-18 21:27:17Z stephane $
 */


#include <gtest/gtest.h>
#include "CamCap.hpp"


TEST(CamCap, GCD)
{
	ASSERT_TRUE(CC::gcd(0	, 0		) == 0	);
	ASSERT_TRUE(CC::gcd(12	, 4		) == 4	);
	ASSERT_TRUE(CC::gcd(640	, 480	) == 160);
	ASSERT_TRUE(CC::gcd(1024, 768	) == 256);
}


TEST(CamCap, Version)
{
	// make sure we have some type of non-empty version string
	const std::string ver = CC::get_version();

	ASSERT_FALSE(ver.empty());

	// version string is normally at least 10 characters, for example:  "0.0.1-2784"
	ASSERT_GE(ver.size(), 10);

	// version string should contain at least 1 digit somewhere in it
	const auto pos = ver.find_first_of("0123456789");
	ASSERT_TRUE(pos != std::string::npos);
}


TEST(CamCap, Constructor)
{
	// constructor should not immediately initialize the camera
	CC::Device camera;
	ASSERT_FALSE(camera.is_initialized());

	// unlikely that someone has 1234 cameras connected to their computer
	ASSERT_THROW(camera.initialize("/dev/video1234"), std::exception);
	ASSERT_FALSE(camera.is_initialized());

	// device name is not stored until initialize is successful
	ASSERT_STREQ(camera.get_name().c_str(), "");
}


TEST(CamCap, Description)
{
	CC::Device camera;
	ASSERT_FALSE(camera.is_initialized());
	const std::string str = camera.describe();

	/* At this point the camera is not initialized so we wont get much, but there should be a few lines of text:
	 *
	 * CamCap v0.0.1-2784M
	 * Filename:		""
	 * Initialized:		false
	 */
	std::cout << str;
	for (const std::string & txt : {"CamCap", "Filename", "Initialized"} )
	{
		std::cout << "finding \"" << txt << "\"..." << std::endl;
		ASSERT_TRUE(str.find(txt) != std::string::npos);
	}
}
