/* GPC (C) 2017-2018 Stephane Charette <stephanecharette@gmail.com>
 * $Id: PrintJob.cpp 2478 2018-03-12 22:40:08Z stephane $
 */

#include "GPC.hpp"


PrintJob::PrintJob(void)
{
	return;
}


PrintJob::~PrintJob(void)
{
	return;
}


PrintJob &PrintJob::load(File file)
{
	try
	{
		load_ijs(file);

		// for each of the images in the IJS, we need to create a new IJB
		for (size_t idx = 0; idx < ijs.bitmaps.size(); idx++)
		{
			load_ijb(idx);
		}
	}
	catch (const Lox::Exception &e)
	{
		LOG_MSG("failed to load either the .ijs or one of the referenced .ijb file: " << file.getFullPathName().toStdString() << std::endl << e.what());
		throw;
	}

	return *this;
}


PrintJob &PrintJob::load_ijs(File file)
{
	if (file.existsAsFile() == false)
	{
		/// @throw Lox::Exception if the file doesn't exist
		throw Lox::Exception(LOX_WHERE, "Failed to open the print job: " + file.getFullPathName().toStdString());
	}

	if (file.hasFileExtension(".ijs") == false)
	{
		/// @throw Lox::Exception if the file doesn't has an extension of .ijs
		throw Lox::Exception(LOX_WHERE, "File does not seem to be .ijs: " + file.getFullPathName().toStdString());
	}

	root_directory = file.getParentDirectory();

	try
	{
		ijs.load(file);
		LOG_MSG(ijs.summary());

		const auto v = ijs.warnings();
		if (v.empty() == false)
		{
			std::string msg = "The following potential problems were noted while loading the .ijs file:\r\n\r\n";
			for (const std::string &warning : v)
			{
				msg += "- " + warning + "\r\n";
			}
			msg += "\r\nDo you still want to try and continue to load this print job?";

			const bool result = AlertWindow::showOkCancelBox(AlertWindow::AlertIconType::QuestionIcon, file.getFileName(), msg, "continue", "abort");
			if (result == false)
			{
				/// @throw Lox::Exception if the user chooses to abort loading the .ijs file.
				throw Lox::Exception(LOX_WHERE, "User aborted IJS load due to errors: " + file.getFullPathName().toStdString());
			}
		}
	}
	catch (const Lox::Exception &e)
	{
		LOG_MSG("failed to load either the .ijs or one of the referenced .ijb file: " << file.getFullPathName().toStdString() << std::endl << e.what());
		throw;
	}

	return *this;
}


PrintJob &PrintJob::load_ijb(const size_t idx)
{
	if (idx >= ijs.bitmaps.size())
	{
		/// @throw Lox::Exception if the bitmap index is past the end of the array
		throw Lox::Exception(LOX_WHERE, "Bitmap index " + std::to_string(idx) + " is greater than the number of bitmaps.");
	}

	const auto *bitmap = ijs.bitmaps[idx];

	ijbs.emplace_back();
	IJB & ijb = ijbs.back();

	ijb.load(root_directory.getChildFile(bitmap->long_filename).withFileExtension(".ijb"));
	LOG_MSG(ijb.summary());

	ijb.create_browseable_image();

	return *this;
}


PrintJob &PrintJob::copy_to_new_directory(File new_dir)
{
	new_dir.createDirectory();

	File target = new_dir.getChildFile(ijs.file.getFileName());
	bool result = ijs.file.copyFileTo(target);

	for (const auto & ijb : ijbs)
	{
		if (result)
		{
			target = new_dir.getChildFile(ijb.file.getFileName());
			result = ijb.file.copyFileTo(target);
		}
	}

	if (result == false)
	{
		/// @throw Lox::Exception if any of the .ijs or .ijb files fail to copy to the new destination
		throw Lox::Exception(LOX_WHERE, "Failed to copy the .ijs or one of the .ijb files to " + new_dir.getFullPathName().toStdString() + ".");
	}

	root_directory = new_dir;

	return *this;
}
