/* GPC (C) 2017-2018 Stephane Charette <stephanecharette@gmail.com>
 * $Id: IJS.hpp 2484 2018-03-17 09:01:40Z stephane $
 */

#pragma once

#include "GPC.hpp"


class IJS final
{
	public:

		#pragma pack(push, 1)

		/** A 127-byte (0x7f) record that appears at the start of every print job.
		 *
		 * @note For 2-byte and 4-byte values, remember to convert endianness,
		 * as the @p IJS files seem to be big-endian.  See @ref ByteOrder::swap().
		 *
		 * Example #1:
		 * ~~~~{.txt}
		 * 0x00:  00 00 00 0e 4a 6f 62 31  2e 6a 6f 62 00 00 00 00  |....Job1.job....|
		 * 0x10:  00 00 57 73 74 62 6e 6b  5f 50 6c 6e 72 31 5f 45  |..Wstbnk_Plnr1_E|
		 * 0x20:  6e 64 70 72 6e 74 72 32  5f 47 6f 72 5f 41 70 72  |ndprntr2_Gor_Apr|
		 * 0x30:  32 39 5f 31 35 2e 69 6a  6a 00 00 00 00 00 00 00  |29_15.ijj.......|
		 * 0x40:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x50:  00 00 00 00 00 00 00 00  00 00 22 04 00 01 28 e0  |.........."...(.|
		 * 0x60:  00 00 00 14 00 00 03 84  00 00 01 2c 00 00 00 00  |...........,....|
		 * 0x70:  00 00 00 00 00 00 00 00  00 00 00 0e fc 00 08
		 * ~~~~
		 *
		 * Example #2:
		 * ~~~~{.txt}
		 * 0x00:  00 00 00 0e 4a 6f 62 31  2e 6a 6f 62 00 00 00 00  |....Job1.job....|
		 * 0x10:  00 00 54 65 73 74 6a 6f  62 5f 44 65 63 37 61 5f  |..Testjob_Dec7a_|
		 * 0x20:  32 30 31 37 2e 69 6a 6a  00 00 00 00 00 00 00 00  |2017.ijj........|
		 * 0x30:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x40:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x50:  00 00 00 00 00 00 00 00  00 00 22 04 00 01 28 e0  |.........."...(.|
		 * 0x60:  00 00 00 14 00 00 03 84  00 00 01 2c 00 00 00 00  |...........,....|
		 * 0x70:  00 00 00 00 00 00 00 00  00 00 00 00 ae 00 01
		 * ~~~~
		 */
		struct RecJob
		{
			uint32_t	unknown1;			///< 0x00 - 0x03:  always 0x0000000e?
			char		short_name[14];		///< 0x04 - 0x11:  "Job1.job"; enough room to store full 8.3 + NULL
			char		long_filename[64];	///< 0x12 - 0x51:  "Wstbnk_Plnr1_Endprntr2_Gor_Apr29_15.ijj" ...length unknown, 64 is just a guess
			uint8_t		unknown2;			///< 0x52:  always zero?
			uint8_t		unknown3;			///< 0x53:  always zero?
			uint32_t	unknown4;			///< 0x54 - 0x57:  always zero?
			uint32_t	unknown5;			///< 0x58 - 0x5b:  always 0x00002204?  (8708)
			uint32_t	unknown6;			///< 0x5c - 0x5f:  0x128e0 (76000d), 0xcb20 (52000d), 0x24220 (148000d), 0x157c0 (88000)
			uint32_t	unknown7;			///< 0x60 - 0x63:  always 0x00000014?  (20d)
			uint32_t	unknown8;			///< 0x64 - 0x67:  0x0384 (900d), 0x0258 (600d), 0x0708 (1800d), 0x041a (1050d)
			uint32_t	unknown9;			///< 0x68 - 0x6b:  always 0x0000012c?  (300d)
			uint32_t	unknown10;			///< 0x6c - 0x6f:  always zero?
			uint32_t	unknown11;			///< 0x70 - 0x73:  always zero?
			uint32_t	unknown12;			///< 0x74 - 0x77:  always zero?
			uint8_t		unknown13;			///< 0x78:  always zero?
			uint32_t	unknown14;			///< 0x79 - 0x7c:  0xae (174d), 0x0efc (3836d), 0x076e (1902d), 0x1380 (4992d), 0x0bda (3034d), 0x106b (4203d), 0x11c7 (4551d)
			uint16_t	unknown15;			///< 0x7d - 0x7e:  ?
		};
		static_assert(sizeof(RecJob) == 127, "RecJob is the wrong length");

		/** A 25-byte (0x19) record that appears before each bitmap group.
		 *
		 * @note For 2-byte and 4-byte values, remember to convert endianness,
		 * as the @p IJS files seem to be big-endian.  See @ref ByteOrder::swap().
		 *
		 * Example #1:
		 * ~~~~{.txt}
		 * 0x00:  00 00 02 e4 01 00 00 00  a1 00 00 00 00 00 00 02
		 * 0x10:  46 00 00 00 96 00 00 00  05
		 * ~~~~
		 *
		 * Example #2:
		 * ~~~~{.txt}
		 * 0x00:  00 00 01 c6 01 00 00 00  9e 00 00 00 96 00 00 02
		 * 0x10:  49 00 00 00 96 00 00 00  03
		 * ~~~~
		 *
		 * Example #3:
		 * ~~~~{.txt}
		 * 0x00:  00 00 01 c6 01 00 00 00  9e 00 00 01 29 00 00 02
		 * 0x10:  49 00 00 00 99 00 00 00  03
		 * ~~~~
		 *
		 * Example #4:
		 * ~~~~{.txt}
		 * 0x00:  00 00 00 A8 01 00 00 00  00 00 00 00 00 00 00 02
		 * 0x10:  44 00 00 00 96 00 00 00  01
		 * ~~~~
		 *
		 * @warning Beware, we've seen some @p IJS files where the groups
		 * appear out-of-order.  Meaning we have group #2, 3, 4, and then #1.
		 * Since the group id appears in the bitmap record and not in the
		 * group record, the only way we can determine the group is by
		 * peeking ahead at the image that comes immediately after a group.
		 * Important:  This also means the vectors of structures in the @p IJS
		 * file are not going to be ordered correctly!
		 */
		struct RecGroup
		{
			uint32_t	unknown1;			///< 0x00 - 0x03:  0xa8 (167d), 0x2e4 (740d), 0xbd4 (3028d), 0xc63 (3171d), 0x1c6 (454d)
			uint8_t		unknown2;			///< 0x04:  some kind of flag, always 1?
			uint32_t	horizontal_offset;	///< 0x05 - 0x08:  small number, such as 0x00, 0xa1 (161d), 0x9c (156d), 0xc0 (192d)
			uint32_t	vertical_offset;	///< 0x09 - 0x0c:  small number, such as 0x00, 0x1be (446d), 0x96 (150d)
			uint32_t	unknown3;			///< 0x0d - 0x10:  small number, such as 0x244 (580d), 0x246 (582d), 0x248 (584d), 0x252 (594d), 0x245 (581d), 0x249 (585d)
			uint32_t	unknown4;			///< 0x11 - 0x14:  small number, such as 0x96 (150d), 0x93 (147d), 0x90 (144d)
			uint32_t	number_of_images;	///< 0x15 - 0x18:  number of images to expect in this group; remember to use @ref ByteOrder::swap()!
		};
		static_assert(sizeof(RecGroup) == 25, "RecGroup is the wrong length");

		/** The bitmap records are exactly 143 bytes (0x8f) in length.
		 *
		 * @note For 2-byte and 4-byte values, remember to convert endianness,
		 * as the @p IJS files seem to be big-endian.  See @ref ByteOrder::swap().
		 *
		 * Example #1:
		 * ~~~~{.txt}
		 * 0x00:  4a 30 31 5f 30 31 30 2e  62 6d 70 00 00 00 4a 30  |J01_010.bmp...J0|
		 * 0x10:  31 5f 30 31 30 5f 7b 44  33 31 34 31 44 43 39 2d  |1_010_{D3141DC9-|
		 * 0x20:  30 37 31 34 2d 34 32 46  31 2d 39 45 43 41 2d 34  |0714-42F1-9ECA-4|
		 * 0x30:  44 31 43 37 44 33 37 46  36 44 43 7d 00 00 00 00  |D1C7D37F6DC}....|
		 * 0x40:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 03 30  |...............0|
		 * 0x50:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x60:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x70:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x80:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00     |............... |
		 * ~~~~
		 *
		 * Example #2:
		 * ~~~~{.txt}
		 * 0x00:  4a 30 31 5f 30 31 34 2e  62 6d 70 00 00 00 4a 30  |J01_014.bmp...J0|
		 * 0x10:  31 5f 30 31 34 5f 7b 46  42 46 46 46 38 44 33 2d  |1_014_{FBFFF8D3-|
		 * 0x20:  39 32 42 46 2d 34 31 35  44 2d 39 36 31 36 2d 37  |92BF-415D-9616-7|
		 * 0x30:  31 38 41 34 35 33 36 38  35 42 36 7d 00 00 00 00  |18A453685B6}....|
		 * 0x40:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 04 31  |...............1|
		 * 0x50:  36 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |6...............|
		 * 0x60:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x70:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x80:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00     |............... |
		 * ~~~~
		 */
		struct RecBitmap
		{
			char	short_filename[12];	///< 0x00 - 0x0b:  "J01_000.bmp" + NULL; note the filenames are 7+3, not 8+3
			uint8_t	unknown1[2];		///< 0x0c - 0x0d:  seems to be NULL
			char	long_filename[47];	///< 0x0e - 0x3c:  "J01_010_{D3141DC9-0714-42F1-9ECA-4D1C7D37F6DC}" + NULL
			uint8_t	unknown2[17];		///< 0x3d - 0x4d:  seems to be NULL
			uint8_t	group_number;		///< 0x4e:  group number (1-based value)
			char	trigger_value[4];	///< 0x4f - 0x52:  trigger value (length is a guess; at least 2 bytes for certain)
			uint8_t	unknown3[60];		///< 0x53 - 0x8e:  seems to be NULL
		};
		static_assert(sizeof(RecBitmap) == 143, "RecBitmap is the wrong length");

		/** Font record is a total of 288 (0x120) bytes.
		 *
		 * @note For 2-byte and 4-byte values, remember to convert endianness,
		 * as the @p IJS files seem to be big-endian.  See @ref ByteOrder::swap().
		 *
		 * It starts with a 23-byte header, similar to but 2 bytes shorter than a @ref RecGroup:
		 * ~~~~{.txt}
		 * 0x00:  00 00 01 21 05 00 00 00  0e 00 00 02 0a 00 00 00
		 * 0x10:  84 00 00 00 33 02 00
		 * ~~~~
		 * Then the actual font file information, which is an additional 266 bytes in length:
		 * ~~~~{.txt}
		 * 0x0000:  46 6f 6e 74 5f 31 2e 66  6e 74 00 00 00 00 54 69  |Font_1.fnt....Ti|
		 * 0x0010:  6d 65 73 20 4e 65 77 20  52 6f 6d 61 6e 5f 31 30  |mes New Roman_10|
		 * 0x0020:  5f 57 52 31 30 30 5f 54  42 53 30 42 00 00 00 00  |_WR100_TBS0B....|
		 * 0x0030:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x0040:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x0050:  00 00 00 01 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x0060:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x0070:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x0080:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x0090:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x00A0:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x00B0:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x00C0:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x00D0:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x00E0:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x00F0:  00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00  |................|
		 * 0x0100:  00 00 00 00 00 00 01 01  00 07                    |..........      |
		 * ~~~~
		 */
		struct RecFont
		{
			uint8_t	unknown1[23];		///< some type of header, similar to @ref RecGroup to 2 bytes shorter
			char	short_filename[14];	///< 0x00 - 0x0d
			char	long_filename[64];	///< 0x0e - 0x4d
			uint8_t	unknown[188];		///< 0x4e - 0x109
		};
		static_assert(sizeof(RecFont) == 289, "RecFont is the wrong length");

		/** 10-byte footer at the bottom of the file.
		 *
		 * Example:
		 * ~~~~{.txt}
		 * 0x00:  00 00 00 05 00 00 00 00  05 00
		 * ~~~~
		 */
		struct RecFooter
		{
			uint8_t unknown[10];
		};
		static_assert(sizeof(RecFooter) == 10, "RecFooter is the wrong length");

		#pragma pack(pop)

		/// Empty constructor.  @see @ref load()
		IJS(void);

		/// Destructor.
		~IJS(void);

		/// Load the given .ijs file.
		IJS &load(const File &filename);

		/// Get any warning messages that should be shown to the user.
		VStr warnings(void) const;

		/// Summarize the .ijs as a text message for the user.
		std::string summary(void) const;

		/// The entire contents of the file is loaded into this vector.
		VBytes contents;

		/** Even though this is a vector, we only support a single @ref RecJob
		 * pointer per @p ijs file.
		 */
		std::vector<const RecJob*> jobs;

		/** The groups stored in this vector @b may be stored in a different
		 * order than what they appear in the @p ijs file.  When they're
		 * parsed, the groups are re-ordered so this vector contains the
		 * groups id #1, 2, 3, 4, ... while in the @p ijs file they sometimes
		 * appear out-of-order, such as #2, 3, 4, 1.
		 */
		std::vector<const RecGroup*>	groups;

		std::vector<const RecBitmap*>	bitmaps;

		std::vector<const RecFont*>		fonts;

		std::vector<const RecFooter*>	footers;

		/// .ijs file
		File file;

		/// Used to keep track of whether or not there were unrecognized bytes when the .IJS file was loaded.
		size_t number_of_unknown_bytes;
};
