/* GPC (C) 2017-2018 Stephane Charette <stephanecharette@gmail.com>
 * $Id: SessionRecord.hpp 2485 2018-03-17 23:22:13Z stephane $
 */

#pragma once

#include "GPC.hpp"


class SessionRecord final
{
	public:

		/// Destructor.
		~SessionRecord(void);

		/// Simple empty constructor sets everything to default values.
		SessionRecord(void);

		/** Create a session from a JSON object.  Used when loading the
		 * sessions from disk.  @see @ref load_all_sessions()
		 */
		SessionRecord(Json &json);

		/// Something that happens often, so provide an easy way to get to the uuid.
		std::string uuid(void) const
		{
			return field[EField::kUUID];
		}

		/// Get the directory in which this session is stored.
		File get_dir(void) const
		{
			return File(field[SessionRecord::EField::kDirectory].c_str());
		}

		/// Get the .ijs file for this print job session.
		File get_ijs_file(void) const
		{
			return get_dir().getChildFile(field[SessionRecord::EField::kIjsFilename].c_str());
		}

		/** Initialize a number of fields and values which are derived from
		 * other sources.  For example, the fields @ref kCreationTimestamp and
		 * @ref kCreationText are both derived from @ref crationEpoch.  This
		 * may be called multiple times without ill effects.
		 */
		SessionRecord &initialize(void);

		/// Epoch when the print job session was created.
		std::time_t creationEpoch;

		/// Epoch when the print job session was last loaded onto the IncJet device.
		std::time_t recentlyUsedEpoch;

		/// Various fields we need to track, most of which are also shown in the summary window.
		enum EField
		{
			kUnknown				= 0,
			kUUID					,	// "123-456-789"
			kCreationTimestamp		,	// "2017-12-18 19:21:30"
			kCreationText			,	// "3 days ago"
			kRecentlyUsedTimestamp	,	// "2017-12-18 19:21:30"
			kRecentlyUsedText		,	// "3 days ago"
			kPrinterName			,	// populated using "printer_number
			kIjsFilename			,	// "Wstbnk_Plnr1_Endprntr2_Gor_Apr29_15.ijs"; note this is only the filename, not the full path
			kNumberOfUses			,	// "1"
			kNumberOfImages			,	// "6"
			kUsername				,	// "Stephane@polarbear"
			kDescription			,	// "test files"
			kDirectory				,	// "C:\\...etc...\\uuid\\"
			kMax
		};

		/// Array of text strings that define this print job.
		std::string field[EField::kMax];

		/// Determines if this print job applies to printer #1 or printer #2.
		size_t printer_number;

		/** Each image can have a text string description.  The key is the
		 * short image name, and the value is the description to show the user.
		 */
		MStr image_descriptions;

		/** Flag set to indicate changes have been made to configuration.
		 * Don't set this directly; instead, call the function
		 * @ref scehdule_sessions_to_be_saved() which will start a timer.
		 */
		static bool configuration_needs_to_be_saved;

		/** Sets the flag indicating that configuration should be saved, and
		 * starts (or restarts) the timer in @ref GPCApp to ensure the session
		 * records are eventually saved to disk.
		 */
		static void schedule_sessions_to_be_saved(void);
};


/// Returns a reference to the given print session.  Will throw if the UUID doesn't exist.
SessionRecord &get_session_record(const std::string &uuid);

/// Save all the sessions to disk.  This creates the .json file.
void save_all_sessions(void);

/// Load all of the sessions from the .json file.
void load_all_sessions(void);

void test_save_image_as_png(const Image &img, const std::string &filename);
