/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_101020015_TemperatureSensor.hpp 1730 2016-03-30 07:38:58Z stephane $
 */

#pragma once

#include "sg_GroveI2C.hpp"


namespace SG
{
	class I2CADC;

	/** Temperature sensor.  This analog sensor requires a analog-to-digital converter, such as the I2CADC.
	 *
	 * There are several revisions of this sensor, each with different thermistors.  The correct value must be provided
	 * for the readings to be interpreted correctly.
	 *
	 * Revision	| Thermistor		| Nominal B Constant
	 * ---------|-------------------|-------------------
	 * v1.0		| TTC3A103*39H		| 3975
	 * v1.1		| NCP18WF104F03RC	| 4250
	 * v1.2		| NCP18WF104F03RC	| 4250
	 *
	 * Description | Image
	 * ------------|------
	 * I2C ADC with a temperature sensor.<br/>@p J1 connects to the BeagleBone Green, and @p J2 connects to the temperature sensor. | @image html sg_101020015_TemperatureSensor.jpg
	 *
	 * @see http://www.seeedstudio.com/wiki/Grove_-_Temperature_Sensor_V1.2
	 * @see http://www.seeedstudio.com/depot/Grove-Temperature-Sensor-p-774.html
	 */
	class TemperatureSensor : public GroveI2C
	{
		public:

			enum class EType
			{
				vInvalid	= 0,
				v10_3975	= 3975,	///< temperature sensor v1.0
				v11_4250	= 4250,	///< temperature sensor v1.1
				v12_4250	= 4250,	///< temperature sensor v1.2 (default)
				v12_4275	= 4275	///< temperature sensor v1.2 (according to Seeed's wiki example code)
			};

			/// Destructor.
			virtual ~TemperatureSensor( void );

			/// Constructor.  This analog twig must be connected to an I2CADC.
			TemperatureSensor( I2CADC &adc, const std::string &n="", const EType etype = EType::v12_4250 );

			/// Copy constructor.
			TemperatureSensor( const TemperatureSensor &rhs );

			/// Assignment.
			TemperatureSensor &operator=( const TemperatureSensor &rhs );

			/// Test object equality. @{
			bool operator==( const TemperatureSensor &rhs ) const;
			bool operator!=( const TemperatureSensor &rhs ) const { return ! operator==(rhs); }
			/// @}

			/// Get the current temperature. @{
			double get_celsius(void);
			double get_fahrenheit(void);
			/// @}

			I2CADC &i2cadc;
			EType	type;
	};
}
