/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_103020013_I2CADC.cpp 1730 2016-03-30 07:38:58Z stephane $
 */

#include "sg_103020013_I2CADC.hpp"
#include "sg++.hpp"
#include "sg_i2c-dev.h"

#include <sstream>
#include <iomanip>
#include <system_error>

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>


SG::I2CADC::~I2CADC( void )
{
	close( i2c_file_handle );
	i2c_file_handle = -1;

	return;
}


SG::I2CADC::I2CADC( const uint8_t addr, const std::string &n ) :
		GroveI2C( SG::EGroveType::kI2CADC, n ),
		i2c_address( addr ),
		i2c_file_handle( -1 )
{
	const std::string filename = "/dev/i2c-1";
	i2c_file_handle = open(filename.c_str(), O_RDWR);
	if (i2c_file_handle < 0)
	{
		/// @throw std::system_error if the I2C device cannot be opened.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, "SG++ failed to open " + filename + " for read/write" );
	}

	const int i = static_cast<int>(i2c_address);
	const int rc = ioctl( i2c_file_handle, I2C_SLAVE, i );
	if (rc < 0)
	{
		close( i2c_file_handle );

		std::stringstream ss;
		ss << std::hex << std::setfill('0') << std::setw(2) << i;

		/// @throw std::system_error if the I2C slave address cannot be set.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, "SG++ failed to set the I2C slave address to 0x" + ss.str() );
	}

	return;
}


bool SG::I2CADC::operator==( const SG::I2CADC &rhs ) const
{
	return GroveI2C::operator==(rhs) && i2c_address == rhs.i2c_address;
}


SG::I2CADC &SG::I2CADC::reset(void)
{
	/// The exact default values for each register are described in the ADC121C021 datasheet, pages 18-22.

//	write16	( ERegister::kConversionResult	, 0x00		); // result register (read-only, do not write!)
	write8	( ERegister::kAlertStatus		, 0x00		); // alert status register
	write8	( ERegister::kConfiguration		, 0x00		); // configuration register
	write16	( ERegister::kLowLimit			, 0x0000	); // alert limit register (under range)
	write16	( ERegister::kHighLimit			, 0x0fff	); // alert limit register (over range)
	write16	( ERegister::kHysteresis		, 0x0000	); // alert hysteresis register
	write16	( ERegister::kLowestConversion	, 0x0fff	); // lowest conversion register
	write16	( ERegister::kHighestConversion	, 0x0000	); // highest conversion register

	return *this;
}


SG::I2CADC &SG::I2CADC::enable_automatic_mode( const uint8_t value )
{
	/* The configuration register has 8 bits, of which D7-D5 define the cycle time (automatic mode):
	 *
	 *		[D7] cycle time MSB
	 *		[D6] cycle time
	 *		[D5] cycle time LSB
	 *		[D4] alert hold
	 *		[D3] alert flag
	 *		[D2] alert pin
	 *		[D1] reserved (must be zero)
	 *		[D0] polarity
	 */

	// only the first 3 bits are relevant when setting the cycle time
	const uint8_t cycle_time = (value & 0x03) << 5;

	uint8_t config = read8( ERegister::kConfiguration );
	config &= 0x1f;			// remove D7-D5
	config |= cycle_time;	// set the new D7-D5

	write8( ERegister::kConfiguration, config );

	return *this;
}


uint8_t SG::I2CADC::read8( const uint8_t register_address )
{
	errno = 0;
	const uint8_t r = i2c_smbus_read_byte_data( i2c_file_handle, register_address );
	if (r == 0xff && errno)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;
		
		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)register_address;
		
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, "SG++ failed to read 8 bytes from register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	return r;
}


uint16_t SG::I2CADC::read16( const uint8_t register_address )
{
	errno = 0;
	const uint16_t r = i2c_smbus_read_word_data( i2c_file_handle, register_address );
	if (r == 0xffff && errno)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;
		
		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)register_address;
		
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, "SG++ failed to read 16 bytes from register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	/// @note The high/low bytes in the 16-bit word are swapped after reading.
	const uint16_t hibyte = (r & 0xff00) >> 8;
	const uint16_t lobyte = (r & 0x00ff) << 8;
	const uint16_t result = hibyte | lobyte;

	return result;
}


SG::I2CADC &SG::I2CADC::write8( const uint8_t register_address, const uint8_t value )
{
	errno = 0;
	const int rc = i2c_smbus_write_byte_data( i2c_file_handle, register_address, value );
	if (rc)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;

		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)register_address;

		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, "SG++ failed to write 8 bytes to register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	return *this;
}


SG::I2CADC &SG::I2CADC::write16( const uint8_t register_address, const uint16_t value )
{
	/// @note The high/low bytes in the 16-bit word are swapped prior to writting.
	const uint16_t hibyte = (value & 0xff00) >> 8;
	const uint16_t lobyte = (value & 0x00ff) << 8;
	const uint16_t result = hibyte | lobyte;

	errno = 0;
	const int rc = i2c_smbus_write_word_data( i2c_file_handle, register_address, result );
	if (rc)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;
		
		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)register_address;
		
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, "SG++ failed to write 16 bytes to register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	return *this;
}
