/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_GroveI2C.hpp 1729 2016-03-30 07:04:48Z stephane $
 */

#pragma once

// Central include file for SeeedGrove++ is "sg++.hpp".

#include "sg_groves.hpp"
#include "sg_GroveBase.hpp"


namespace SG
{
	/** Base class for Groves that use the I2C (Inter-Integrated Circuit) interface.
	 * 
	 * The BeagleBone Green has two I2C buses.  They can be seen at @p /dev/i2c-0 and @p /dev/i2c-1.  The first one,
	 * @p i2c-0, is not exposed and isn't of interest to the SG++ library.  The second one, @p i2c-1 is where I2C Grove
	 * twigs are connected.
	 *
	 * When looking at the top of a BeagleBone Green with the network interface towards the top and the Grove interfaces
	 * at the bottom, @p i2c-1 is the left-hand-side Grove interface labelled @p J4.  On the board, you'll also see the
	 * labels @p SCL (serial clock line), @p SDA (serial data line), @p 3V3, and @p GND.
	 *
	 * Several parts of the BeagleBone Green are always connected to the I2C bus.  It is important to note the
	 * addresses, because they can conflict with the address of Seeed twigs, such as Seeed's I2C ADC twig which
	 * defaults to address @p 0x55.
	 *
	 * Bus/Device		| Address	| Name		| Description					| Also See
	 * -----------------|-----------|-----------|-------------------------------|---------
	 * @p /dev/i2c-0	| @p 0x24	| TPS65217	| PMU (power management unit)	| @p /sys/bus/i2c/devices/0-0024
	 * @p /dev/i2c-0	| @p 0x50	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/0-0050
	 * @p /dev/i2c-1	| @p 0x54	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/1-0054
	 * @p /dev/i2c-1	| @p 0x55	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/1-0055
	 * @p /dev/i2c-1	| @p 0x56	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/1-0056
	 * @p /dev/i2c-1	| @p 0x57	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/1-0057
	 *
	 * The other interface labelled @p J5 on the bottom right side is the UART Grove interface.
	 *
	 * @see @ref GroveUART
	 * @see @ref GroveGPIO
	 * @see https://www.kernel.org/doc/Documentation/i2c/dev-interface
	 */
	class GroveI2C : public GroveBase
	{
		public:

			/// Constructor.
			GroveI2C( const SG::EGroveType t=SG::EGroveType::kUnknown, const std::string &n="" );

			/// Destructor.
			virtual ~GroveI2C( void );

			virtual bool is_grove_uart	( void ) const { return false;	}
			virtual bool is_grove_i2c	( void ) const { return true;	}
			virtual bool is_grove_gpio	( void ) const { return false;	}
	};
}
