/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_groves.hpp 1731 2016-03-31 13:10:51Z stephane $
 */

#pragma once

// Central include file for SeeedGrove++ is "sg++.hpp".

#include <string>
#include <vector>
#include <map>
#include <set>


namespace SG
{
	/**
	 * Grove types ordered by Seeed SKU:
	 *
	 * Grove Type														| SKU		| Comments																	| URL																								| Wiki
	 * -----------------------------------------------------------------|-----------|---------------------------------------------------------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------|-----
	 * @ref SG::EGroveType::kSerialCameraKit							| 101020000 | 																			| http://www.seeedstudio.com/depot/Grove-Serial-Camera-Kit-p-1608.html								| http://www.seeedstudio.com/wiki/Grove_-_Serial_Camera
	 * @ref SG::EGroveType::kHCHOSensor									| 101020001 | 																			| http://www.seeedstudio.com/depot/Grove-HCHO-Sensor-p-1593.html									| http://www.seeedstudio.com/wiki/Grove_-_HCHO_Sensor
	 * @ref SG::EGroveType::kGasSensorO2								| 101020002 | 																			| http://www.seeedstudio.com/depot/Grove-Gas-SensorO2-p-1541.html									| http://www.seeedstudio.com/wiki/Grove_-_Gas_Sensor(O%E2%82%82)
	 * @ref SG::EGroveType::kButton										| 101020003 | See @ref SG::EGroveType::kButtonP.										| http://www.seeedstudio.com/depot/Grove-Button-p-766.html											| http://www.seeedstudio.com/wiki/Grove_-_Button
	 * @ref SG::EGroveType::kSwitchP									| 101020004 |																			| http://www.seeedstudio.com/depot/Grove-SwitchP-p-1252.html										| http://www.seeedstudio.com/wiki/Grove_-_Switch(P)
	 * @ref SG::EGroveType::kCollisionSensor							| 101020005 |																			| http://www.seeedstudio.com/depot/Grove-Collision-Sensor-p-1132.html								| http://www.seeedstudio.com/wiki/Grove_-_Collision_Sensor
	 * @ref SG::EGroveType::kGasSensorMQ3								| 101020006 |																			| http://www.seeedstudio.com/depot/Grove-Gas-SensorMQ3-p-1418.html									| http://www.seeedstudio.com/wiki/Grove_-_Gas_Sensor
	 * @ref SG::EGroveType::kMiniCamera									| 101020007 | Discontinued.  See @ref SG::EGroveType::kSerialCamera.					| http://www.seeedstudio.com/depot/Grove-Mini-Camera-p-1578.html									| http://www.seeedstudio.com/wiki/Grove_-_Mini_Camera
	 * @ref SG::EGroveType::kMoistureSensor								| 101020008 |																			| http://www.seeedstudio.com/depot/Grove-Moisture-Sensor-p-955.html									| http://www.seeedstudio.com/wiki/Grove_-_Moisture_Sensor
	 * @ref SG::EGroveType::kLineFinder									| 101020009 |																			| http://www.seeedstudio.com/depot/Grove-Line-Finder-p-825.html										| http://www.seeedstudio.com/wiki/Grove_-_Line_Finder
	 * @ref SG::EGroveType::kUltrasonicRanger							| 101020010 |																			| http://www.seeedstudio.com/depot/Grove-Ultrasonic-Ranger-p-960.html								| http://www.seeedstudio.com/wiki/Grove_-_Ultrasonic_Ranger
	 * @ref SG::EGroveType::kTemperatureAndHumiditySensor				| 101020011 | See @ref SG::EGroveType::kTemperatureAndHumidityHighAccuracyAndMini.		| http://www.seeedstudio.com/depot/Grove-TempHumi-Sensor-p-745.html									| http://www.seeedstudio.com/wiki/Grove-_Temperature_and_Humidity_Sensor
	 * @ref SG::EGroveType::kDustSensor									| 101020012 |																			| http://www.seeedstudio.com/depot/Grove-Dust-Sensor-p-1050.html									| http://www.seeedstudio.com/wiki/Grove_-_Dust_Sensor
	 * @ref SG::EGroveType::kRTC										| 101020013 |																			| http://www.seeedstudio.com/depot/Grove-RTC-p-758.html												| http://www.seeedstudio.com/wiki/Grove_-_RTC
	 * @ref SG::EGroveType::kAnalogLightSensor							| 101020014 | SKU #101020014 and #101020022 are nearly identical.						| http://www.seeedstudio.com/depot/Grove-Light-Sensor-p-746.html									| http://www.seeedstudio.com/wiki/Grove_-_Light_Sensor
	 * @ref SG::EGroveType::kTemperatureSensorV12						| 101020015 |																			| http://www.seeedstudio.com/depot/Grove-Temperature-Sensor-p-774.html								| http://www.seeedstudio.com/wiki/Grove_-_Temperature_Sensor_V1.2
	 * @ref SG::EGroveType::kInfraredReceiver							| 101020016 |																			| http://www.seeedstudio.com/depot/Grove-Infrared-Receiver-p-994.html								| http://www.seeedstudio.com/wiki/Grove_-_Infrared_Receiver
	 * @ref SG::EGroveType::kRotaryAngleSensor							| 101020017 |																			| http://www.seeedstudio.com/depot/Grove-Rotary-Angle-Sensor-p-770.html								| http://www.seeedstudio.com/wiki/Grove_-_Rotary_Angle_Sensor
	 * @ref SG::EGroveType::kWaterSensor								| 101020018 |																			| http://www.seeedstudio.com/depot/Grove-Water-Sensor-p-748.html									| http://www.seeedstudio.com/wiki/Grove_-_Water_Sensor
	 * @ref SG::EGroveType::kTemperatureAndHumiditySensorPro			| 101020019 |																			| http://www.seeedstudio.com/depot/Grove-TemperatureHumidity-Sensor-Pro-p-838.html					| http://www.seeedstudio.com/wiki/Grove_-_Temperature_and_Humidity_Sensor_Pro
	 * @ref SG::EGroveType::kPIRMotionSensor							| 101020020 |																			| http://www.seeedstudio.com/depot/Grove-PIR-Motion-Sensor-p-802.html								| http://www.seeedstudio.com/wiki/Grove_-_PIR_Motion_Sensor
	 * @ref SG::EGroveType::kAirQualitySensor							| 101020021 | Discontinued. See @ref SG::EGroveType::kAirQualitySensorV13.				| http://www.seeedstudio.com/depot/Grove-Air-quality-sensor-p-1065.html								| http://www.seeedstudio.com/wiki/Grove_-_Air_Quality_Sensor
	 * @ref SG::EGroveType::kAnalogLightSensorP							| 101020022 | SKU #101020014 and #101020022 are nearly identical.						| http://www.seeedstudio.com/depot/Grove-Light-SensorP-p-1253.html									| http://www.seeedstudio.com/wiki/Grove_-_Light_Sensor
	 * @ref SG::EGroveType::kSoundSensor								| 101020023 |																			| http://www.seeedstudio.com/depot/Grove-Sound-Sensor-p-752.html									| http://www.seeedstudio.com/wiki/Grove_-_Sound_Sensor
	 * @ref SG::EGroveType::kMegaShield									| 101020024 |																			| http://www.seeedstudio.com/depot/Grove-Mega-Shield-p-934.html										| http://www.seeedstudio.com/wiki/Grove_-_Mega_Shield
	 * @ref SG::EGroveType::kTiltSwitch									| 101020025 |																			| http://www.seeedstudio.com/depot/Grove-Tilt-Switch-p-771.html										| http://www.seeedstudio.com/wiki/Grove_-_Tilt_Switch
	 * @ref SG::EGroveType::kInfraredEmitter							| 101020026 |																			| http://www.seeedstudio.com/depot/Grove-Infrared-Emitter-p-993.html								| http://www.seeedstudio.com/wiki/Grove_-_Infrared_Emitter
	 * @ref SG::EGroveType::kElectricitySensor							| 101020027 |																			| http://www.seeedstudio.com/depot/Grove-Electricity-Sensor-p-777.html								| http://www.seeedstudio.com/wiki/Grove_-_Electricity_Sensor
	 * @ref SG::EGroveType::kThumbJoystick								| 101020028 |																			| http://www.seeedstudio.com/depot/Grove-Thumb-Joystick-p-935.html									| http://www.seeedstudio.com/wiki/Grove_-_Thumb_Joystick
	 * @ref SG::EGroveType::kInfraredReflectiveSensor					| 101020029 |																			| http://www.seeedstudio.com/depot/Grove-Infrared-Reflective-Sensor-p-1230.html						| http://www.seeedstudio.com/wiki/Grove_-_Infrared_Reflective_Sensor
	 * @ref SG::EGroveType::kDigitalLightSensor							| 101020030 |																			| http://www.seeedstudio.com/depot/Grove-Digital-Light-Sensor-p-1281.html							| http://www.seeedstudio.com/wiki/Grove_-_Digital_Light_Sensor
	 * @ref SG::EGroveType::kPiezoVibrationSensor						| 101020031 |																			| http://www.seeedstudio.com/depot/Grove-Piezo-Vibration-Sensor-p-1411.html							| http://www.seeedstudio.com/wiki/Grove-_Piezo_Vibration_Sensor
	 * @ref SG::EGroveType::kBarometerSensor							| 101020032 | Discontinued.  See @ref SG::EGroveType::kBarometerSensorBMP180.			| http://www.seeedstudio.com/depot/grove-barometer-sensor-p-1199.html								| http://www.seeedstudio.com/wiki/Grove_-_Barometer_Sensor
	 * @ref SG::EGroveType::kEarClipHeartRateSensor						| 101020033 |																			| http://www.seeedstudio.com/depot/Grove-Earclip-Heart-Rate-Sensor-p-1116.html						| http://www.seeedstudio.com/wiki/Grove_-_Ear-clip_Heart_Rate_Sensor
	 * @ref SG::EGroveType::k3AxisDigitalCompass						| 101020034 |																			| http://www.seeedstudio.com/depot/Grove-3Axis-Digital-Compass-p-759.html							| http://www.seeedstudio.com/wiki/Grove_-_3-Axis_Digital_Compass
	 * @ref SG::EGroveType::kProtoShield								| 101020035 |																			| http://www.seeedstudio.com/depot/Grove-Protoshield-p-772.html										| http://www.seeedstudio.com/wiki/Grove_-_Protoshield
	 * @ref SG::EGroveType::kSlidePotentiometer							| 101020036 |																			| http://www.seeedstudio.com/depot/Grove-Slide-Potentiometer-p-1196.html							| http://www.seeedstudio.com/wiki/Grove_-_Slide_Potentiometer
	 * @ref SG::EGroveType::kTouchSensor								| 101020037 |																			| http://www.seeedstudio.com/depot/Grove-Touch-Sensor-p-747.html									| http://www.seeedstudio.com/wiki/Grove_-_I2C_Touch_Sensor
	 * @ref SG::EGroveType::kMagneticSwitch								| 101020038 |																			| http://www.seeedstudio.com/depot/Grove-Magnetic-Switch-p-744.html									| http://www.seeedstudio.com/wiki/Grove_-_Magnetic_Switch
	 * @ref SG::EGroveType::k3AxisDigitalAccelerometer1_5g				| 101020039 |																			| http://www.seeedstudio.com/depot/Grove-3Axis-Digital-Accelerometer15g-p-765.html					| http://www.seeedstudio.com/wiki/Grove_-_3-Axis_Digital_Accelerometer(%C2%B11.5g)
	 * @ref SG::EGroveType::kIRDistanceInterrupt						| 101020040 |																			| http://www.seeedstudio.com/depot/Grove-IR-Distance-Interrupter-p-1278.html						| http://www.seeedstudio.com/wiki/Grove_-_IR_Distance_Interrupt
	 * @ref SG::EGroveType::kI2CColorSensor								| 101020041 |																			| http://www.seeedstudio.com/depot/Grove-I2C-Color-Sensor-p-854.html								| http://www.seeedstudio.com/wiki/Grove_-_I2C_Color_Sensor
	 * @ref SG::EGroveType::k80cmInfraredProximitySensor				| 101020042 |																			| http://www.seeedstudio.com/depot/Grove-80cm-Infrared-Proximity-Sensor-p-788.html					| http://www.seeedstudio.com/wiki/Grove_-_80cm_Infrared_Proximity_Sensor
	 * @ref SG::EGroveType::kUVSensor									| 101020043 |																			| http://www.seeedstudio.com/depot/Grove-UV-Sensor-p-1540.html										| http://www.seeedstudio.com/wiki/Grove_-_UV_Sensor
	 * @ref SG::EGroveType::kAlcoholSensor								| 101020044 |																			| http://www.seeedstudio.com/depot/Grove-Alcohol-Sensor-p-764.html									| http://www.seeedstudio.com/wiki/Grove_-_Alcohol_Sensor
	 * @ref SG::EGroveType::kGasSensorMQ9								| 101020045 |																			| http://www.seeedstudio.com/depot/Grove-Gas-SensorMQ9-p-1419.html									| http://www.seeedstudio.com/wiki/Grove_-_Gas_Sensor
	 * @ref SG::EGroveType::kHallSensor									| 101020046 |																			| http://www.seeedstudio.com/depot/Grove-Hall-Sensor-p-965.html										| http://www.seeedstudio.com/wiki/Grove_-_Hall_Sensor
	 * @ref SG::EGroveType::kI2CTouchSensor								| 101020047 |																			| http://www.seeedstudio.com/depot/Grove-I2C-Touch-Sensor-p-840.html								| http://www.seeedstudio.com/wiki/Grove_-_I2C_Touch_Sensor
	 * @ref SG::EGroveType::kRotaryAngleSensorP							| 101020048 |																			| http://www.seeedstudio.com/depot/Grove-Rotary-Angle-SensorP-p-1242.html							| http://www.seeedstudio.com/wiki/Grove_-_Rotary_Angle_Sensor
	 * @ref SG::EGroveType::kFlameSensor								| 101020049 |																			| http://www.seeedstudio.com/depot/Grove-Flame-Sensor-p-1450.html									| http://www.seeedstudio.com/wiki/Grove_-_Flame_Sensor
	 * @ref SG::EGroveType::k3AxisDigitalGyro							| 101020050 |																			| http://www.seeedstudio.com/depot/Grove-3Axis-Digital-Gyro-p-750.html								| http://www.seeedstudio.com/wiki/Grove_-_3-Axis_Digital_Gyro
	 * @ref SG::EGroveType::k3AxisAnalogAccelerometer					| 101020051 |																			| http://www.seeedstudio.com/depot/Grove-3Axis-Analog-Accelerometer-p-1086.html						| http://www.seeedstudio.com/wiki/Grove_-_3-Axis_Analog_Accelerometer
	 * @ref SG::EGroveType::kGSRSensor									| 101020052 |																			| http://www.seeedstudio.com/depot/Grove-GSR-sensor-p-1614.html										| http://www.seeedstudio.com/wiki/Grove_-_GSR_Sensor
	 * @ref SG::EGroveType::kSingleAxisAnalogGyro						| 101020053 |																			| http://www.seeedstudio.com/depot/Grove-Single-Axis-Analog-Gyro-p-1451.html						| http://www.seeedstudio.com/wiki/Grove_-_Single_Axis_Analog_Gyro
	 * @ref SG::EGroveType::k3AxisDigitalAccelerometer16g				| 101020054 |																			| http://www.seeedstudio.com/depot/Grove-3Axis-Digital-Accelerometer16g-p-1156.html					| http://www.seeedstudio.com/wiki/Grove_-_3-Axis_Digital_Accelerometer(%C2%B116g)
	 * @ref SG::EGroveType::kGasSensorMQ2								| 101020055 |																			| http://www.seeedstudio.com/depot/Grove-Gas-SensorMQ2-p-937.html									| http://www.seeedstudio.com/wiki/Grove_-_Gas_Sensor
	 * @ref SG::EGroveType::kGasSensorMQ5								| 101020056 |																			| http://www.seeedstudio.com/depot/Grove-Gas-SensorMQ5-p-938.html									| http://www.seeedstudio.com/wiki/Grove_-_Gas_Sensor
	 * @ref SG::EGroveType::kFingerprintSensor							| 101020057 |																			| http://www.seeedstudio.com/depot/Grove-Fingerprint-Sensor-p-1424.html								| http://www.seeedstudio.com/wiki/Grove_-_Fingerprint_Sensor
	 * @ref SG::EGroveType::kEMGDetector								| 101020058 |																			| http://www.seeedstudio.com/depot/Grove-EMG-Detector-p-1737.html									| http://www.seeedstudio.com/wiki/Grove_-_EMG_Detector
	 * @ref SG::EGroveType::kIMU9DOFV1									| 101020059 |																			| http://www.seeedstudio.com/depot/Grove-IMU-9DOF-p-1728.html										| http://www.seeedstudio.com/wiki/Grove_-_IMU_9DOF_v1.0
	 * @ref SG::EGroveType::kMotionSensorLargeLens						| 101020060 |																			| http://www.seeedstudio.com/depot/PIR-Motion-Sensor-Large-Lens-version-p-1976.html					| http://www.seeedstudio.com/wiki/PIR_Motion_sensor_module
	 * @ref SG::EGroveType::k6AxisAccelerometerAndCompassV1				| 101020061 | Discontinued. See @ref SG::EGroveType::k6AxisAccelerometerAndCompassV2.	| http://www.seeedstudio.com/depot/Grove-6Axis-AccelerometerCompass-p-1448.html						| http://www.seeedstudio.com/wiki/Grove_-_6-Axis_Accelerometer%26Compass_V2.0
	 * @ref SG::EGroveType::kAnalogInfraredTemperatureSensor			| 101020062 | See @ref SG::EGroveType::kDigitalInfraredTemperatureSensor.				| http://www.seeedstudio.com/depot/Grove-Infrared-Temperature-Sensor-p-1058.html					| http://www.seeedstudio.com/wiki/Grove_-_Infrared_temperature_sensor
	 * @ref SG::EGroveType::kLoudnessSensor								| 101020063 |																			| http://www.seeedstudio.com/depot/Grove-Loudness-Sensor-p-1382.html								| http://www.seeedstudio.com/wiki/Grove_-_Loudness_Sensor
	 * @ref SG::EGroveType::kSerialCamera								| 101020064 |																			| http://www.seeedstudio.com/depot/Grove-Serial-Camera-p-945.html									| http://www.seeedstudio.com/wiki/Grove_-_Serial_Camera
	 * @ref SG::EGroveType::kPHSensor									| 101020065 | Discontinued.																| http://www.seeedstudio.com/depot/Grove-PH-Sensor-p-1564.html										| http://www.seeedstudio.com/wiki/Grove_-_PH_Sensor
	 * @ref SG::EGroveType::kChestStrapHeartRateSensor					| 101020066 | Discontinued.																| http://www.seeedstudio.com/depot/Grove-Chest-Strap-Heart-Rate-Sensor-p-1115.html					| http://www.seeedstudio.com/wiki/Grove_-_Chest_Strap_Heart_Rate_Sensor
	 * @ref SG::EGroveType::kCO2Sensor									| 101020067 |																			| http://www.seeedstudio.com/depot/Grove-CO2-Sensor-p-1863.html										| http://www.seeedstudio.com/wiki/Grove_-_CO2_Sensor
	 * @ref SG::EGroveType::kBarometerHighAccuracy						| 101020068 | See @ref SG::EGroveType::kBarometerSensorBMP180.							| http://www.seeedstudio.com/depot/Grove-Barometer-HighAccuracy-p-1865.html							| http://www.seeedstudio.com/wiki/Grove_-_Barometer_(High-Accuracy)
	 * @ref SG::EGroveType::kQTouchSensor								| 101020069 |																			| http://www.seeedstudio.com/depot/GroveQ-Touch-Sensor-p-1854.html									| http://www.seeedstudio.com/wiki/Grove_-_Q_Touch_Sensor
	 * @ref SG::EGroveType::kNFCTag										| 101020070 |																			| http://www.seeedstudio.com/depot/Grove-NFC-Tag-p-1866.html										| http://www.seeedstudio.com/wiki/Grove_-_NFC_Tag
	 * @ref SG::EGroveType::k3AxisDigitalAccelerometer400g				| 101020071 |																			| http://www.seeedstudio.com/depot/Grove-3Axis-Digital-Accelerometer400g-p-1897.html				| http://www.seeedstudio.com/wiki/Grove_-_3-Axis_Digital_Accelerometer(%C2%B1400g)
	 * @ref SG::EGroveType::kBarometerSensorBMP180						| 101020072 | See @ref SG::EGroveType::kBarometerHighAccuracy.							| http://www.seeedstudio.com/depot/Grove-Barometer-Sensor-BMP180-p-1840.html						| http://www.seeedstudio.com/wiki/Grove_-_Barometer_Sensor_(BMP180)
	 * @ref SG::EGroveType::kElectromagnet								| 101020073 |																			| http://www.seeedstudio.com/depot/Grove-Electromagnet-p-1820.html									| http://www.seeedstudio.com/wiki/Grove_-_Electromagnet
	 * @ref SG::EGroveType::kTemperatureAndHumidityHighAccuracyAndMini	| 101020074 | See @ref SG::EGroveType::kTemperatureAndHumiditySensor.					| http://www.seeedstudio.com/depot/Grove-TemperatureHumidity-Sensor-HighAccuracy-Mini-p-1921.html	| http://www.seeedstudio.com/wiki/Grove_-_Tempture%26Humidity_Sensor_(High-Accuracy_%26Mini)_v1.0
	 * n/a																| 101020075 | n/a																		| n/a																								| n/a
	 * @ref SG::EGroveType::kLuminanceSensor							| 101020076 |																			| http://www.seeedstudio.com/depot/Grove-Luminance-Sensor-p-1941.html								| http://www.seeedstudio.com/wiki/Grove_-_Luminance_Sensor
	 * @ref SG::EGroveType::kDigitalInfraredTemperatureSensor			| 101020077 | See @ref SG::EGroveType::kAnalogInfraredTemperatureSensor.				| http://www.seeedstudio.com/depot/Grove-Digital-Infrared-Temperature-Sensor-p-2385.html			| http://www.seeedstudio.com/wiki/Grove_-_Digital_Infrared_Temperature_Sensor
	 * @ref SG::EGroveType::kAirQualitySensorV13						| 101020078 |																			| http://www.seeedstudio.com/depot/Grove-Air-quality-sensor-v13-p-2439.html							| http://www.seeedstudio.com/wiki/Grove_-_Air_Quality_Sensor_v1.3
	 * @ref SG::EGroveType::kIMU10DOF									| 101020079 |																			| http://www.seeedstudio.com/depot/Grove-IMU-10DOF-p-2386.html										| http://www.seeedstudio.com/wiki/Grove_-_IMU_10DOF
	 * @ref SG::EGroveType::kIMU9DOFV2									| 101020080 |																			| http://www.seeedstudio.com/depot/Grove-IMU-9DOF-v20-p-2400.html									| http://www.seeedstudio.com/wiki/Grove_-_IMU_9DOF_v2.0
	 * @ref SG::EGroveType::k6AxisAccelerometerAndCompassV2				| 101020081 | See @ref SG::EGroveType::k6AxisAccelerometerAndCompassV1.					| http://www.seeedstudio.com/depot/Grove-6Axis-AccelerometerCompass-v20-p-2476.html					| http://www.seeedstudio.com/wiki/Grove_-_6-Axis_Accelerometer&Compass_V2.0
	 * @ref SG::EGroveType::kFingerClipHeartRateSensorWithShell			| 101020082 |																			| http://www.seeedstudio.com/depot/Grove-Fingerclip-Heart-Rate-Sensor-with-shell-p-2420.html		| http://www.seeedstudio.com/wiki/Grove_-_Finger-clip_Heart_Rate_Sensor_with_shell
	 * @ref SG::EGroveType::kGestureV1									| 101020083 |																			| http://www.seeedstudio.com/depot/Grove-Gesture-p-2463.html										| http://www.seeedstudio.com/wiki/Grove_-_Gesture_v1.0
	 * n/a																| 101020084 | n/a																		| n/a																								| n/a
	 * n/a																| 101020085 | n/a																		| n/a																								| n/a
	 * n/a																| 101020086 | n/a																		| n/a																								| n/a
	 * @ref SG::EGroveType::kTemperatureAndHumiditySensorHDC1000		| 101020087 |																			| http://www.seeedstudio.com/depot/Grove-TemperatureHumidity-Sensor-HDC1000-p-2535.html				| http://www.seeedstudio.com/wiki/Grove_-_Temperature&Humidity_Sensor_(HDC1000)
	 * @ref SG::EGroveType::kMultichannelGasSensor						| 101020088 |																			| http://www.seeedstudio.com/depot/Grove-Multichannel-Gas-Sensor-p-2502.html						| http://www.seeedstudio.com/wiki/Grove_-_Multichannel_Gas_Sensor
	 * @ref SG::EGroveType::kSunlightSensor								| 101020089 |																			| http://www.seeedstudio.com/depot/Grove-Sunlight-Sensor-p-2530.html								| http://www.seeedstudio.com/wiki/Grove_-_Sunlight_Sensor
	 * @ref SG::EGroveType::kWaterAtomization							| 101020090 |																			| http://www.seeedstudio.com/depot/Grove-Water-Atomization-v10-p-2542.html							| http://www.seeedstudio.com/wiki/Grove_-_Water_Atomization
	 * @ref SG::EGroveType::kMiniTrackBall								| 101020091 |																			| http://www.seeedstudio.com/depot/Grove-Mini-Track-Ball-p-2586.html								| http://www.seeedstudio.com/wiki/Grove_-_Mini_Track_Ball
	 * @ref SG::EGroveType::kDMX512										| 103020000 |																			| http://www.seeedstudio.com/depot/Grove-DMX512-p-1447.html											| http://www.seeedstudio.com/wiki/Grove_-_DMX512
	 * n/a																| 103020001 | n/a																		| n/a																								| n/a
	 * @ref SG::EGroveType::kBeeSocket									| 103020002 |																			| http://www.seeedstudio.com/depot/Grove-Bee-Socket-p-1449.html										| http://www.seeedstudio.com/wiki/Grove_-_Bee_Socket
	 * @ref SG::EGroveType::kPS2Adapter									| 103020003 |																			| http://www.seeedstudio.com/depot/Grove-PS2-Adapter-p-966.html										| http://www.seeedstudio.com/wiki/Grove_-_PS/2_Adapter
	 * @ref SG::EGroveType::kSolidStateRelay							| 103020004 |																			| http://www.seeedstudio.com/depot/Grove-Solid-State-Relay-p-1359.html								| http://www.seeedstudio.com/wiki/Grove_-_Solid_State_Relay
	 * @ref SG::EGroveType::kRelayV12									| 103020005 |																			| http://www.seeedstudio.com/depot/Grove-Relay-p-769.html											| http://www.seeedstudio.com/wiki/Grove_-_Relay
	 * @ref SG::EGroveType::kI2CHub										| 103020006 |																			| http://www.seeedstudio.com/depot/Grove-I2C-Hub-p-851.html											| http://www.seeedstudio.com/wiki/Grove_-_I2C_Hub
	 * @ref SG::EGroveType::kScrewTerminal								| 103020007 |																			| http://www.seeedstudio.com/depot/Grove-Screw-Terminal-p-996.html									| http://www.seeedstudio.com/wiki/Grove_-_Screw_Terminal
	 * @ref SG::EGroveType::kMOSFET										| 103020008 |																			| http://www.seeedstudio.com/depot/Grove-MOSFET-p-1594.html											| http://www.seeedstudio.com/wiki/Grove_-_MOSFET
	 * @ref SG::EGroveType::kNunChuck									| 103020009 |																			| http://www.seeedstudio.com/depot/Grove-Nunchuck-p-1474.html										| http://www.seeedstudio.com/wiki/Grove_-_NunChuck
	 * @ref SG::EGroveType::k2CoilLatchingRelay							| 103020010 |																			| http://www.seeedstudio.com/depot/Grove-2Coil-Latching-Relay-p-1446.html							| http://www.seeedstudio.com/wiki/Grove_-_2-Coil_Latching_Relay
	 * @ref SG::EGroveType::kRJ45Adapter								| 103020011 |																			| http://www.seeedstudio.com/depot/Grove-RJ45-Adapter-p-1426.html									| http://www.seeedstudio.com/wiki/Grove_-_RJ45_Adapter
	 * @ref SG::EGroveType::kSPDTRelay30A								| 103020012 |																			| http://www.seeedstudio.com/depot/Grove-SPDT-Relay30A-p-1473.html									| http://www.seeedstudio.com/wiki/Grove_-_SPDT_Relay(30A)
	 * @ref SG::EGroveType::kI2CADC										| 103020013 |																			| http://www.seeedstudio.com/depot/Grove-I2C-ADC-p-1580.html										| http://www.seeedstudio.com/wiki/Grove_-_I2C_ADC
	 * @ref SG::EGroveType::kDryReedRelay								| 103020014 |																			| http://www.seeedstudio.com/depot/Grove-DryReed-Relay-p-1412.html									| http://www.seeedstudio.com/wiki/Grove_-_Dry-Reed_Relay
	 * @ref SG::EGroveType::kGroveCapeForBeagleBone						| 103020015 |																			| http://www.seeedstudio.com/depot/Grove-Cape-for-BeagleBone-Series-p-1718.html						| http://www.seeedstudio.com/wiki/Grove_Cape_for_BeagleBone_Series
	 * @ref SG::EGroveType::kDifferentialAmplifierV12					| 103020016 |																			| http://www.seeedstudio.com/depot/Grove-Differential-Amplifier-p-1284.html							| http://www.seeedstudio.com/wiki/Grove_-_Differential_Amplifier_v1.2
	 * n/a																| 103020017 | n/a																		| n/a																								| n/a
	 * @ref SG::EGroveType::kRecorder									| 103020018 |																			| http://www.seeedstudio.com/depot/Grove-Recorder-p-1825.html										| http://www.seeedstudio.com/wiki/Grove_-_Recorder
	 * @ref SG::EGroveType::kVoltageDivider								| 104020000 |																			| http://www.seeedstudio.com/depot/Grove-Voltage-Divider-p-1472.html								| http://www.seeedstudio.com/wiki/Grove_-_Voltage_Divider
	 * @ref SG::EGroveType::kVariableColorLED							| 104020001 |																			| http://www.seeedstudio.com/depot/Grove-Variable-Color-LED-p-852.html								| http://www.seeedstudio.com/wiki/Grove_-_Variable_Color_LED
	 * @ref SG::EGroveType::kPurpleLED									| 104020002 |																			| http://www.seeedstudio.com/depot/Grove-Purple-LED-3mm-p-1143.html									| http://www.seeedstudio.com/wiki/Grove_-_LED_Socket_Kit
	 * @ref SG::EGroveType::kBlinkM										| 104020003 | Discontinued.																| http://www.seeedstudio.com/depot/Grove-BlinkM-p-826.html											| http://www.seeedstudio.com/wiki/Grove_-_BlinkM
	 * @ref SG::EGroveType::kSerialLCD									| 104020004 | Discontinued.  See @ref SG::EGroveType::kLCDRGBBacklight.					| http://www.seeedstudio.com/depot/Grove-Serial-LCD-p-773.html										| http://www.seeedstudio.com/wiki/Grove_-_Serial_LCD_V0.9
	 * @ref SG::EGroveType::kLEDLightString								| 104020005 |																			| http://www.seeedstudio.com/depot/Grove-LED-String-Light-p-2324.html								| http://www.seeedstudio.com/wiki/Grove_-_LED_String_Light
	 * @ref SG::EGroveType::kLEDBarV2									| 104020006 |																			| http://www.seeedstudio.com/depot/Grove-LED-Bar-v20-p-2474.html									| http://www.seeedstudio.com/wiki/Grove_-_LED_Bar
	 * @ref SG::EGroveType::kLCDRGBBacklight							| 104030001 |																			| http://www.seeedstudio.com/depot/Grove-LCD-RGB-Backlight-p-1643.html								| http://www.seeedstudio.com/wiki/Grove_-_LCD_RGB_Backlight
	 * @ref SG::EGroveType::kLEDBar										| 104030002 | Discontinued.  See @ref SG::EGroveType::kLEDBarV2.						| http://www.seeedstudio.com/depot/Grove-LED-Bar-p-1178.html										| http://www.seeedstudio.com/wiki/Grove_-_LED_Bar
	 * @ref SG::EGroveType::k4DigitDisplay								| 104030003 |																			| http://www.seeedstudio.com/depot/Grove-4Digit-Display-p-1198.html									| http://www.seeedstudio.com/wiki/Grove_-_4-Digit_Display
	 * n/a																| 104030004 | n/a																		| n/a																								| n/a
	 * @ref SG::EGroveType::kRedLED										| 104030005 |																			| http://www.seeedstudio.com/depot/Grove-Red-LED-p-1142.html										| http://www.seeedstudio.com/wiki/Grove_-_LED
	 * @ref SG::EGroveType::kChainableRGBLED							| 104030006 |																			| http://www.seeedstudio.com/depot/Grove-Chainable-RGB-LED-p-850.html								| http://www.seeedstudio.com/wiki/Grove_-_Chainable_RGB_LED
	 * @ref SG::EGroveType::kGreenLED									| 104030007 |																			| http://www.seeedstudio.com/depot/Grove-Green-LED-p-1144.html										| http://www.seeedstudio.com/wiki/Grove_-_LED_Socket_Kit
	 * @ref SG::EGroveType::kOLEDDisplay1_12							| 104030008 | 128x64																	| http://www.seeedstudio.com/depot/Grove-OLED-Display-112-p-781.html								| http://www.seeedstudio.com/wiki/Grove_-_OLED_Display_128%2A64
	 * @ref SG::EGroveType::kWhiteLED									| 104030009 |																			| http://www.seeedstudio.com/depot/Grove-White-LED-p-1140.html										| http://www.seeedstudio.com/wiki/Grove_-_LED
	 * @ref SG::EGroveType::kBlueLED									| 104030010 |																			| http://www.seeedstudio.com/depot/Grove-Blue-LED-p-1139.html										| http://www.seeedstudio.com/wiki/Grove_-_LED
	 * @ref SG::EGroveType::kOLEDDisplay0_96							| 104030011 | 96x96																		| http://www.seeedstudio.com/depot/Grove-OLED-Display-096-p-824.html								| http://www.seeedstudio.com/wiki/Grove_-_OLED_Display_96%2A96
	 * n/a																| 104030012 | n/a																		| n/a																								| n/a
	 * @ref SG::EGroveType::kCircularLED								| 104030013 |																			| http://www.seeedstudio.com/depot/Grove-Circular-LED-p-1353.html									| http://www.seeedstudio.com/wiki/Grove_-_Circular_LED
	 * @ref SG::EGroveType::kMultiColorFlashLED							| 104030014 |																			| http://www.seeedstudio.com/depot/Grove-Multi-Color-Flash-LED-5mm-p-1141.html						| http://www.seeedstudio.com/wiki/Grove_-_LED
	 * @ref SG::EGroveType::kLED										| 104030015 | Discontinued.  See @ref SG::EGroveType::kGreenLED.						| http://www.seeedstudio.com/depot/Grove-LED-p-767.html												| n/a
	 * @ref SG::EGroveType::kI2CMotorDriverV12							| 105020001 |																			| http://www.seeedstudio.com/depot/Grove-I2C-Motor-Driver-p-907.html								| http://www.seeedstudio.com/wiki/Grove_-_I2C_Motor_Driver_V1.2
	 * @ref SG::EGroveType::kLEDStripDriver								| 105020002 |																			| http://www.seeedstudio.com/depot/Grove-LED-Strip-Driver-p-1197.html								| http://www.seeedstudio.com/wiki/Grove_-_LED_Strip_Driver
	 * @ref SG::EGroveType::kVibrationMotor								| 105020003 |																			| http://www.seeedstudio.com/depot/Grove-Vibration-Motor-p-839.html									| http://www.seeedstudio.com/wiki/Grove_-_Vibration_Motor
	 * @ref SG::EGroveType::kMiniFan									| 105020004 |																			| http://www.seeedstudio.com/depot/Grove-Mini-Fan-p-1819.html										| http://www.seeedstudio.com/wiki/Grove_-_Mini_Fan
	 * @ref SG::EGroveType::kELDriver									| 105020005 |																			| http://www.seeedstudio.com/depot/GroveEL-Driver-p-2269.html										| http://www.seeedstudio.com/wiki/Grove_-_EL_Driver
	 * n/a																| 105020006 | n/a																		| n/a																								| n/a
	 * n/a																| 105020007 | n/a																		| n/a																								| n/a
	 * n/a																| 105020008 | n/a																		| n/a																								| n/a
	 * n/a																| 105020009 | n/a																		| n/a																								| n/a
	 * @ref SG::EGroveType::kI2CMotorDriverV10							| 105020010 |																			| http://www.seeedstudio.com/depot/Grove%C2%A0%C2%A0I2C%C2%A0Mini%C2%A0Motor%C2%A0Driver-p-2508.html | http://www.seeedstudio.com/wiki/Grove_-_Mini_I2C_Motor_Driver_v1.0
	 * @ref SG::EGroveType::kHapticMotor								| 105020011 |																			| http://www.seeedstudio.com/depot/Grove%C2%A0%C2%A0Haptic%C2%A0Motor-p-2546.html					| http://www.seeedstudio.com/wiki/Grove_-_Haptic_Motor
	 * @ref SG::EGroveType::kBuzzer										| 107020000 |																			| http://www.seeedstudio.com/depot/Grove-Buzzer-p-768.html											| http://www.seeedstudio.com/wiki/Grove_-_Buzzer
	 * @ref SG::EGroveType::kSpeaker									| 107020001 |																			| http://www.seeedstudio.com/depot/Grove-Speaker-p-1445.html										| http://www.seeedstudio.com/wiki/Grove_-_Speaker
	 * @ref SG::EGroveType::kSerialMP3Player							| 107020002 | See @ref SG::EGroveType::kMP3V2.											| http://www.seeedstudio.com/depot/Grove-Serial-MP3-Player-p-1542.html								| http://www.seeedstudio.com/wiki/Grove_-_Serial_MP3_Player
	 * n/a																| 107020003 | n/a																		| n/a																								| n/a
	 * @ref SG::EGroveType::kSoundRecorder								| 107020004 | Discontinued.																| http://www.seeedstudio.com/depot/Grove-Sound-Recorder-p-904.html									| http://www.seeedstudio.com/wiki/Grove_-_Sound_Recorder
	 * @ref SG::EGroveType::kFMReceiver									| 107020005 |																			| http://www.seeedstudio.com/depot/Grove-FM-Receiver-p-1841.html									| http://www.seeedstudio.com/wiki/Grove_-_FM_Receiver
	 * @ref SG::EGroveType::kI2CFMReceiver								| 107020006 |																			| http://www.seeedstudio.com/depot/Grove-I2C-FM-Receiver-p-1953.html								| http://www.seeedstudio.com/wiki/Grove_-_I2C_FM_Receiver
	 * n/a																| 107020007 | n/a																		| n/a																								| n/a
	 * @ref SG::EGroveType::kMP3V2										| 107020008 | See @ref SG::EGroveType::kSerialMP3Player.								| http://www.seeedstudio.com/depot/Grove-MP3-v20-p-2597.html										| http://www.seeedstudio.com/wiki/Grove_-_MP3_v2.0
	 * @ref SG::EGroveType::kButtonP									| 111020000 | See @ref SG::EGroveType::kButton.											| http://www.seeedstudio.com/depot/Grove-ButtonP-p-1243.html										| http://www.seeedstudio.com/wiki/Grove_-_Button
	 * @ref SG::EGroveType::kEncoder									| 111020001 |																			| http://www.seeedstudio.com/depot/Grove-Encoder-p-1352.html										| http://www.seeedstudio.com/wiki/Grove_-_Encoder
	 * @ref SG::EGroveType::kHighTemperatureSensor						| 111020002 |																			| http://www.seeedstudio.com/depot/Grove-High-Temperature-Sensor-p-1810.html						| http://www.seeedstudio.com/wiki/Grove_-_High_Temperature_Sensor
	 * @ref SG::EGroveType::kSerialRFPro								| 113020000 |																			| http://www.seeedstudio.com/depot/Grove-Serial-RF-Pro-p-794.html									| http://www.seeedstudio.com/wiki/Grove_-_Serial_RF_Pro
	 * @ref SG::EGroveType::k315MHzSimpleRFLinkKit						| 113020001 |																			| http://www.seeedstudio.com/depot/Grove-315MHz-Simple-RF-Link-Kit-p-1061.html						| http://www.seeedstudio.com/wiki/Grove_-_315MHz_Simple_RF_Link_Kit
	 * @ref SG::EGroveType::k125KHzRFIDReader							| 113020002 |																			| http://www.seeedstudio.com/depot/Grove-125KHz-RFID-Reader-p-1008.html								| http://www.seeedstudio.com/wiki/Grove_-_125KHz_RFID_Reader
	 * @ref SG::EGroveType::kGPS										| 113020003 |																			| http://www.seeedstudio.com/depot/Grove-GPS-p-959.html												| http://www.seeedstudio.com/wiki/Grove_-_GPS
	 * @ref SG::EGroveType::kXBeeCarrier								| 113020004 |																			| http://www.seeedstudio.com/depot/Grove-XBee-Carrier-p-905.html									| http://www.seeedstudio.com/wiki/Grove_-_XBee_Carrier
	 * @ref SG::EGroveType::kSerialBluetooth							| 113020005 | Discontinued.  See @ref SG::EGroveType::kSerialBluetoothV3.				| http://www.seeedstudio.com/depot/Grove-Serial-Bluetooth-p-795.html								| http://www.seeedstudio.com/wiki/Grove_-_Serial_Bluetooth
	 * @ref SG::EGroveType::kNFC										| 113020006 |																			| http://www.seeedstudio.com/depot/Grove-NFC-p-1804.html											| http://www.seeedstudio.com/wiki/Grove_-_NFC
	 * @ref SG::EGroveType::kBLE										| 113020007 |																			| http://www.seeedstudio.com/depot/Grove-BLE-p-1929.html											| http://www.seeedstudio.com/wiki/Grove_BLE_v1
	 * @ref SG::EGroveType::kSerialBluetoothV3							| 113020008 |																			| http://www.seeedstudio.com/depot/Grove-Serial-Bluetooth-v30-p-2475.html							| http://www.seeedstudio.com/wiki/Grove_-_Serial_Bluetooth_v3.0
	 * @ref SG::EGroveType::kBLEDualModel								| 113020009 |																			| http://www.seeedstudio.com/depot/Grove-BLE-dual-model-p-2407.html									| http://www.seeedstudio.com/wiki/Grove_-_BLE_(dual_model)_v1.0
	 * @ref SG::EGroveType::kUartWifi									| 113020010 |																			| http://www.seeedstudio.com/depot/Grove-Uart-Wifi-p-2495.html										| http://www.seeedstudio.com/wiki/Grove_-_UART_WiFi
	 * @ref SG::EGroveType::k433MHzSimpleRFLinkKit						| 113060000 |																			| http://www.seeedstudio.com/depot/Grove-433MHz-Simple-RF-link-kit-p-1062.html						| http://www.seeedstudio.com/wiki/Grove_-_433MHz_Simple_RF_Link_Kit
	 * @ref SG::EGroveType::kServo										| 316010005 |																			| http://www.seeedstudio.com/depot/Grove-Servo-p-1241.html											| http://www.seeedstudio.com/wiki/Grove_-_Servo
	 */
	enum class EGroveType
	{
		// If you add new types, remember to also update SG::get_grove_type_name_map().

		kInvalid										= 0,
		kUnknown										,
		kNone											,

		// beware, if you change these numbers, you'll also need to fix up SG::get_sku_from_grove_type()

		kSerialCameraKit								= 4000,		///< SKU #101020000
		kHCHOSensor										,			///< SKU #101020001
		kGasSensorO2									,			///< SKU #101020002
		kButton											,			///< SKU #101020003
		kSwitchP										,			///< SKU #101020004
		kCollisionSensor								,			///< SKU #101020005
		kGasSensorMQ3									,			///< SKU #101020006
		kMiniCamera										,			///< SKU #101020007
		kMoistureSensor									,			///< SKU #101020008
		kLineFinder										,			///< SKU #101020009
		kUltrasonicRanger								,			///< SKU #101020010
		kTemperatureAndHumiditySensor					,			///< SKU #101020011
		kDustSensor										,			///< SKU #101020012
		kRTC											,			///< SKU #101020013
		kAnalogLightSensor								,			///< SKU #101020014
		kTemperatureSensorV12							,			///< SKU #101020015
		kInfraredReceiver								,			///< SKU #101020016
		kRotaryAngleSensor								,			///< SKU #101020017
		kWaterSensor									,			///< SKU #101020018
		kTemperatureAndHumiditySensorPro				,			///< SKU #101020019
		kPIRMotionSensor								,			///< SKU #101020020
		kAirQualitySensor								,			///< SKU #101020021
		kAnalogLightSensorP								,			///< SKU #101020022
		kSoundSensor									,			///< SKU #101020023
		kMegaShield										,			///< SKU #101020024
		kTiltSwitch										,			///< SKU #101020025
		kInfraredEmitter								,			///< SKU #101020026
		kElectricitySensor								,			///< SKU #101020027
		kThumbJoystick									,			///< SKU #101020028
		kInfraredReflectiveSensor						,			///< SKU #101020029
		kDigitalLightSensor								,			///< SKU #101020030
		kPiezoVibrationSensor							,			///< SKU #101020031
		kBarometerSensor								,			///< SKU #101020032
		kEarClipHeartRateSensor							,			///< SKU #101020033
		k3AxisDigitalCompass							,			///< SKU #101020034
		kProtoShield									,			///< SKU #101020035
		kSlidePotentiometer								,			///< SKU #101020036
		kTouchSensor									,			///< SKU #101020037
		kMagneticSwitch									,			///< SKU #101020038
		k3AxisDigitalAccelerometer1_5g					,			///< SKU #101020039
		kIRDistanceInterrupt							,			///< SKU #101020040
		kI2CColorSensor									,			///< SKU #101020041
		k80cmInfraredProximitySensor					,			///< SKU #101020042
		kUVSensor										,			///< SKU #101020043
		kAlcoholSensor									,			///< SKU #101020044
		kGasSensorMQ9									,			///< SKU #101020045
		kHallSensor										,			///< SKU #101020046
		kI2CTouchSensor									,			///< SKU #101020047
		kRotaryAngleSensorP								,			///< SKU #101020048
		kFlameSensor									,			///< SKU #101020049
		k3AxisDigitalGyro								,			///< SKU #101020050
		k3AxisAnalogAccelerometer						,			///< SKU #101020051
		kGSRSensor										,			///< SKU #101020052
		kSingleAxisAnalogGyro							,			///< SKU #101020053
		k3AxisDigitalAccelerometer16g					,			///< SKU #101020054
		kGasSensorMQ2									,			///< SKU #101020055
		kGasSensorMQ5									,			///< SKU #101020056
		kFingerprintSensor								,			///< SKU #101020057
		kEMGDetector									,			///< SKU #101020058
		kIMU9DOFV1										,			///< SKU #101020059
		kMotionSensorLargeLens							,			///< SKU #101020060
		k6AxisAccelerometerAndCompassV1					,			///< SKU #101020061
		kAnalogInfraredTemperatureSensor				,			///< SKU #101020062
		kLoudnessSensor									,			///< SKU #101020063
		kSerialCamera									,			///< SKU #101020064
		kPHSensor										,			///< SKU #101020065
		kChestStrapHeartRateSensor						,			///< SKU #101020066
		kCO2Sensor										,			///< SKU #101020067
		kBarometerHighAccuracy							,			///< SKU #101020068
		kQTouchSensor									,			///< SKU #101020069
		kNFCTag											,			///< SKU #101020070
		k3AxisDigitalAccelerometer400g					,			///< SKU #101020071
		kBarometerSensorBMP180							,			///< SKU #101020072
		kElectromagnet									,			///< SKU #101020073
		kTemperatureAndHumidityHighAccuracyAndMini		,			///< SKU #101020074
																	//   SKU #101020075 unused
		kLuminanceSensor								= 4076,		///< SKU #101020076
		kDigitalInfraredTemperatureSensor				,			///< SKU #101020077
		kAirQualitySensorV13							,			///< SKU #101020078
		kIMU10DOF										,			///< SKU #101020079
		kIMU9DOFV2										,			///< SKU #101020080
		k6AxisAccelerometerAndCompassV2					,			///< SKU #101020081
		kFingerClipHeartRateSensorWithShell				,			///< SKU #101020082
		kGestureV1										,			///< SKU #101020083
																	//   SKU #101020084 unused
																	//   SKU #101020085 unused
																	//   SKU #101020086 unused
		kTemperatureAndHumiditySensorHDC1000			= 4087,		///< SKU #101020087
		kMultichannelGasSensor							,			///< SKU #101020088
		kSunlightSensor									,			///< SKU #101020089
		kWaterAtomization								,			///< SKU #101020090
		kMiniTrackBall									,			///< SKU #101020091
		//
		kDMX512											= 5000,		///< SKU #103020000
																	//   SKU #103020001 unused
		kBeeSocket										= 5002,		///< SKU #103020002
		kPS2Adapter										,			///< SKU #103020003
		kSolidStateRelay								,			///< SKU #103020004
		kRelayV12										,			///< SKU #103020005
		kI2CHub											,			///< SKU #103020006
		kScrewTerminal									,			///< SKU #103020007
		kMOSFET											,			///< SKU #103020008
		kNunChuck										,			///< SKU #103020009
		k2CoilLatchingRelay								,			///< SKU #103020010
		kRJ45Adapter									,			///< SKU #103020011
		kSPDTRelay30A									,			///< SKU #103020012
		kI2CADC											,			///< SKU #103020013
		kDryReedRelay									,			///< SKU #103020014
		kGroveCapeForBeagleBone							,			///< SKU #103020015
		kDifferentialAmplifierV12						,			///< SKU #103020016
																	//   SKU #103020017 unused
		kRecorder										= 5018,		///< SKU #103020018
		//
		kVoltageDivider									= 6000,		///< SKU #104020000
		kVariableColorLED								,			///< SKU #104020001
		kPurpleLED										,			///< SKU #104020002
		kBlinkM											,			///< SKU #104020003
		kSerialLCD										,			///< SKU #104020004
		kLEDLightString									,			///< SKU #104020005
		kLEDBarV2										,			///< SKU #104020006
		//
		kLCDRGBBacklight								= 7001,		///< SKU #104030001
		kLEDBar											,			///< SKU #104030002
		k4DigitDisplay									,			///< SKU #104030003
																	//   SKU #104030004 unused
		kRedLED											= 7005,		///< SKU #104030005
		kChainableRGBLED								,			///< SKU #104030006
		kGreenLED										,			///< SKU #104030007
		kOLEDDisplay1_12								,			///< SKU #104030008
		kWhiteLED										,			///< SKU #104030009
		kBlueLED										,			///< SKU #104030010
		kOLEDDisplay0_96								,			///< SKU #104030011
																	//   SKU #104030012
		kCircularLED									,			///< SKU #104030013
		kMultiColorFlashLED								,			///< SKU #104030014
		kLED											,			///< SKU #104030015
		//
		kI2CMotorDriverV12								= 8001,		///< SKU #105020001
		kLEDStripDriver									,			///< SKU #105020002
		kVibrationMotor									,			///< SKU #105020003
		kMiniFan										,			///< SKU #105020004
		kELDriver										,			///< SKU #105020005
																	//   SKU #105020006 unused
																	//   SKU #105020007 unused
																	//   SKU #105020008 unused
																	//   SKU #105020009 unused
		kI2CMotorDriverV10								= 8010,		///< SKU #105020010
		kHapticMotor									,			///< SKU #105020011
		//
		kBuzzer											= 9000,		///< SKU #107020000
		kSpeaker										,			///< SKU #107020001
		kSerialMP3Player								,			///< SKU #107020002
																	//   SKU #107020003 unused
		kSoundRecorder									= 9004,		///< SKU #107020004
		kFMReceiver										,			///< SKU #107020005
		kI2CFMReceiver									,			///< SKU #107020006
																	//   SKU #107020007 unused
		kMP3V2											= 9008,		///< SKU #107020008
		//
		kButtonP										= 10000,	///< SKU #111020000
		kEncoder										,			///< SKU #111020001
		kHighTemperatureSensor							,			///< SKU #111020002
		//
		kSerialRFPro									= 11000,	///< SKU #113020000
		k315MHzSimpleRFLinkKit							,			///< SKU #113020001
		k125KHzRFIDReader								,			///< SKU #113020002
		kGPS											,			///< SKU #113020003
		kXBeeCarrier									,			///< SKU #113020004
		kSerialBluetooth								,			///< SKU #113020005
		kNFC											,			///< SKU #113020006
		kBLE											,			///< SKU #113020007
		kSerialBluetoothV3								,			///< SKU #113020008
		kBLEDualModel									,			///< SKU #113020009
		kUartWifi										,			///< SKU #113020010
		//
		k433MHzSimpleRFLinkKit							= 15000,	///< SKU #113060000
		kServo											= 15001		///< SKU #316010005
	};

	/// Vector of Seeed Grove types.
	typedef std::vector<EGroveType> VGroveTypes;

	/// Set of unique Seeed Grove types, ordered by numeric type.
	typedef std::set<EGroveType> SGroveTypes;

	/// Map of Seeed Grove types to strings, used to map the types to a usable name.
	typedef std::map<EGroveType, std::string> MGroveTypeNames;

	/** Return the Seeed SKU for the given Grove type.  This is just a guess since I don't work for Seeed and
	 * have no insight into their SKU numbers other than what is publicly available on the Seeed web site.
	 */
	uint32_t get_sku_from_grove_type( const EGroveType &grove_type );

	/// Return a vector containing every possible Grove type.
	VGroveTypes get_all_grove_types( void );

	/// Get a map of all the grove types and the corresponding names.
	MGroveTypeNames get_grove_type_name_map( void );

	/// Get a short text name for the given type.
	std::string to_string( const EGroveType &type );

	/// I2C address for Seeed Grove products.
	typedef uint8_t I2CAddress;

	/// Map specific Grove type to a I2C address.  Note that some types have multiple addresses.
	typedef std::multimap<EGroveType, I2CAddress> MGroveTypeI2CAddresses;

	/// Vector of I2C addresses.
	typedef std::vector<I2CAddress> VI2CAddresses;

	/// Get a map of all known Grove types and their I2C address.  @see http://www.seeedstudio.com/wiki/I2C_And_I2C_Address_of_Seeed_Product
	MGroveTypeI2CAddresses get_grove_type_addresses( void );

	/// Get all the known addresses for a given Grove type.
	VI2CAddresses get_grove_type_addresses( const EGroveType &type );
	
	VGroveTypes detect( void );
}
