/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_GpioManagement.cpp 1720 2016-03-24 08:56:29Z stephane $
 */

#include "sg_GpioManagement.hpp"
#include <system_error>
#include <fstream>
#include <thread>
#include <unistd.h>
#include <sys/stat.h>


SG::GpioManagement::GpioManagement( void ) :
		unexport_all_gpio_on_destruction( true )
{
	return;
}


SG::GpioManagement::~GpioManagement( void )
{
	if (unexport_all_gpio_on_destruction)
	{
		cleanup();
	}

	return;
}


SG::GpioManagement &SG::GpioManagement::get( void )
{
	static GpioManagement sg_gpio;
	
	return sg_gpio;
}


SG::GpioManagement &SG::GpioManagement::cleanup( void )
{
	const SGPIO s = get_exported();
	for (const GPIO gpio : s)
	{
		gpio_unexport( gpio );
	}

	return *this;
}


SG::SGPIO SG::GpioManagement::get_exported( void )
{
	SGPIO s;

	for (const auto iter : gpio_direction_map)
	{
		s.insert( iter.first );
	}

	return s;
}


SG::GpioManagement::MGPIODirection SG::GpioManagement::get_direction_map( void )
{
	MGPIODirection m( gpio_direction_map );

	return m;
}


SG::GpioManagement &SG::GpioManagement::gpio_export( const GPIO gpio, const SG::GpioManagement::EDirection direction )
{
	if (is_exported(gpio) == true)
	{
		/// @throw std::invalid_argument if the specified GPIO has already been exported.
		throw std::invalid_argument( "GPIO " + std::to_string(gpio) + " has already been exported." );
	}

	const uid_t euid = geteuid();

	const std::string filename = "/sys/class/gpio/export";
	std::ofstream ofs( filename );
	if (ofs.bad())
	{
		/// @throw std::system_error if the GPIO export file cannot be opened with write access.
		const std::error_code ec( errno, std::system_category() );
		throw std::system_error( ec, "SG++ failed to open " + filename + " for writing." + (euid == 0 ? "" : " (Try as root?)") );
	}

	ofs << gpio << std::endl;
	ofs.close();

	if (is_exported( gpio, 10 ) == false)
	{
		/// @throw std::runtime_error if the specified GPIO fails to export after several tries.
		throw std::runtime_error( "SG++ failed to export GPIO " + std::to_string(gpio) + "." + (euid == 0 ? "" : " (Try as root?)") );
	}

	gpio_direction_map[ gpio ] = EDirection::kUnknown;

	set_direction( gpio, direction );

	return *this;
}


SG::GpioManagement &SG::GpioManagement::gpio_unexport( const GPIO gpio )
{
	const std::string filename = "/sys/class/gpio/unexport";

	std::ofstream ofs( filename );
	if (ofs.bad())
	{
		/// @throw std::system_error if the GPIO unexport file cannot be opened with write access.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, "SG++ failed to open " + filename + " for writing." );
	}

	ofs << gpio << std::endl;
	ofs.close();

	gpio_direction_map.erase( gpio );

	return *this;
}


SG::GpioManagement &SG::GpioManagement::set_direction( const GPIO gpio, const SG::GpioManagement::EDirection direction )
{
	if (direction == EDirection::kInput		||
		direction == EDirection::kOutput	)
	{
		if (is_exported( gpio ) == false)
		{
			/// @throw invalid_argument if the specified GPIO has not been exported.
			throw std::invalid_argument( "SG++ cannot set the direction of unexported GPIO " + std::to_string(gpio) + "." );
		}

		const std::string filename = "/sys/class/gpio/gpio" + std::to_string(gpio) + "/direction";
		std::ofstream ofs( filename );
		if (ofs.bad())
		{
			/// @throw std::system_error if the GPIO direction file cannot be opened with write access.
			const std::error_code ec( errno, std::system_category());
			throw std::system_error( ec, "SG++ failed to open " + filename + " to set direction." );
		}
		ofs << (direction == EDirection::kInput ? "in" : "out") << std::endl;
	}

	gpio_direction_map[ gpio ] = direction;

	return *this;
}


bool SG::GpioManagement::is_exported( const GPIO gpio, int number_of_times_to_check )
{
	const std::string filename = "/sys/class/gpio/gpio" + std::to_string(gpio);

	// even if zero or a negative number is specified, run the check at least once
	do
	{
		struct stat buf;
		const int result = stat( filename.c_str(), &buf );

		if (result == 0)
		{
			// looks like the gpio has been exported
			return true;
		}

		// if we get here, then it doesn't seem to have been exported

		number_of_times_to_check --;

		if (number_of_times_to_check > 0)
		{
			// we'll need to check again, so sleep for a short duration before looping
			std::this_thread::sleep_for( std::chrono::milliseconds(150) );
		}

	} while (number_of_times_to_check > 0);

	return false;
}


SG::GpioManagement &SG::GpioManagement::set_value( const GPIO gpio, const SG::GpioManagement::EValue &value )
{
	if (value != EValue::kHigh	&&
		value != EValue::kLow	)
	{
		/// @throw invalid_argument if the value is anything other than kLow or kHigh.
		throw std::invalid_argument( "SG++ cannot set GPIO " + std::to_string(gpio) + " to " + std::to_string((int)value) + ". Must be kLow or kHigh." );
	}

	if (is_exported( gpio ) == false)
	{
		/// @throw invalid_argument if the specified GPIO has not been exported.
		throw std::invalid_argument( "SG++ cannot set unexported GPIO " + std::to_string(gpio) + (value == EValue::kHigh ? " high." : " low.") );
	}

	const std::string filename = "/sys/class/gpio/gpio" + std::to_string(gpio) + "/value";

	std::ofstream ofs( filename );
	if (ofs.bad())
	{
		/// @throw std::system_error if the GPIO value file cannot be opened with write access.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, "SG++ failed to open " + filename + " to set the value " + (value == EValue::kHigh ? "high." : "low.") );
	}
	ofs << (value == EValue::kLow ? 0 : 1) << std::endl;

	return *this;
}
