/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_BeagleBone_LED.hpp 1732 2016-04-01 05:22:06Z stephane $
 */

#pragma once

#include <string>


namespace SG
{
	namespace BeagleBone
	{
		/// Enumerator for the various LEDs on BeagleBone Black and BeagleBone Green.
		enum class LED
		{
			invalid		= 0,
			none		= 1,
			all			= 2,
			usr0		= 10,	///< LED closest to the outside and the @p RESET button.
			usr1		= 11,
			usr2		= 12,
			usr3		= 13,	///< LED closest to the Ethernet cable and the @p POWER button.
			min			= usr0,
			max			= usr3
		};

		/// Enumerator for the LED trigger type to use.  @see https://www.kernel.org/doc/Documentation/devicetree/bindings/leds/common.txt
		enum class Trigger
		{
			invalid		= 0	,
			none			,	///< Prevents the LED from automatically changing.  Use this trigger if you want to manually turn on and turn off the LED.
			nand_disk		,
			usb_gadget		,
			usb_host		,
			mmc0			,	///< Micro SD card is accessed.
			mmc1			,	///< Internal eMMC is accessed.
			timer			,
			oneshot			,	///< https://www.kernel.org/doc/Documentation/leds/ledtrig-oneshot.txt
			heartbeat		,	///< Periodic double-blink from the Linux kernel to indicate the device is still running.
			backlight		,
			gpio			,
			cpu0			,	///< Blink rate is determined by the load average.
			default_on
		};

		/// Convert the trigger enum to the string expected in the @p trigger file.
		inline std::string to_string( const Trigger trigger )
		{
			switch( trigger )
			{
				case Trigger::none:			return "none";
				case Trigger::nand_disk:	return "nand-disk";
				case Trigger::usb_gadget:	return "usb-gadget";
				case Trigger::usb_host:		return "usb-host";
				case Trigger::mmc0:			return "mmc0";
				case Trigger::mmc1:			return "mmc1";
				case Trigger::timer:		return "timer";
				case Trigger::oneshot:		return "oneshot";
				case Trigger::heartbeat:	return "heartbeat";
				case Trigger::backlight:	return "backlight";
				case Trigger::gpio:			return "gpio";
				case Trigger::cpu0:			return "cpu0";
				case Trigger::default_on:	return "default-on";
				default:					return "none";
			}
		}

		/// Return the default trigger typically used by each LED.
		inline Trigger default_trigger( const SG::BeagleBone::LED led )
		{
			switch( led )
			{
				case LED::usr0:		return Trigger::heartbeat;
				case LED::usr1:		return Trigger::mmc0;
				case LED::usr2:		return Trigger::cpu0;
				case LED::usr3:		return Trigger::mmc1;
				default:			return Trigger::none;
			}
		}

		/** Determine the index of the specified LED.  For example, the index of @ref LED::usr0 is @p 0, while the
		 * index of @ref LED::usr3 is @p 3.  All other enums such as @ref LED::all will result in @p -1.
		 */
		inline int index_of( const SG::BeagleBone::LED led )
		{
			switch( led )
			{
				case LED::usr0:		return 0;
				case LED::usr1:		return 1;
				case LED::usr2:		return 2;
				case LED::usr3:		return 3;
				default:			return -1;
			}
		}

		/// Determine if the specified LED enum is a single physical LED, or a "pseudo" LED such as @ref LED::all.
		inline bool is_single_LED( const SG::BeagleBone::LED led ) { return index_of(led) >= 0; }

		/// Determine if the specified LED enum represents a "pseudo" LED, such as @ref LED::all or @ref LED::none.
		inline bool is_pseudo_LED( const SG::BeagleBone::LED led ) { return index_of(led) == -1; }
	}
}
