/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_BeagleBone_LEDControl.hpp 1828 2016-05-05 04:34:53Z stephane $
 */

#pragma once

#include "sg_BeagleBone_LED.hpp"
#include "sg_LED_patterns.hpp"
#include <string>
#include <thread>
#include <atomic>
#include <chrono>
#include <ctime>


namespace SG
{
	namespace BeagleBone
	{
		/** %BeagleBone LED software controller.
		 * The %BeagleBone LEDs are unrelated to Grove, but the SG++ library can be used to control some aspects of the
		 * LEDs.  There are 4 LEDs on the top of both the %BeagleBone Black (BBB) and the %BeagleBone Green (BBG).  The
		 * LEDs can be independently turned on, turned off, made to blink according to a pattern, or set to
		 * automatically turn on and off based on some hardware events.
		 *
		 * Write access to the LEDs is typically only for the @p root user.  If your application is not running as
		 * @p root, you'll want to ensure the @p trigger and @p brightness files have the right permissions and/or
		 * owner so they can be written to by SG++.  For example:
		 *
		 * ~~~~{.sh}
		 * sudo chmod a+w /sys/devices/platform/leds/leds/beaglebone:green:usr0/brightness
		 * sudo chmod a+w /sys/devices/platform/leds/leds/beaglebone:green:usr0/trigger
		 * ~~~~
		 *
		 * The full path to these files depends on both the distribution and version of Linux in use on the %BeagleBone.
		 * The method @ref LEDControl::get_base_path() can be used to find the right files.  Note there are 4 LEDs, and
		 * the SG++ library needs write access to both files per LED (@p trigger and @p brightness), so 8 files in all.
		 *
		 * Typical use would be:
		 *
		 * ~~~~
		 * SG::BeagleBone::LEDControl led_control;
		 * led_control.start_blink_thread();
		 * led_control.load_pattern( SG::BeagleBone::LEDPattern::cylon );
		 *
		 * // ...continue to do other work here while the blink
		 * // loop thread controls the LEDs in the given pattern...
		 * ~~~~
		 * 
		 * Another example where specific LEDs are turned on or off:
		 * 
		 * ~~~~
		 * if (foo)
		 * {
		 *     SG::BeagleBone::LEDControl led_control;
		 *
		 *     // make sure the LEDs remain "as-is" when led_control object goes out of scope
		 *     led_control.action_on_destruction = SG::BeagleBone::LEDControl::ActionOnDestruction::do_nothing;
		 *
		 *     // setup the LEDs to be manually controlled
		 *     led_control.set_trigger( SG::BeagleBone::LED::all, SG::BeagleBone::Trigger::none );
		 *
		 *     // turn off all the LEDs
		 *     led_control.turn_off( SG::BeagleBone::LED::all );
		 *
		 *     // turn on a single LED
		 *     led_control.turn_on( SG::BeagleBone::LED::usr2 );
		 * }
		 * ~~~~
		 *
		 * @note If LEDControl isn't successful in detecting or controlling the LEDs on your %BeagleBone, check the
		 * content of the text string stored in @ref error_message.
		 */
		class LEDControl
		{
			public:

				/// Destructor.  @see @ref action_on_destruction
				virtual ~LEDControl( void );

				/** Constructor.
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.load_pattern( SG::BeagleBone::LEDPattern::blink );
				 * led_control.start_blink_thread();
				 * ~~~~
				 */
				LEDControl( void );

				/** Look for the %BeagleBone LEDs and determine if they can be accessed and controled by this class.
				 * If this returns @p false -- indicating that the LEDs weren't properly detected -- all other methods
				 * in this class will return without performing any action on the LEDs when called.
				 *
				 * This method is automatically called by the constructor.
				 */
				virtual bool is_valid( void );

				/** Turn on or turn off the specified LED.
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.set_trigger( SG::BeagleBone::LED::all, SG::BeagleBone::Trigger::none );
				 * led_control.toggle( SG::BeagleBone::LED::all, true );
				 * ~~~~
				 */
				virtual LEDControl &toggle( const SG::BeagleBone::LED led, const bool enabled );

				/** Turn on the specified LED.  This is an alias for @ref LEDControl::toggle(..., true).
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.set_trigger( SG::BeagleBone::LED::all, SG::BeagleBone::Trigger::none );
				 * led_control.turn_off( SG::BeagleBone::LED::all );
				 * led_control.turn_on( SG::BeagleBone::LED::usr2 );
				 * ~~~~
				 */
				virtual LEDControl &turn_on( const SG::BeagleBone::LED led ) { return toggle(led, true); }

				/** Turn off the specified LED.  This is an alias for @ref LEDControl::toggle(..., false).
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.set_trigger( SG::BeagleBone::LED::all, SG::BeagleBone::Trigger::none );
				 * led_control.turn_on( SG::BeagleBone::LED::all );
				 * led_control.turn_off( SG::BeagleBone::LED::usr1 );
				 * ~~~~
				 */
				virtual LEDControl &turn_off( const SG::BeagleBone::LED led ) { return toggle(led, false); }

				/** Determine the base path to access the given LED.  You normally wont need to call this, as it is
				 * used internally by the LEDControl class.
				 * @see Some of the paths involved are described in @ref is_valid().
				 */
				virtual std::string get_base_path( const SG::BeagleBone::LED led );

				/** The trigger determines how a LED may be automatically enabled or disabled.  This method sets each
				 * individual @p trigger file to a specific value.  For example, the name of a trigger file could be
				 * @p "/sys/devices/platform/leds/leds/beaglebone:green:usr3/trigger".  By default, the 4 LEDs have
				 * triggers such as @p heartbeat and @p mmc0 to indicate different activies on the %BeagleBone.
				 *
				 * If you want to manually toggle some LEDs on and off, the trigger must first be set to @p none.
				 * Otherwise, your @p on or @p off values will get overwritten by the normal activity, such as the
				 * heartbeat or CPU activity.
				 *
				 * @note This method is automatically called with @p Trigger::none for all 4 LEDs in @ref blink_loop(),
				 * so you'll only need to call @ref set_trigger() if you want to do something with the LEDs other
				 * than the usual pattern and blink loop thread.
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 *
				 * // first, we set all 4 LED triggers to 'none' so they can be manually controlled
				 * led_control.set_trigger( SG::BeagleBone::LED::all, SG::BeagleBone::Trigger::none );
				 *
				 * // then we enable the Linux kernel 'heartbeat' double blink on the first LED
				 * led_control.set_trigger( SG::BeagleBone::LED::usr0, SG::BeagleBone::Trigger::heartbeat );
				 *
				 * // and we manually turn on the 2nd LED
				 * led_control.turn_on( SG::BeagleBone::LED::usr1 );
				 *
				 * // in this example, the on/off state of the 3rd and 4th LEDs is undefined, since
				 * // the LEDs could have been either on or off when the triggers were set to 'none'
				 * ~~~~
				 */
				virtual LEDControl &set_trigger( const SG::BeagleBone::LED led, const SG::BeagleBone::Trigger trigger );

				/** Set the trigger back to the default value typically used when a %BeagleBone first boots up.
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.set_default_trigger( SG::BeagleBone::LED::all );
				 * ~~~~
				 */
				virtual LEDControl &set_default_trigger( const SG::BeagleBone::LED led );

				/** After a certain amount of time when LEDControl isn't told to do something, it can consider this as
				 * the device having gone idle and apply a certain blink pattern.  This will only happen if both
				 * @ref set_idle_timeout() and @ref start_blink_thread() have been called, since the blink thread is
				 * what checks for idleness and takes care of toggling the lights.
				 *
				 * For example, the LEDs can be set to blink after 5 minutes of idleness:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.set_idle_timeout( 60, LEDPattern::blink );
				 * led_control.load_pattern( SG::BeagleBone::LEDPattern::inchwormBoth );
				 * led_control.start_blink_thread();
				 * ~~~~
				 * @{
				 */
				virtual LEDControl &set_idle_timeout( const size_t timeout_in_seconds = 600, const LEDPattern pattern = LEDPattern::none );
				virtual LEDControl &set_idle_timeout( const std::chrono::high_resolution_clock::duration timeout, const LEDPattern pattern = LEDPattern::none );
				virtual LEDControl &set_idle_timeout( const std::chrono::high_resolution_clock::duration timeout, const SG::BeagleBone::BlinkPattern pattern );
				/// @}

				/** Remove the idle timeout.  This prevents LEDControl from ever going into the <em>"idle timeout"</em>
				 * state.  This is the same as calling @ref set_idle_timeout() with a timeout of zero seconds.
				 */
				virtual LEDControl &clear_idle_timeout( void ) { return set_idle_timeout( 0 ); }

				/** Update the activity timestamp so LEDControl knows the device has not gone idle.  Several LEDControl
				 * methods call this internally to indicate the LEDControl object isn't idle.  Applications can also
				 * regularly call this to prevent LEDControl from going into an idle state.  Otherwise, once the time
				 * described in @ref set_idle_timeout() has elapsed, the device will be considered <em>"idle"</em>.
				 */
				virtual LEDControl &update_activity_timestamp( void );

				/** Determine if the device is idle.  If @ref set_idle_timeout() has not been called, then this method
				 * will return @p false.  Otherwise, the return value depends on the timeout compared to when the
				 * method @ref reset_idle_timestamp() was last called. @{
				 */
				virtual bool is_idle( const std::chrono::high_resolution_clock::time_point &now ) const;
				virtual bool is_idle( void ) const;
				/// @}

				/** Load the given blink pattern.  The default pattern set in @ref LEDControl::LEDControl() is
				 * @ref LEDPattern::none which turns off all LEDs, so you'll likely want to provide a different
				 * pattern to use if you want to see the LEDs blink.
				 *
				 * @note Calling @ref load_pattern() does not automatically start the blink loop.  To get the
				 * selected pattern to show on the LEDs, you also need to call @ref start_blink_thread() once,
				 * or manually setup your own timers or thread to manage the blink loop.
				 *
				 * If the blink loop is already running when @ref load_pattern() is called, the new pattern will
				 * take effect within 100 milliseconds.
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.load_pattern( SG::BeagleBone::LEDPattern::inchwormBoth );
				 * led_control.start_blink_thread();
				 * ~~~~
				 *
				 * @see @ref load_random_pattern()
				 *  @{
				 */
				virtual LEDControl &load_pattern( const BlinkPattern &pattern );
				virtual LEDControl &load_pattern( const LEDPattern pattern ) { return load_pattern( LED_get_pattern(pattern) ); }
				/// @}

				/** Load a random pattern.  The pattern will be different than the one currently loaded (if any) and
				 * may have multiple steps.  By requiring at least 2 steps in the pattern, this means extremely simple
				 * patterns like @ref LEDPattern::none and @ref LEDPattern::all will be skipped.
				 *
				 * @note Ensure you call @p srand() somewhere in your application prior to calling
				 * @p load_random_pattern(), otherwise the order in which patterns are loaded wont be random.
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.load_random_pattern();
				 * led_control.start_blink_thread();
				 * ~~~~
				 *
				 * @see @ref load_pattern()
				 * 
				 */
				virtual LEDPattern load_random_pattern( const size_t minimum_number_of_steps = 2 );


				/** Invert the pattern in LEDControl (if any).
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.load_pattern( SG::BeagleBone::LEDPattern::kitt );
				 * led_control.start_blink_thread();
				 *
				 * // the LEDs now show a single light sweeping from side to side
				 * std::this_thread::sleep_for( std::chrono::seconds(5) );
				 *
				 * led_control.invert_pattern();
				 * // the LEDs now show all lights on, with a single dark LED moving from side to side
				 * std::this_thread::sleep_for( std::chrono::seconds(5) );
				 * ~~~~
				 *
				 * @see @ref SG::BeagleBone::LED_invert_pattern()
				 */
				virtual LEDControl &invert_pattern( void );

				/** Start the LED blink thread.  This can safely be called multiple times.  It doesn't matter if this
				 * is called before or after @ref load_pattern().
				 *
				 * Example code:
				 * ~~~~
				 * SG::BeagleBone::LEDControl led_control;
				 * led_control.load_pattern( SG::BeagleBone::LEDPattern::kitt );
				 * led_control.start_blink_thread();
				 * ~~~~
				 */
				virtual LEDControl &start_blink_thread( void );

				/// Stop the LED blink thread.  This can safely be called multiple times.
				virtual LEDControl &stop_blink_thread( void );

				/** Step through the blink pattern.  This method will loop until @ref stop_blink_thread() is called,
				 * or until @ref blink_loop_should_continue is set to @p false.  This method is normally started as a
				 * secondary thread by calling @ref start_blink_thread().
				 *
				 * @see @ref blink_loop_should_continue
				 * @see @ref blink_interval_in_milliseconds
				 * @see @ref blink_step_index
				 */
				virtual void blink_loop( void );

				/// Fine-tune how @ref blink_loop() behaves while it is running. @{
				std::atomic<bool>	blink_loop_should_continue;			///< Set this to @p false to force the blink loop to immediately exit.
				std::atomic<size_t>	blink_interval_in_milliseconds;		///< The amount of time to wait between each step in the blink pattern.  Default value is 250 milliseconds.
				std::atomic<size_t>	blink_step_index;					///< How far into the blink pattern we are.  This resets back to zero once the end of a pattern is reached.
				/// @}

				/// Used to start a secondary thread to blink the lights in a given pattern.  @see @ref start_blink_thread()
				std::thread blink_thread;

				/** Determine what to do about the LEDs when the LEDControl object goes out of scope.  Normally, the
				 * default triggers are restored so there is some LED activity on the %BeagleBone.  But you can also
				 * decide to leave the LEDs as-is, turn them all on, or turn them all off.  @{
				 */
				enum class ActionOnDestruction
				{
					invalid				= 0	,
					do_nothing				,
					turn_on_all_leds		,
					turn_off_all_leds		,
					restore_default_triggers
				};

				ActionOnDestruction action_on_destruction;
				/// @}

				/// Store a short text message if LEDControl runs into a problem.
				std::string error_message;

			protected:

				/** Determine if this %BeagleBone LED control class has found the LEDs.  If this is @p false, then it
				 * means the %BeagleBone LEDs were not detected, or cannot be accessed.  @see @ref is_valid()
				 */
				bool valid;

				/// Last recorded activity.  Used by @ref blink_thread to determine if the device is idle.  @see @ref set_idle_timeout()
				std::chrono::high_resolution_clock::time_point last_activity;

				/// The length of time that must elapse before the device is considered idle.
				std::chrono::high_resolution_clock::duration idle_timeout;

				/// Timestamp when the device is considered to have gone idle.  @see @ref set_idle_timeout()
				std::chrono::high_resolution_clock::time_point idle_timepoint;

				/// Pattern to use when the device goes idle.  @see @ref set_idle_timeout()
				BlinkPattern idle_pattern;

				/// The current blink pattern for the lights.
				BlinkPattern blink_pattern;

				/// The base path used to access the individual LEDs.  @see @ref get_base_path()
				std::string led_base_path;

				/// Used by @ref load_random_pattern() to keep track of which patterns haven't yet been shown.
				std::vector<size_t> vector_of_random_pattern_enum_values;
		};
	}
}
