/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_GroveI2CDigital.cpp 1817 2016-05-01 00:18:10Z stephane $
 */

#include "sg_GroveI2CDigital.hpp"
#include "sg_i2c-dev.h"

#include <sstream>
#include <iomanip>
#include <cstdlib>
#include <system_error>

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <fcntl.h>


SG::GroveI2CDigital::I2CBus SG::GroveI2CDigital::detect_i2c_bus_number( const bool force_detect, const int force_bus_number )
{
	// -1 is used to indicate "not-yet-detected"
	static int cached_bus_number = -1;

	if (force_bus_number >= 0)
	{
		cached_bus_number = force_bus_number;
	}

	if (cached_bus_number < 0 || force_detect)
	{
		DIR *dir = opendir( "/dev" );
		if (dir == nullptr)
		{
			/// @throw std::system_error if /dev cannot be opened.
			const std::error_code ec( errno, std::system_category());
			throw std::system_error( ec, "SG++ failed to open /dev to find the I2C bus." );
		}

		// look for any files similar to /dev/i2c-* and store them in a (sorted) set
		SG::SStr names;
		while (true)
		{
			dirent *entry = readdir( dir );
			if (entry == nullptr)
			{
				// no more files
				break;
			}

			std::string filename = entry->d_name;
			if (filename.substr(0, 4) == "i2c-")
			{
				names.insert( filename );
			}
		}
		closedir( dir );

		if (names.empty())
		{
			/// @throw std::runtime_error if the I2C bus cannot be found in /dev.
			throw std::runtime_error( "Failed to find I2C bus using /dev/i2c-*." );
		}

		/* Start with the first file (probably "i2c-0") we found in /dev, but if the bus number happens to be zero,
		 * then keep looking to see if there is anything else available, as I believe it is unlikely that the
		 * exposed I2C bus is number 0.
		 */
		for ( const std::string filename : names )
		{
			// extract the bus number from the filename
			cached_bus_number = std::atoi( filename.substr(4).c_str() );
			if (cached_bus_number > 0)
			{
				// we found a non-zero bus number -- stop looking for more
				break;
			}
		}
	}

	return static_cast<I2CBus>(cached_bus_number);
}


SG::GroveI2CDigital::I2CAddress SG::GroveI2CDigital::get_address_from_type( const EGroveType &type )
{
	// initially populated using the information at http://www.seeedstudio.com/wiki/I2C_And_I2C_Address_of_Seeed_Product

	switch (type)
	{
		case EGroveType::kMultichannelGasSensor:			return 0x04;
		case EGroveType::kI2CMotorDriverV10:				return 0x0f;
		case EGroveType::kI2CMotorDriverV12:				return 0x0f;
		case EGroveType::kI2CFMReceiver:					return 0x10;
		case EGroveType::k3AxisDigitalAccelerometer400g:	return 0x18;
		case EGroveType::kQTouchSensor:						return 0x1b;
		case EGroveType::k3AxisDigitalCompass:				return 0x1e; // and 0x3c
		case EGroveType::k6AxisAccelerometerAndCompassV1:	return 0x1e;
		case EGroveType::k6AxisAccelerometerAndCompassV2:	return 0x1e;
		case EGroveType::kDigitalLightSensor:				return 0x29;
		case EGroveType::kI2CColorSensor:					return 0x39;
		case EGroveType::kOLEDDisplay0_96:					return 0x3c;
		case EGroveType::kOLEDDisplay1_12:					return 0x3c;
		case EGroveType::kLCDRGBBacklight:					return 0x3e; // and 0x62
		case EGroveType::kTemperatureAndHumiditySensor:		return 0x40;
		case EGroveType::k3AxisDigitalAccelerometer1_5g:	return 0x4c;
		case EGroveType::kI2CADC:							return 0x50; // used to be 0x55
		case EGroveType::k3AxisDigitalAccelerometer16g:		return 0x53;
		case EGroveType::kNFCTag:							return 0x53;
		case EGroveType::kTouchSensor:						return 0x5a; // and 0x5b, 0x5c, 0x5d
		case EGroveType::kDigitalInfraredTemperatureSensor:	return 0x5b;
		case EGroveType::kRTC:								return 0x68;
		case EGroveType::k3AxisDigitalGyro:					return 0x68;
		case EGroveType::kIMU9DOFV1:						return 0x68; // and 0x77
		case EGroveType::kIMU9DOFV2:						return 0x68; // and 0x77
		case EGroveType::kIMU10DOF:							return 0x68; // and 0x77
		case EGroveType::kGestureV1:						return 0x73;
		case EGroveType::kBarometerHighAccuracy:			return 0x76;
		case EGroveType::kBarometerSensor:					return 0x77;
		case EGroveType::kBarometerSensorBMP180:			return 0x77;
		default:											break;
	}

	// if we get here, then we have no idea what address to use
	return 0;
}


SG::GroveI2CDigital::GroveI2CDigital( const SG::EGroveType t, const std::string &n, const SG::GroveI2CDigital::I2CAddress addr ) :
		GroveI2C		( t, n						),
		i2c_bus_number	( detect_i2c_bus_number()	),
		i2c_address		( addr						),
		i2c_file_handle	( -1						)
{
	if (i2c_address == 0)
	{
		i2c_address = get_address_from_type();
	}

	const std::string filename = "/dev/i2c-" + std::to_string(i2c_bus_number);
	i2c_file_handle = open(filename.c_str(), O_RDWR);
	if (i2c_file_handle < 0)
	{
		/// @throw std::system_error if the I2C device cannot be opened.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ failed to open " + filename + " for read/write" );
	}

	const int i		= static_cast<int>(i2c_address);
	const int rc	= ioctl( i2c_file_handle, I2C_SLAVE, i );
	if (rc < 0)
	{
		close( i2c_file_handle );

		std::stringstream ss;
		ss << std::hex << std::setfill('0') << std::setw(2) << i;

		/// @throw std::system_error if the I2C slave address cannot be set.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ failed to set the I2C slave address to 0x" + ss.str() );
	}

	return;
}


SG::GroveI2CDigital::~GroveI2CDigital( void )
{
	close( i2c_file_handle );
	i2c_file_handle = -1;

	return;
}


uint8_t SG::GroveI2CDigital::read8( const SG::GroveI2CDigital::I2CRegister reg )
{
	errno = 0;
	const uint8_t r = i2c_smbus_read_byte_data( i2c_file_handle, reg );
	if (r == 0xff && errno)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;

		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)reg;

		/// @throw std::system_error if the register cannot be read.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ failed to read 8 bytes from register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}
	
	return r;
}


uint16_t SG::GroveI2CDigital::read16( const SG::GroveI2CDigital::I2CRegister reg )
{
	errno = 0;
	const uint16_t r = i2c_smbus_read_word_data( i2c_file_handle, reg );
	if (r == 0xffff && errno)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;

		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)reg;

		/// @throw std::system_error if the register cannot be read.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ failed to read 16 bytes from register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	/// @note The high/low bytes in the 16-bit word are swapped after reading.
	const uint16_t hibyte = (r & 0xff00) >> 8;
	const uint16_t lobyte = (r & 0x00ff) << 8;
	const uint16_t result = hibyte | lobyte;

	return result;
}


SG::GroveI2CDigital::I2CBlock SG::GroveI2CDigital::read_block( const SG::GroveI2CDigital::I2CRegister reg )
{
	I2CBlock block( I2C_SMBUS_BLOCK_MAX, '\0' );

	errno = 0;
	const int bytes_read = i2c_smbus_read_block_data( i2c_file_handle, reg, block.data() );
	if (bytes_read < 0)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;

		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)reg;
		
		/// @throw std::system_error if the register cannot be read.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ failed to read block from register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	// resize the vector to match the number of bytes read
	block.resize( bytes_read, '\0' );

	return block;
}


SG::GroveI2CDigital &SG::GroveI2CDigital::write_byte( const uint8_t value )
{
	errno = 0;
	const int rc = i2c_smbus_write_byte( i2c_file_handle, value );
	if (rc)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;

		/// @throw std::system_error if the value cannot be writen.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ failed to write a byte to I2C address " + ss_addr.str() );
	}

	return *this;
}


SG::GroveI2CDigital &SG::GroveI2CDigital::write8( const SG::GroveI2CDigital::I2CRegister reg, const uint8_t value )
{
	errno = 0;
	const int rc = i2c_smbus_write_byte_data( i2c_file_handle, reg, value );
	if (rc)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;

		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)reg;

		/// @throw std::system_error if the value cannot be written.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ failed to write a byte to register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	return *this;
}


SG::GroveI2CDigital &SG::GroveI2CDigital::write16( const SG::GroveI2CDigital::I2CRegister reg, const uint16_t value )
{
	/// @note The high/low bytes in the 16-bit word are swapped prior to writting.
	const uint16_t hibyte = (value & 0xff00) >> 8;
	const uint16_t lobyte = (value & 0x00ff) << 8;
	const uint16_t result = hibyte | lobyte;

	errno = 0;
	const int rc = i2c_smbus_write_word_data( i2c_file_handle, reg, result );
	if (rc)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;

		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)reg;

		/// @throw std::system_error if the value cannot be written.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ failed to write 2 bytes to register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	return *this;
}


SG::GroveI2CDigital &SG::GroveI2CDigital::write_block( const SG::GroveI2CDigital::I2CRegister reg, const SG::GroveI2CDigital::I2CBlock &v )
{
	if (v.size() > I2C_SMBUS_BLOCK_MAX)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;

		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)reg;

		/// @throw std::length_error if the block is too long to write.
		throw std::length_error( get_description() + ": SG++ cannot write " + std::to_string(v.size()) + "-byte block to register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	errno = 0;
	const int rc = i2c_smbus_write_block_data( i2c_file_handle, reg, v.size(), v.data() );
	if (rc)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;
		
		std::stringstream ss_reg;
		ss_reg << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)reg;
		
		/// @throw std::system_error if the block failed to write.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ failed to write " + std::to_string(v.size()) + "-byte block to register " + ss_reg.str() + " of I2C address " + ss_addr.str() );
	}

	return *this;
}


SG::GroveI2CDigital &SG::GroveI2CDigital::write_block( const SG::GroveI2CDigital::I2CBlock &v )
{
	const ssize_t bytes_written = write( i2c_file_handle, v.data(), v.size() );
	if (v.size() != (size_t)bytes_written)
	{
		std::stringstream ss_addr;
		ss_addr << "0x" << std::hex << std::setfill('0') << std::setw(2) << (int)i2c_address;

		/// @throw std::system_error if the block failed to write.
		const std::error_code ec( errno, std::system_category());
		throw std::system_error( ec, get_description() + ": SG++ only wrote " + std::to_string(bytes_written) + " bytes of " + std::to_string(v.size()) + "-byte block to file handle for I2C address " + ss_addr.str() );
	}

	return *this;
}
