/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_LED_patterns.hpp 1862 2016-05-25 00:06:15Z stephane $
 */

#pragma once

// Central include file for SeeedGrove++ is "sg++.hpp".

#include "sg_BeagleBone_LED.hpp"
#include <vector>
#include <map>


namespace SG
{
	namespace BeagleBone
	{
		/// Several "built-in" (to this library) blink patterns that can easily be used with @ref SG::BeagleBone::LEDControl.
		enum class LEDPattern
		{
			invalid		= 0	,
			min			= 1	,
			none		= min,	///< turn off all lights
			all				,	///< turn on all lights
			blink			,	///< all lights blink on-and-off at the exact same time
			blinkOffset		,	///< all lights blink on-and-off, but split even/odd
			blinkSplit		,	///< all lights blink on-and-off, but split left/right
			cylon			,	///< sweep multiple lights
			kitt			,	///< sweep single light
			mergeLeft		,	///< lights moving from right-to-left
			mergeRight		,	///< lights moving from left-to-right
			mergeBoth		,	///< combination of merge left and merge right
			inchwormLeft	,	///< lights moving from right-to-left
			inchwormRight	,	///< lights moving from left-to-right
			inchwormBoth	,	///< combination of inchworm left and inchworm right
			insideOut		,	///< inside lights, then outside lights
			max
		};

		/// Every blink pattern is a vector of steps, where each step is map of LEDs to turn on and off. @{
		typedef std::map	< SG::BeagleBone::LED, bool		> PatternStep;
		typedef std::vector	< SG::BeagleBone::PatternStep	> BlinkPattern;
		/// @}

		/// Invert the pattern.
		BlinkPattern LED_invert_pattern( const BlinkPattern &pattern );
		
		/// Turn all the lights off.
		BlinkPattern LED_pattern_none( void );

		/// Turn all lights on.
		BlinkPattern LED_pattern_all( void );

		/// Blink all lights in unison.
		BlinkPattern LED_pattern_blink( void );

		/// Blink even and odd lights.
		BlinkPattern LED_pattern_blinkOffset( void );

		/// Blink left and right lights.
		BlinkPattern LED_pattern_blinkSplit( void );

		/// Multi-light sweep similar to BSG Cylons.
		BlinkPattern LED_pattern_cylon( void );

		/// Single light sweep similar to KITT ("Knight Rider") from the 1980s.
		BlinkPattern LED_pattern_kitt( void );

		/// Lights moving from right to left.
		BlinkPattern LED_pattern_mergeLeft( void );

		/// Lights moving from left to right.
		BlinkPattern LED_pattern_mergeRight( void );

		/// Mix of merge left and merge right.
		BlinkPattern LED_pattern_mergeBoth( void );

		/// Lights moving from right to left.
		BlinkPattern LED_pattern_inchwormLeft( void );
		
		/// Lights moving from left to right.
		BlinkPattern LED_pattern_inchwormRight( void );
		
		/// Mix of merge left and merge right.
		BlinkPattern LED_pattern_inchwormBoth( void );

		/// Blink inside lights, then outside lights.
		BlinkPattern LED_pattern_insideOut( void );
		
		/// Return the corresponding blink pattern that corresponds to the given enum value.
		inline BlinkPattern LED_get_pattern( const LEDPattern pattern )
		{
			switch( pattern )
			{
				case LEDPattern::invalid:		return LED_pattern_none();
				case LEDPattern::none:			return LED_pattern_none();
				case LEDPattern::all:			return LED_pattern_all();
				case LEDPattern::blink:			return LED_pattern_blink();
				case LEDPattern::blinkOffset:	return LED_pattern_blinkOffset();
				case LEDPattern::blinkSplit:	return LED_pattern_blinkSplit();
				case LEDPattern::cylon:			return LED_pattern_cylon();
				case LEDPattern::kitt:			return LED_pattern_kitt();
				case LEDPattern::mergeLeft:		return LED_pattern_mergeLeft();
				case LEDPattern::mergeRight:	return LED_pattern_mergeRight();
				case LEDPattern::mergeBoth:		return LED_pattern_mergeBoth();
				case LEDPattern::inchwormLeft:	return LED_pattern_inchwormLeft();
				case LEDPattern::inchwormRight:	return LED_pattern_inchwormRight();
				case LEDPattern::inchwormBoth:	return LED_pattern_inchwormBoth();
				case LEDPattern::insideOut:		return LED_pattern_insideOut();
				case LEDPattern::max:			return LED_pattern_none();
			}

			// we should never get here -- the switch() statement should handle all cases
			return LED_pattern_blink();
		}
	}
}
