/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_101020003_Button.hpp 1869 2016-05-28 18:40:36Z stephane $
 */

#pragma once

#include "sg_GroveGPIO.hpp"


namespace SG
{
	/** Push button.
	 *
	 * Description | Image
	 * ------------|------
	 * %Button connected to the Grove UART interface. | @image html sg_101020003_Button.jpg
	 *
	 * @note There is no difference between the SKU #101020003 button and the SKU #111020000 panel mount button.
	 *
	 * Example code:
	 * ~~~~
	 * SG::Button button;
	 * bool is_up = button.is_up();
	 * std::cout << "button is " << (is_up ? "up" : "down") << std::endl;
	 * ~~~~
	 *
	 * @see http://www.seeedstudio.com/wiki/Grove_-_Button
	 * @see http://www.seeedstudio.com/depot/Grove-Button-p-766.html
	 * @see http://www.seeedstudio.com/depot/Grove-ButtonP-p-1243.html
	 */
	class Button : public GroveGPIO
	{
		public:

			/// Destructor.
			virtual ~Button( void );

			/// Constructor.
			Button( const std::string &n="" );

			/// Determine if the button is currently pushed down.
			virtual bool is_down( void );

			/// Determine if the button is currently up.
			virtual bool is_up( void ) { return ! is_down(); }

			/** Wait for the button to be pressed down.  Returns immediately if the button is already down.
			 * @param [in] timeout_in_milliseconds Determines the length of time to wait for a button push.  Will wait
			 * indefinitely if the timeout is set to zero.
			 * @return Returns @p TRUE if the button is pushed down, or @p FALSE if a timeout occurred.
			 * @see @ref wait_for_button_up()
			 * @see @ref wait_for_button()
			 */
			virtual bool wait_for_button_down( const size_t timeout_in_milliseconds = 0 );

			/** Wait for the button to be released.  Returns immediately if the button is already up.
			 * @param [in] timeout_in_milliseconds Determines the length of time to wait for the button to be released.
			 * Will wait indefinitely if the timeout is set to zero.
			 * @return Returns @p TRUE if the button is up, or @p FALSE if a timeout occurred.
			 * @see @ref wait_for_button_down()
			 * @see @ref wait_for_button()
			 */
			virtual bool wait_for_button_up( const size_t timeout_in_milliseconds = 0 );

			/** Wait for the button to transition to @p up or @p down, depending on the current state.
			 * @param [in] timeout_in_milliseconds Determines the length of time to wait for the button to change
			 * state.  Will wait indefinitely if the timeout is set to zero.
			 * @return Returns @p TRUE if a transition was detected, or @p FALSE if a timeout occurred.
			 */
			virtual bool wait_for_transition( const size_t timeout_in_milliseconds = 0 );

			/** Wait for any sort of user interaction with the button, either to the @p up or @p down state.  This is
			 * an alias for the method @ref wait_for_transition().
			 * @param [in] timeout_in_milliseconds Determines the length of time to wait for the button to change
			 * state.  Will wait indefinitely if the timeout is set to zero.
			 * @return Returns @p TRUE if a transition was detected, or @p FALSE if a timeout occurred.
			 *
			 * Example code:
			 * ~~~~
			 * std::cout << "Press the button to continue." << std::endl;
			 * SG::Button button;
			 * button.wait_for_button();
			 * ~~~~
			 *
			 * @see @ref wait_for_button_up()
			 * @see @ref wait_for_button_down()
			 */
			virtual bool wait_for_button( const size_t timeout_in_milliseconds = 0 ) { return wait_for_transition(timeout_in_milliseconds); }
	};
}
