/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_BeagleBone_Detect.hpp 1886 2016-06-01 05:35:45Z stephane $
 */

#pragma once

#include "sg_types.hpp"


namespace SG
{
	namespace BeagleBone
	{
		/** The @p %Detect namespace contains several related functions to help determine if the device where the code
		 * is running is a %BeagleBone Black, %BeagleBone Green, or some other type of device.  For example, note the
		 * following code:
		 * ~~~~
		 * if (SG::BeagleBone::Detect::is_beaglebone_green_compatible())
		 * {
		 *     std::cout << "This device is a BeagleBone Green with Grove connections." << std::endl;
		 * }
		 * else if (SG::BeagleBone::Detect::is_beaglebone_black_compatible())
		 * {
		 *     std::cout << "This device is a BeagleBone Black." << std::endl;
		 * }
		 * else if (SG::BeagleBone::Detect::is_beaglebone_compatible())
		 * {
		 *     std::cout << "This device is a BeagleBone." << std::endl;
		 * }
		 * else
		 * {
		 *     std::cout << "This is an unknown type of computer." << std::endl;
		 * }
		 * ~~~~
		 *
		 * In addition, the actual model name can also be obtained.  This contains a single text string which looks
		 * similar to <tt>"TI AM335x BeagleBone Green"<tt>.  For example:
		 * ~~~~
		 * const std::string model = SG::BeagleBone::Detect::get_model();
		 * std::cout << model << std::endl;
		 * ~~~~
		 */
		namespace Detect
		{
			/// Compability enums for this device.
			enum class ECompatibility
			{
				invalid				= 0,
				other				,
				ti_am33xx			,	///< Texas Instrument Sitara ARM processor AM33XX
				ti_am335x_bone		,	///< %BeagleBone
				ti_am335x_bone_black,	///< %BeagleBone Black
				ti_am335x_bone_green	///< %BeagleBone Green
				// if you add a new entry, also update get_compatibilities()
			};

			/** A vector of capabilities describing a device.  You'd typically obtain this by calling
			 * @ref SG::BeagleBone::Detect::get_compatibilities().  %BeagleBone devices typically report they are
			 * compatible with multiple devices.  For example, a %BeagleBone Green doesn't just call itself a
			 * %BeagleBone Green, but also reports as a %BeagleBone and a %BeagleBone Black.
			 */
			typedef std::vector<ECompatibility> Compatibilities;

			/// Convert a single compatibility to a text string.
			std::string to_string( const ECompatibility type );

			/// Convert a vector of capabilities to a vector of strings.
			VStr to_string( const Compatibilities &compabilities );

			/// Get the vector of compabilities for this device.  @see @p /proc/device-tree/compatible
			Compatibilities get_compatibilities( void );

			/** Return the device model or product name as a text string.
			 *
			 * For example:
			 * 	* <tt>TI AM335x %BeagleBone Black</tt>
			 * 	* <tt>TI AM335x %BeagleBone Green</tt>
			 *
			 * @note Even on non-BeagleBone devices, this is likely to return a text string.  For example, this can
			 * return strings such as @p "VirtualBox" or @p "Vostro 470".
			 *
			 * @see @p /proc/device-tree/model
			 * @see @p /sys/devices/virtual/dmi/id/product_name
			 */
			std::string get_model( void );

			/// Determine if the device contains @ref ECompatibility::ti_am335x_bone.
			bool is_beaglebone_compatible( void );

			/// Determine if the device contains @ref ECompatibility::ti_am335x_bone_black.
			bool is_beaglebone_black_compatible( void );

			/// Determine if the device contains @ref ECompatibility::ti_am335x_bone_green.
			bool is_beaglebone_green_compatible( void );
		}
	}
}
