/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_BeagleBone_Watchdog.cpp 1862 2016-05-25 00:06:15Z stephane $
 */

#include "sg_BeagleBone_Watchdog.hpp"
#include <fstream>
#include <fcntl.h>
#include <memory.h>
#include <unistd.h>
#include <sys/ioctl.h>


SG::BeagleBone::Watchdog::~Watchdog( void )
{
	stop_simple_thread();

	int options = WDIOS_DISABLECARD;
	ioctl(shared_fd->fd, WDIOC_SETOPTIONS, &options);

	return;
}


SG::BeagleBone::Watchdog::Watchdog( const size_t timeout_in_seconds ) :
		path					( "/dev/watchdog"									),
		shared_fd				( std::make_shared<SG::FD>(false, path, O_WRONLY)	),
		watchdog_can_continue	( true												)
{
	if (is_valid() && timeout_in_seconds > 0)
	{
		int options = WDIOS_DISABLECARD;
		ioctl(shared_fd->fd, WDIOC_SETOPTIONS, &options);

		set_timeout( timeout_in_seconds );
	}

	return;
}


SG::BeagleBone::Watchdog &SG::BeagleBone::Watchdog::kick( void )
{
	/** If you periodically call @ref kick() from within your code, then don't start the watchdog's <em>simple
	 * thread</em>.  Otherwise, the watchdog object's simple thread regularly calling @ref kick() will completely
	 * defeat manually calling @ref kick().
	 */

	if (is_valid())
	{
		// https://www.kernel.org/doc/Documentation/watchdog/watchdog-api.txt
		ioctl(shared_fd->fd, WDIOC_KEEPALIVE, 0 );
	}

	return *this;
}


SG::BeagleBone::Watchdog &SG::BeagleBone::Watchdog::magic_close(void)
{
	if (is_valid())
	{
		const ssize_t len = write( shared_fd->fd, "V", 1 );
		if (len == 1)
		{
			close( shared_fd->fd );
			shared_fd->fd = -1;
		}
	}

	return *this;
}

size_t SG::BeagleBone::Watchdog::get_timeout( void )
{
	int timeout = -1;

	if (is_valid())
	{
		ioctl(shared_fd->fd, WDIOC_GETTIMEOUT, &timeout);
	}

	if (timeout < 0)
	{
		// something is wrong -- return a more decent value since -1 converted to a size_t will be huge
		timeout = 60;
	}

	return timeout;
}


SG::BeagleBone::Watchdog &SG::BeagleBone::Watchdog::set_timeout( const size_t timeout_in_seconds )
{
	if (is_valid())
	{
		const int timeout = (int)timeout_in_seconds;
		ioctl(shared_fd->fd, WDIOC_SETTIMEOUT, &timeout);
	}

	if (get_timeout() != timeout_in_seconds)
	{
		/// @throw std::invalid_argument if the watchdog timeout cannot be set.
		throw std::invalid_argument( "SG Watchdog timeout cannot be set (" + std::to_string(timeout_in_seconds) + ")" );
	}

	return *this;
}


size_t SG::BeagleBone::Watchdog::get_seconds_remaining_before_reboot(void)
{
	int time_left_in_seconds = 0;
	ioctl(shared_fd->fd, WDIOC_GETTIMELEFT, &time_left_in_seconds);

	return time_left_in_seconds;
}


watchdog_info SG::BeagleBone::Watchdog::get_info(void)
{
	struct watchdog_info info;
	memset( &info, '\0', sizeof(info) );

	ioctl(shared_fd->fd, WDIOC_GETSUPPORT, &info);

	return info;
}


int SG::BeagleBone::Watchdog::get_status(void)
{
	int flags = 0;
	if (is_valid())
	{
		ioctl(shared_fd->fd, WDIOC_GETSTATUS, &flags);
	}

	return flags;
}


int SG::BeagleBone::Watchdog::get_boot_status(void)
{
	int flags = 0;
	if (is_valid())
	{
		ioctl(shared_fd->fd, WDIOC_GETBOOTSTATUS, &flags);
	}

	return flags;
}


SG::BeagleBone::Watchdog &SG::BeagleBone::Watchdog::start_simple_thread( const size_t sleep_time_in_seconds )
{
	/// If you call @ref start_simple_thread(), then there is no need to manually call @ref kick() within your code.

	// stop any previous thread (if any, otherwise does nothing)
	stop_simple_thread();

	watchdog_can_continue	= true;
	simple_watchdog_thread	= std::thread( &SG::BeagleBone::Watchdog::run_simple_loop, this, sleep_time_in_seconds );

	return *this;
}


SG::BeagleBone::Watchdog &SG::BeagleBone::Watchdog::stop_simple_thread( void )
{
	/// @note @ref stop_simple_thread() only returns once the secondary watchdog thread has finished running.

	watchdog_can_continue = false;
	if (simple_watchdog_thread.joinable())
	{
		simple_watchdog_thread.join();
	}

	return *this;
}


SG::BeagleBone::Watchdog &SG::BeagleBone::Watchdog::run_simple_loop( const size_t sleep_time_in_seconds )
{
	/** This method is usually called on a secondary thread by @ref start_simple_thread(), though it could also be
	 * called directly from user code if required.
	 */
	const size_t milliseconds_to_sleep_per_loop	= 1000;
	const size_t number_of_loops = sleep_time_in_seconds * 1000 / milliseconds_to_sleep_per_loop;

	while (watchdog_can_continue)
	{
		for (size_t counter = 0; watchdog_can_continue && counter < number_of_loops; counter ++)
		{
			std::this_thread::sleep_for( std::chrono::milliseconds(milliseconds_to_sleep_per_loop) );
		}

		kick();
	}

	// secondary watchdog thread ends when we get here
	return *this;
}
