/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_103020013_I2CADC.cpp 1741 2016-04-06 01:17:28Z stephane $
 */

#include "sg_103020013_I2CADC.hpp"
#include "sg++.hpp"
#include "sg_i2c-dev.h"

#include <sstream>
#include <iomanip>
#include <system_error>

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>


SG::I2CADC::~I2CADC( void )
{
	return;
}


SG::I2CADC::I2CADC( const std::string &n, const SG::GroveI2CDigital::I2CRegister addr ) :
		GroveI2CDigital( SG::EGroveType::kI2CADC, n, addr )
{
	return;
}


bool SG::I2CADC::operator==( const SG::I2CADC &rhs ) const
{
	return GroveI2C::operator==(rhs) && i2c_address == rhs.i2c_address;
}


SG::I2CADC &SG::I2CADC::reset(void)
{
	/// The exact default values for each register are described in the ADC121C021 datasheet, pages 18-22.

//	write16	( ERegister::kConversionResult	, 0x00		); // result register (read-only, do not write!)
	write8	( ERegister::kAlertStatus		, 0x00		); // alert status register
	write8	( ERegister::kConfiguration		, 0x00		); // configuration register
	write16	( ERegister::kLowLimit			, 0x0000	); // alert limit register (under range)
	write16	( ERegister::kHighLimit			, 0x0fff	); // alert limit register (over range)
	write16	( ERegister::kHysteresis		, 0x0000	); // alert hysteresis register
	write16	( ERegister::kLowestConversion	, 0x0fff	); // lowest conversion register
	write16	( ERegister::kHighestConversion	, 0x0000	); // highest conversion register

	return *this;
}


SG::I2CADC &SG::I2CADC::enable_automatic_mode( const uint8_t value )
{
	/* The configuration register has 8 bits, of which D7-D5 define the cycle time (automatic mode):
	 *
	 *		[D7] cycle time MSB
	 *		[D6] cycle time
	 *		[D5] cycle time LSB
	 *		[D4] alert hold
	 *		[D3] alert flag
	 *		[D2] alert pin
	 *		[D1] reserved (must be zero)
	 *		[D0] polarity
	 */

	// only the first 3 bits are relevant when setting the cycle time
	const uint8_t cycle_time = (value & 0x03) << 5;

	uint8_t config = read8( ERegister::kConfiguration );
	config &= 0x1f;			// remove D7-D5
	config |= cycle_time;	// set the new D7-D5

	write8( ERegister::kConfiguration, config );

	return *this;
}


uint16_t SG::I2CADC::read12( const ERegister reg )
{
	return GroveI2CDigital::read12( static_cast<I2CRegister>(reg) );
}


uint8_t SG::I2CADC::read8( const ERegister reg )
{
	return GroveI2CDigital::read8( static_cast<I2CRegister>(reg) );
}


SG::I2CADC &SG::I2CADC::write8( const ERegister reg, const uint8_t value )
{
	GroveI2CDigital::write8( static_cast<I2CRegister>(reg), value	);
	return *this;
}


SG::I2CADC &SG::I2CADC::write16( const ERegister reg, const uint16_t value )
{
	GroveI2CDigital::write16( static_cast<I2CRegister>(reg), value	);
	return *this;
}
