/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_104020001_VariableColorLED.hpp 1904 2016-06-15 04:20:07Z stephane $
 */

#pragma once

#include "sg_GroveGPIO.hpp"
#include <chrono>


namespace SG
{
	/** LED which can be turned on or off.  The exact red, green, and blue component is determined by physical screws
	 * that must be turned on the back of the twig.
	 *
	 * Description | Image
	 * ------------|------
	 * Variable Color LED connected to the Grove UART interface. | @image html sg_104020001_VariableColorLED.jpg
	 *
	 * @see http://www.seeedstudio.com/wiki/Grove_-_Variable_Color_LED
	 * @see https://www.seeedstudio.com/item_detail.html?p_id=852
	 *
	 * @note This is not the same as the chainable RGB LED, SKU #104030006.  The colour on the chainable RGB LED can
	 * be set programmatically.  The colour on the variable colour LED can only be set by turning tiny screws on the
	 * back of the twig.
	 */
	class VariableColorLED : public GroveGPIO
	{
		public:

		/// Determines whether the destructor attempts to turn off the light.  This can be specified in the constructor.
		bool turn_off_LED_when_destructing;

		/// Destructor.  @see @ref turn_off_LED_when_destructing
		virtual ~VariableColorLED( void );

		/** Constructor.
			* @param [in] n User-supplied name which can be used to identify this LED.  The name can be anything, it isn't parsed by SG++.
			* @param [in] turn_off_at_destruction Normally when an object of this class is destructed, the LED retains
			* the very last state ("colour").  If @p turn_off_at_destruction is set to @p true, then the LED will be
			* turned off by the destructor.
			*/
		VariableColorLED( const std::string &n="", const bool turn_off_at_destruction=false );

		/// Turn the LED on.
		VariableColorLED &turn_on( void );

		/// Turn the LED off.
		VariableColorLED &turn_off( void );

		/** Turn the LED on for a limited amount of time, then turn it off again.  This call is synchronous,
		 * and will not return until the LED has cycled on then off again.  The duration should be more
		 * than 5 milliseconds, otherwise the LED may not behave as expected.  @{
		 */
		VariableColorLED &turn_on( const std::chrono::high_resolution_clock::duration &duration );
		VariableColorLED &turn_on( const size_t duration_in_milliseconds ) { return turn_on( std::chrono::milliseconds(duration_in_milliseconds) ); }
		/// @}

		/** Turn the LED off for a limited amount of time, then turn it on again.  This call is synchronous,
		 * and will not return until the LED has cycled off then on again.  The duration should be more
		 * than 5 milliseconds, otherwise the LED may not behave as expected.  @{
		 */
		VariableColorLED &turn_off( const std::chrono::high_resolution_clock::duration &duration );
		VariableColorLED &turn_off( const size_t duration_in_milliseconds ) { return turn_off( std::chrono::milliseconds(duration_in_milliseconds) ); }
		/// @}
	};

	/// Alias with the international English spelling of @p colour.
	typedef VariableColorLED VariableColourLED;
}
