/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_GroveI2CDigital.hpp 1862 2016-05-25 00:06:15Z stephane $
 */

#pragma once

// Central include file for SeeedGrove++ is "sg++.hpp".

#include "sg_groves.hpp"
#include "sg_GroveI2C.hpp"


namespace SG
{
	/** Base class for digital I2C Groves.
	 *
	 * The BeagleBone Green has two I2C buses.  The first is @p /dev/i2c-0 and the second is @p /dev/i2c-1 or
	 * @p /dev/i2c-2.  The first one, @p i2c-0, is not exposed and isn't of interest to the SG++ library.  The
	 * second one is where I2C Grove twigs are connected.
	 *
	 * When looking at the top of a %BeagleBone Green with the network interface towards the top and the Grove
	 * interfaces at the bottom, @p i2c-1 (or @p i2c-2) is the left-hand-side Grove interface labelled @p J4.
	 * On the board, you'll also see the labels @p SCL (serial clock line), @p SDA (serial data line), @p 3V3,
	 * and @p GND.
	 *
	 * Several parts of the BeagleBone Green are always connected to the I2C bus.  It is important to note the
	 * addresses, because they can conflict with the address of Seeed twigs, such as Seeed's I2C ADC twig which
	 * defaults to address @p 0x55.
	 *
	 * Bus/Device		| Address	| Name		| Description					| Also See
	 * -----------------|-----------|-----------|-------------------------------|---------
	 * @p /dev/i2c-0	| @p 0x24	| TPS65217	| PMU (power management unit)	| @p /sys/bus/i2c/devices/0-0024
	 * @p /dev/i2c-0	| @p 0x50	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/0-0050
	 * @p /dev/i2c-1	| @p 0x54	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/1-0054
	 * @p /dev/i2c-1	| @p 0x55	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/1-0055
	 * @p /dev/i2c-1	| @p 0x56	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/1-0056
	 * @p /dev/i2c-1	| @p 0x57	| AT24C256	| EEPROM						| @p /sys/bus/i2c/devices/1-0057
	 *
	 * The other interface labelled @p J5 on the bottom right side is the UART Grove interface.
	 *
	 * @see @ref GroveUART
	 * @see @ref GroveGPIO
	 * @see https://www.kernel.org/doc/Documentation/i2c/dev-interface
	 */
	class GroveI2CDigital : public GroveI2C
	{
		public:

			/// I2C bus.  @see @ref detect_i2c_bus_number()
			typedef uint8_t I2CBus;

			/// I2C addresses.  @see @ref get_address_from_type()
			typedef uint8_t I2CAddress;

			/// I2C register.
			typedef uint8_t I2CRegister;

			/// Block of sequential bytes to read or write.
			typedef std::vector<uint8_t> I2CBlock;

			/// Constructor.
			GroveI2CDigital( const SG::EGroveType t = SG::EGroveType::kUnknown, const std::string &n = "", const I2CAddress addr = 0 );

			/// Destructor.
			virtual ~GroveI2CDigital( void );

			virtual bool is_analog	( void ) const { return false;	}	///< This is always @p false for GroveI2CDigital.
			virtual bool is_digital	( void ) const { return true;	}	///< This is always @p true for GroveI2CDigital.

			/** Detect the I2C bus to use to talk to Grove I2C devices.  This is normally @p 1 or @p 2.
			 * The detection consists of looking at /dev/i2c* devices.
			 *
			 * @param [in] force_detect The detected bus number is usually remembered to prevent having to re-detect
			 * it every time a new GroveI2C object is instantiated.  If @p force_detect is set to @p true, then this
			 * will force GroveI2C to ignore the previously cached bus number and re-detect the I2C bus.
			 *
			 * @param [in] force_bus_number If an explicit bus number should be used, then @p force_bus_number can
			 * be set to the correct bus value.  This will prevent calls to @p detect_i2c_bus_number() from attempting
			 * to auto-detect the I2C bus.  Setting the I2C bus number this way wont change existing GroveI2C objects
			 * that have already been instantiated.  Only new objects will use the new bus number.  The actual bus
			 * number used by specific objects is @ref i2c_bus_number.
			 */
			static I2CBus detect_i2c_bus_number( const bool force_detect = false, const int force_bus_number = -1 );

			/** Get the default I2C address for a given type.  Note that several Grove types have multiple
			 * addresses.  This only returns the first address.
			 * @see http://www.seeedstudio.com/wiki/I2C_And_I2C_Address_of_Seeed_Product
			 */
			static I2CAddress get_address_from_type( const EGroveType &type );

			/// Get the address for this Grove type.
			I2CAddress get_address_from_type( void ) const { return get_address_from_type( get_type() ); }

			/// Read a byte (8 bits) from the specificed register.
			virtual uint8_t read8( const I2CRegister reg );

			/// Read a 16-bit word from the specified register.
			virtual uint16_t read16( const I2CRegister reg );

			/** Read a 16-bit word from the specified register, but only keep the bottom 12 bits.
			 * Some devices such as the @ref I2CADC return 12-bit values.
			 */
			virtual uint16_t read12( const I2CRegister reg ) { return 0x0fff & read16(reg); }

			/// Read a consecutive block of bytes, anywhere between 1 and 32 bytes in length.  @see @ref write_block()
			virtual I2CBlock read_block( const I2CRegister reg );

			/// Write a single byte to the I2C bus.
			virtual GroveI2CDigital &write_byte	( const uint8_t value );

			/// Write the given value to the specified register. @{
			virtual GroveI2CDigital &write8		( const I2CRegister reg, const  uint8_t value );
			virtual GroveI2CDigital &write16	( const I2CRegister reg, const uint16_t value );
			/// @}

			/// Write a consecutive block of bytes, anywhere between 1 and 32 bytes in length.  @see @ref read_block()
			virtual GroveI2CDigital &write_block( const I2CRegister reg, const I2CBlock &v );

			/// Write a consecutive block of bytes directly to the I2C file handle.
			virtual GroveI2CDigital &write_block( const I2CBlock &v );

			/** The bus number to use for I2C Grove twigs.  The exposed I2C bus is usually @p 1 or @p 2.  This is
			* normally auto-detected by @ref detect_i2c_bus_number() called by the constructor, and doesn't need to
			* be manually set.
			*/
			I2CBus		i2c_bus_number;

			/// The I2C address used to communicate with the Grove device.
			I2CAddress	i2c_address;

			/// The file handle to /dev/i2c-* used to communicate with the Grove device.
			int			i2c_file_handle;
		};
}
