/* Seeed Grove ++ (C) 2015-2016 Stephane Charette <stephanecharette@gmail.com>
 * $Id: sg_common_OLED.hpp 1817 2016-05-01 00:18:10Z stephane $
 */

#pragma once

#include "sg_GroveI2CDigital.hpp"
#include "sg_Font.hpp"
#include <memory>


namespace SG
{
	/** The I2C 0.96" OLED and I2C 1.12" OLED displays have quite a bit of common functionality.  These two Grove twigs
	 * use chips that function in similar manners.
	 *
	 * Grove Twig	| SG++ Class	| Chip						| Dimensions		| Colour Depth				| Datasheet URL
	 * -------------|---------------|---------------------------|-------------------|---------------------------|--------------
	 * 1.12\" OLED	| @ref OLED112	| Solomon Systech SSD1308	| 128 x 64 pixels	| 1-level on/off (1-bit)	| http://garden.seeedstudio.com/images/4/46/SSD1308_1.0.pdf
	 * 0.96\" OLED	| @ref OLED096	| Solomon Systech SSD1327	| 128 x 128 pixels	| 1-level on/off (1-bit)	| http://garden.seeedstudio.com/images/8/82/SSD1327_datasheet.pdf
	 *
	 * Both of these appear at I2C address @p 0x3c.
	 *
	 * ~~~~
	 * i2cdetect -y -r 2
	 *
	 *      0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f
	 * 00:          -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 10: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 20: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 30: -- -- -- -- -- -- -- -- -- -- -- -- 3c -- -- --
	 * 40: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 50: -- -- -- -- UU UU UU UU -- -- -- -- -- -- -- --
	 * 60: -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
	 * 70: -- -- -- -- -- -- -- --
	 * ~~~~
	 */
	class CommonOLED : public GroveI2CDigital
	{
		public:

			/// Orientation for the display.
			enum class EOrientation
			{
				kInvalid	= 0	,
				kLandscape		,
				kPortrait		,
				kLandscape180	,	///< "upside-down" landscape
				kPortrait180		///< "upside-down" portrait
			};

			/// Destructor.
			virtual ~CommonOLED( void );

			/// Constructor.
			CommonOLED( const SG::EGroveType t = SG::EGroveType::kUnknown, const std::string &name = "", const SG::GroveI2CDigital::I2CAddress addr = 0 );

			virtual size_t pixel_width	(void) const	= 0;	///< Get the width in pixels.
			virtual size_t pixel_height	(void) const	= 0;	///< Get the height in pixels.
			virtual size_t pixel_depth	(void) const	= 0;	///< Get the number of bits per pixel.
			virtual size_t memory_width	(void) const	= 0;	///< Get the number of pixels that can be stored in memory.
			virtual size_t memory_height(void) const	= 0;	///< Get the number of pixels that can be stored in memory.
			virtual size_t memory_depth	(void) const	= 0;	///< Get the number of bits each pixel occupies in memory.

			/// Toggle the various display modes. @{
			virtual CommonOLED &normal_display	(void)	= 0;	///< Normal display mode.
			virtual CommonOLED &all_on_display	(void)	= 0;	///< All pixels set @p ON.
			virtual CommonOLED &all_off_display	(void)	= 0;	///< All pixels set @p OFF.
			virtual CommonOLED &invert_display	(void)	= 0;	///< Invert display mode.
			/// @}

			virtual CommonOLED &reset			(void)	= 0;	///< Reset the OLED.
			virtual CommonOLED &clear_screen	(void)	= 0;	///< Set all of the GDDRAM to 0x00 to clear the screen.

			virtual CommonOLED &turn_on			(void)						{ return send_command( 0xaf );			}	///< Turn @p ON the OLED.
			virtual CommonOLED &turn_off		(void)						{ return send_command( 0xae );			}	///< turn @p OFF the OLED.
			virtual CommonOLED &set_contrast	(const uint8_t level=0x7f)	{ return send_command( 0x81, level );	}	///< Set the contrast level to a value between 0 and 255.  @param [in] level Default level is 127 (0x7f).  0 is dark, and 255 is bright.
			virtual CommonOLED &lock			(void)						{ return send_command( 0xfd, 0x16 );	}	///< Lock the OLED driver IC MCU.  Once locked, the only command that will be accepted is @ref unlock().  All other commands and memory access are ignored.
			virtual CommonOLED &unlock			(void)						{ return send_command( 0xfd, 0x12 );	}	///< Unlock the OLED driver IC MCU.  @see @ref lock()
			virtual CommonOLED &start_scrolling	(void)						{ return send_command( 0x2f );			}	///< Start horizontal scrolling.  Multiple calls can be made to @ref start_scrolling() and @ref stop_scrolling() to start and stop scrolling multiple times.  @see @ref setup_scrolling()  @see @ref stop_scrolling()
			virtual CommonOLED &stop_scrolling	(void)						{ return send_command( 0x2e );			}	/// Stop horizontal scrolling.  @see @ref setup_scrolling()  @see @ref start_scrolling()

			/// Send a single-byte command that doesn't have any parameters.
			virtual CommonOLED &send_command( const uint8_t cmd );

			/// Send a command with an additional 1-byte parameter.
			virtual CommonOLED &send_command( const uint8_t cmd, const uint8_t value );

			/// Set a font to use when writing text to the OLED.  By default, if not explicitely set, the font used will be @ref SG::Font::CapRouge. @{
			virtual CommonOLED &set_font( const Font::EType type = SG::Font::EType::k8x8CapRouge );
			virtual CommonOLED &set_font( std::shared_ptr<SG::Font::Monospace> font_to_use );
			/// @}

			virtual CommonOLED &set_XY( const uint8_t x = 0, const uint8_t y = 0 );

			/// Repeat the given pattern until it fills up the entire OLED.
			virtual CommonOLED &flood( const I2CBlock &pattern );

			/** Fade from white to black.  There are 4 transitions, so the default 500 millisecond pause between
			 * each transition means a total time of 2 seconds before this function returns.
			 * @see @ref fade_black_to_white()
			 */
			virtual CommonOLED &fade_white_to_black( const size_t milliseconds_between_transitions = 500 );

			/** Fade from black to white.  There are 4 transitions, so the default 500 millisecond pause between
			 * each transition means a total time of 2 seconds before this function returns.
			 * @see @ref fade_white_to_black()
			 */
			virtual CommonOLED &fade_black_to_white( const size_t milliseconds_between_transitions = 500 );

			/// Display the given text.  @{
			virtual CommonOLED &show( const char c );
			virtual CommonOLED &show( const std::string &str );
			/// @}

			/** Get the full display bitmap.  The display contains 128 x 128 pixels, and each pixels has 4 bits, for a
			 * total of 65536 bits, or 8192 bytes (8 KiB).
			 */
			virtual I2CBlock get_display_bitmap(void);

			/** Set the full display bitmap.
			 * @param [in] block Must be exactly 8192 bytes to represent 128 x 128 pixels x 4 bits grayscale.
			 */
			virtual CommonOLED &set_display_bitmap( const I2CBlock &block );

			/// Return the current orientation.  The default after @ref reset() is @ref EOrientation::kLandscape.
			virtual EOrientation get_orientation(void) const { return orientation; }

			/// Quickly determine if the OLED is in portrait or landscape mode. @{
			bool is_portrait	(void) const { return (orientation == EOrientation::kPortrait	|| orientation == EOrientation::kPortrait180	); }
			bool is_landscape	(void) const { return (orientation == EOrientation::kLandscape	|| orientation == EOrientation::kLandscape180	); }
			/// @}

			/// Set the orientation for the display.  @{
			virtual CommonOLED &rotate_90	(void);
			virtual CommonOLED &rotate_180	(void);
			virtual CommonOLED &rotate_270	(void);
			virtual CommonOLED &set_orientation( const EOrientation new_orientation = EOrientation::kLandscape );
			/// @}

		protected:

			/// Font used when writing text to the OLED.  Default is @ref SG::Font::CapRouge.  @see @ref set_font() @{
			std::shared_ptr<Font::Monospace> font;
			Font::Monospace::Fontmap font_map;
			/// @}

			/// Landscape or portrait mode.
			EOrientation orientation;


#if 0



			/// Direction in which to scroll.
			enum class EScrollingDirection
			{
				kRightHorizontal	= 0x26,
				kLeftHorizontal		= 0x27
			};

			/// Time interval between each scroll step in terms of frame frequency.
			enum class EScrollingSpeed
			{
				k2Frames	= 0x07,	///< fastest (smooth scroll)
				k3Frames	= 0x04,	///< smooth scroll
				k4Frames	= 0x05,	///< smooth scroll
				k5Frames	= 0x06,	///< smooth scroll
				k6Frames	= 0x00,	///< smooth scroll
				k32Frames	= 0x01,	///< "pulse"; not smooth scroll
				k64Frames	= 0x02,	///< "pulse"; not smooth scroll
				k256Frames	= 0x03	///< slowest ("pulse"; not smooth scroll)
			};

			/** Setup scrolling.
			 * 
			 * The parameters @p start_row, @p end_row, @p start_col, and @p end_col may be used to scroll just a small
			 * portion of the screen.  The default values (rows: 0x00-0x7f, columns: 0x00-0x3f) will scroll the entire
			 * screen.
			 *
			 * @note This method will automatically call @p stop_scrolling(), since the scrolling parameters cannot
			 * be changed while scrolling is active.  To re-start scrolling, you must call @ref start_scrolling().
			 *
			 * @see @ref start_scrolling()
			 * @see @ref stop_scrolling()
			 */
			virtual CommonOLED &setup_scrolling(
					const EScrollingDirection	direction	= EScrollingDirection::kRightHorizontal	,
					const EScrollingSpeed		speed		= EScrollingSpeed::k6Frames				,
					const uint8_t				start_row	= 0x00									,
					const uint8_t				end_row		= 0x7f									,
					const uint8_t				start_col	= 0x00									,
					const uint8_t				end_col		= 0x3f									);

			/// Set column start and end.
			virtual CommonOLED &set_column	( const uint8_t start_col = 0x00, const uint8_t end_col = 0x3f );

			/// Set row start and end.
			virtual CommonOLED &set_row	( const uint8_t start_row = 0x00, const uint8_t end_row = 0x7f );
#endif
	};
}
