/* CamCap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: CamCap.hpp 2785 2019-07-18 21:27:17Z stephane $
 */

/** @file
 * Namespace and class definition to capture images from web cams.
 */

#pragma once

#include <map>
#include <set>
#include <vector>
#include <string>
#include <linux/videodev2.h>


/// Namespace for CamCap.  @see @ref CC::Device
namespace CC
{
	/// Map a numerical ID to a text string.
	typedef std::map<uint32_t, std::string> MU32Str;

	/** Map of indexes to inputs.
	 * @see @ref CC::Device::get_inputs()
	 * @see @ref CC::Device::set_input_index()
	 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-enuminput.html#c.v4l2_input
	 */
	typedef std::map<uint32_t, v4l2_input> MInputs;

	/** Map of image formats.
	 * @see @ref CC::Device::get_formats()
	 * @see @ref CC::Device::set_format()
	 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-enum-fmt.html#c.v4l2_fmtdesc
	 */
	typedef std::map<uint32_t, v4l2_fmtdesc> MFormats;

	/// Vector of bytes, used as a data buffer.  For example, to return the bytes read from the camera.
	typedef std::vector<uint8_t> Bytes;

	/// Vector of controls.
	typedef std::vector<v4l2_queryctrl> VCtrls;

	/// Structure used to keep some information on the memory mapped buffers, and the corresponding mmap() address.
	struct BufferDetail
	{
		/** Buffer details returned by V4L call @p VIDIOC_QUERYBUF.
		 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/buffer.html#c.v4l2_buffer
		 */
		v4l2_buffer buffer;

		/// Memory Address returned by @p mmap() for this buffer.
		void * address;

		/// Simple constructor to zero the members.
		BufferDetail(void);
	};

	/// Map of buffer indexes to @ref BufferDetail structures.  @see @ref CC::Device::buffer_details
	typedef std::map< uint32_t, BufferDetail > MIdxBufferDetail;

	/// Structure used when capturing multiple images.
	struct CaptureAndBuffer
	{
		Bytes		bytes;
		v4l2_buffer	info;

		/// Simple constructor to zero out the info.
		CaptureAndBuffer(void);
	};

	/// Vector of structures used when capturing multiple images.  @see @ref CC::Device::capture_many()
	typedef std::vector<CaptureAndBuffer> Images;

	/// Vector of strings.
	typedef std::vector<std::string> VStr;

	struct Dimension
	{
		uint32_t	width;
		uint32_t	height;

		/// Simple constructor.
		Dimension( const uint32_t w=0, const uint32_t h=0 ) : width(w), height(h) { return; }

		// Comparison operator needed to have a sorted set of these objects.
		inline bool operator<( const Dimension &rhs ) const
		{
			if (width == rhs.width)
			{
				return height < rhs.height;
			}
			return width < rhs.width;
		}
	};

	/// A sorted set of dimensions.
	typedef std::set<Dimension> Dimensions;

	/** Simple function to convert a FourCC value to a text string.
	 *
	 * For example:
	 * ~~~~
	 * CC::Device camera_device;
	 * camera_device.initialize();
	 * v4l2_format format = camera_device.get_format();
	 *
	 * std::cout << "current image format is " << CC::fourcc_to_string(format.fmt.pix.pixelformat) << std::endl;
	 * ~~~~
	 *
	 * @see https://www.fourcc.org/
	 */
	std::string fourcc_to_string( uint32_t fcc );

	/** Find the greatest common denominator.  Used to calculate the image aspect ratio.
	 *
	 * For example:
	 * ~~~~
	 * uint32_t w = 640;
	 * uint32_t h = 480;
	 * uint32_t d = gcd(640, 480);
	 *
	 * std::cout << "aspect ratio is " << (w/d) << ":" << (h/d) << std::endl;
	 * ~~~~
	 *
	 * @see https://en.wikipedia.org/wiki/Binary_GCD_algorithm
	 */
	uint32_t gcd( uint32_t u, uint32_t v );

	/** Obtain the CamCap version number.
	 *
	 * For example:
	 * ~~~~
	 * std::cout << "CamCap library version " << CC::get_version() << std::endl;
	 * ~~~~
	 */
	std::string get_version( void );

	/// Different transfer methods.  @see @ref CC::Device::set_transfer_method()
	enum class ETransferMethod
	{
		kRead,	///< Use the @p read() method to transfer images.  @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/rw.html
		kMmap	///< Use the @p mmap() method to transfer images.  @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/mmap.html
	};

	/** Class to represent a single camera device.  Methods are provided to manipulate the camera and take pictures.
	 * These objects are small and can easily be created on the stack.  Any buffers or memory needed internally is then
	 * allocated from the heap or using standard containers such as std::map and std::vector.
	 *
	 * Example code:
	 * ~~~~
	 * #include <CamCap.hpp>
	 *
	 * int main()
	 * {
	 *     CC::Device camera_device;
	 *     camera_device.initialize( "/dev/video1" );
	 *     camera_device.set_format( 640, 480 );
	 *     camera_device.capture_to_jpeg( "image.jpg" );
	 *
	 *     return 0;
	 * }
	 * ~~~~
	 */
	class Device
	{
		public:

			/** Constructor.
			 * @param [in] fn Specifies the device to open.  If left blank, then an attempt will be made to
			 * auto-detect a device that matches @p /dev/video*.
			 *
			 * Example:
			 * ~~~~
			 * // remember to #include <CamCap.hpp>
			 *
			 * CC::Device camera( "/dev/video1" );
			 * camera.capture_to_jpeg( "out.jpg" );
			 * ~~~~
			 */
			Device( const std::string &fn = "" );

			/// Destructor.
			virtual ~Device( void );

			/** Close the camera device.  This is automatically called by the destructor and @ref initialize() prior
			 * to opening a new device.
			 */
			virtual Device &reset( void );

			/** Creates a large block of text describing the device, what it supports, and the current state.
			 * Useful for debugging.
			 *
			 * Example output:
			 * ~~~~{.txt}
			 * Filename:        "/dev/video0"
			 * Initialized:     true
			 * Driver name:     uvcvideo v4.4.40
			 * Device name:     USB 2.0 Camera
			 * Location:        usb-0000:00:0b.0-1
			 * Flags:           0x85200001
			 *                  The device supports the single-planar API through the Video Capture interface.
			 *                  The device supports the struct v4l2_pix_format extended fields.
			 *                  The device supports the read() and/or write() I/O methods.
			 *                  The device supports the streaming I/O method.
			 *                  The driver fills the device_caps field.
			 * Device input #0:
			 *     name:        Camera 1
			 *     type:        V4L2_INPUT_TYPE_CAMERA
			 * Image format #0:
			 *     buffer type: V4L2_BUF_TYPE_VIDEO_CAPTURE
			 *     flags:       0
			 *     description: YUYV 4:2:2
			 *     Fourcc:      YUYV
			 * ...
			 * ~~~~
			 */
			std::string describe( void );

			/// @see @ref set_v4l2_msg_handling()
			enum class EV4L2_msg_handling
			{
				kStdErr			,	///< Tell libv4l2 to send all messages to @p stderr.
				kDiscard		,	///< Redirects all libv4l2 messages to @p /dev/null.
				kKeep				///< Internally store all libv4l2 messages.  @see @ref get_v4l2_messages()
			};

			/** Set what CamCap should do with messages from libv4l2.  Default behaviour is to redirect them
			 * to @p /dev/null.
			 * @note libv4l2 has a single global error message file pointer, so if you have multiple @p CC::Device
			 * objects then you'll need to leave it at kStdErr, or manage the @p v4l2_log_file file pointer yourself.
			 */
			virtual Device &set_v4l2_msg_handling( const EV4L2_msg_handling type = EV4L2_msg_handling::kDiscard );

			/** Obtain the most recent set of libv4l2 messages.  This only works when the v4l2 message handling has
			 * been set to @ref EV4L2_msg_handling::kKeep, otherwise an empty string is returned.
			 * @note Messages that are retrieved are removed from the internal buffer.
			 * @see @ref set_v4l2_msg_handling()
			 */
			std::string get_v4l2_messages( void );

			/// Determine if the device has been initialized.  Set by @ref initialize(). @{
			virtual bool is_initialized		( void ) const { return   initialized; }
			virtual bool is_not_initialized	( void ) const { return ! initialized; }
			/// @}

			/** Initialize the camera device.  Will attempt to scan for an auto-detect a device, and repeatedly call
			 * initialize() until a device is initialized, or all video devices have been tried.
			 * @param [in] use_libv4l_interface determines if CamCap accesses the camera directly or through the
			 * libv4l interface.  Default @p true is to use libv4l.  Set to @p false to bypass libv4l.  This can
			 * also be toggled through @ref set_libv4l_interface().
			 */
			virtual Device &initialize( const bool use_libv4l_interface = true );

			/** Initialize the given camera device.  Does not make any attempt to auto-detect.
			 * @param [in] fn Provide the name of the camera device to open, such as @p /dev/video0.
			 * @param [in] use_libv4l_interface determines if CamCap accesses the camera directly or through the
			 * libv4l interface.  Default @p true is to use libv4l.  Set to @p false to bypass libv4l.  This can
			 * also be toggled through @ref set_libv4l_interface().
			 */
			virtual Device &initialize( const std::string &fn, const bool use_libv4l_interface = true );

			/// Same as other calls to @p initialize(), but required to prevent string literals from being incorrectly interpreted as booleans.
			virtual Device &initialize( const char * const fn ) { return initialize(std::string(fn)); }

			/** Get a map of all the inputs supported by this device.  Normally a webcam will have a single input.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-enuminput.html
			 */
			virtual MInputs get_inputs( void ) const { return inputs; }

			/** Set the input device to the specified index.  This is automatically called by @ref initialize() to
			 * select the first input index returned by @p VIDIOC_ENUMINPUT.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-g-input.html
			 */
			virtual Device &set_input_index( uint32_t index = 0 ) { return xioctl( VIDIOC_S_INPUT, &index, "VIDIOC_S_INPUT" ); }

			/** Get a map of all the image formats supported by this device.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-enum-fmt.html
			 */
			virtual MFormats get_formats( void ) const { return formats; }

			/** Get a copy of the most recently set image format.
			 *
			 * Example:
			 * ~~~~
			 * CC::Device camera_device;
			 * camera_device.initialize();
			 * v4l2_format format = camera_device.get_format();
			 * std::cout << "width:           " << format.fmt.pix.width                             << std::endl
			 *           << "height:          " << format.fmt.pix.height                            << std::endl
			 *           << "pixel format:    " << CC::fourcc_to_string(format.fmt.pix.pixelformat) << std::endl
			 *           << "field:           " << format.fmt.pix.field                             << std::endl
			 *           << "bytes per line:  " << format.fmt.pix.bytesperline                      << std::endl
			 *           << "image size:      " << format.fmt.pix.sizeimage                         << std::endl
			 *           << "colour space:    " << format.fmt.pix.colorspace                        << std::endl;
			 * ~~~~
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-g-fmt.html#c.v4l2_format
			 */
			virtual v4l2_format get_format( void ) const { return selected_format; }

			/** Set the prefered image format the camera should be using.  This includes the image dimensions and
			 * byte format.
			 * @return Returns the format the camera has decided to use, which may not be exactly the same as what
			 * was requested.  For example, the final image width and height may not be what had been requested.
			 * @see @ref get_formats()
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-g-fmt.html
			 */
			virtual v4l2_format set_format( v4l2_format format );

			/** Set the prefered image format using one of the enumerated formats returned by @ref get_formats().
			 * @return Returns the format the camera has decided to use, which may not be exactly the same as what
			 * was requested.  For example, the final image width and height may not be what had been requested.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-g-fmt.html
			 */
			virtual v4l2_format set_format( v4l2_fmtdesc format_description, const size_t width = 0, const size_t height = 0 );

			/** Keep the format the same, but specify a specific width and height.
			 * @param [in] width The requested image width.  If zero, the camera will use the smallest value it supports.
			 * @param [in] height The requested image height.  If zero, the camera will use the smallest value it supports.
			 * @return Returns the format the camera has decided to use, which may not be exactly the same as what was
			 * requested.  For example, the final image width and height may have been adjusted to dimensions supported
			 * by the camera.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-g-fmt.html
			 */
			virtual v4l2_format set_format( const size_t width, const size_t height );

			/** Set a specific pixel format and image dimension to use.
			 * @param [in] pixel_format One of the V4L2 defines, such as @p V4L2_PIX_FMT_YUYV or @p V4L2_PIX_FMT_RGB24
			 * @param [in] width The requested image width.  If zero, the camera will use the smallest value it supports.
			 * @param [in] height The requested image height.  If zero, the camera will use the smallest value it supports.
			 * @return Returns the format the camera has decided to use, which may not be exactly the same as what was
			 * requested.  For example, the final image width and height may have been adjusted to dimensions supported
			 * by the camera.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-g-fmt.html
			 */
			virtual v4l2_format set_format( const int pixel_format, const size_t width, const size_t height );

			/** Used to request memory blocks used to capture images.  This is called automatically by functions such
			 * as @ref initialize() and @ref set_format().  The only reason you'd need to manually this is if
			 * you want to manually set the number of buffers to use.
			 * @param [in] number_of_buffers_to_use is usually a small number, such as 1 or 2..
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-reqbufs.html
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-querybuf.html
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-qbuf.html
			 */
			virtual Device &request_memory_mapped_buffers( size_t number_of_buffers_to_use=1 );

			/** Release all memory mapped buffers previously allocated through @ref request_memory_mapped_buffers().
			 * This is automatically called when necessary, including in the destructor, so normally you shouldn't
			 * need to call this directly.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-reqbufs.html
			 */
			virtual Device &release_memory_mapped_buffers( void );

			/** Send the V4L call VIDIOC_STREAMON to begin capturing.  This is automatically called when necessary.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-streamon.html
			 */
			virtual Device &stream_start( void );

			/** Send the V4L call VIDIOC_STREAMOFF to stop capturing.  This is automatically called when necessary.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-streamon.html
			 */
			virtual Device &stream_stop( void );

			/// Perform the requested @p ioctl() call via @p v4l2_ioctl() from libv4l2. @{
			virtual Device &xioctl( const int request, void *parm, const std::string &name );
			virtual int xioctl( const int request, void *parm );
			/// @}

			/// Open the video device file.
			virtual Device &xopen( const std::string &fn );

			/// Close the video device file.
			virtual Device &xclose( void );

			/// Mmap a block of memory.
			virtual void *xmmap( size_t length, off_t offset);

			/// Unmmap a block of memory.
			virtual Device &xmunmap( void *addr, size_t length );

			/// Read from the device file descriptor.  @return Returns the errno result after the read finished.
			virtual int xread( Bytes &bytes );

			/** Capture an image from the camera device.  This automatically calls @ref initialize() if it hasn't
			 * already been called.  The format of the captured bytes depends on the currently selected image format.
			 *
			 * Example:
			 * ~~~~
			 * CC::Device camera_device;
			 * CC::Bytes bytes = camera_device.capture();
			 * ~~~~
			 */
			virtual Bytes capture( void );

			/** Capture an image from the camera device.  This automatically calls @ref initialize() if it hasn't
			 * already been called.  The format of the captured bytes depends on the currently selected image format.
			 *
			 * Example:
			 * ~~~~
			 * CC::Device camera_device;
			 * CC::Bytes bytes;
			 * v4l2_buffer meta_data = camera_device.capture( bytes );
			 * ~~~~
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/buffer.html#c.v4l2_buffer
			 */
			virtual v4l2_buffer capture( Bytes &bytes );

			/** Calls @ref capture() and then saves the image to a JPEG file.
			 * @warning The image format may be automatically changed via a call to @ref set_format() if the current
			 * image format is not optimal for converting to JPEG.
			 * @return Returns a V4L2 structure with meta information on the captured image.
			 *
			 * Example:
			 * ~~~~
			 * CC::Device camera_device;
			 * camera_device.capture_to_jpeg( "output.jpg" );
			 * ~~~~
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-qbuf.html
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/buffer.html#c.v4l2_buffer
			 */
			virtual v4l2_buffer capture_to_jpeg( const std::string &output_filename );

			/** Capture a sequence of images.
			 * @param [in] count Number of images to capture.
			 * @param [in,out] images Vector of captured images.
			 *
			 * Example:
			 * ~~~~
			 * CC::Device camera_device;
			 * CC::Device::Images images;
			 * camera_device.capture_many( 10, images );
			 * for ( const auto img : images )
			 * {
			 *     process_image( img.bytes );
			 * }
			 * ~~~~
			 */
			virtual Device &capture_many( const size_t count, Images &images );

			/** Capture a sequence of images as JPEG files.
			 * @param [in] count Number of images to capture.
			 * @param [in] directory Directory into which the jpeg files are saved.
			 * @return Returns a vector of filenames.
			 *
			 * Example:
			 * ~~~~
			 * CC::Device camera_device;
			 * CC::VStr v = camera_device.capture_many_to_jpeg( 10, "/tmp" );
			 * for ( const auto filename : v )
			 * {
			 *     std::cout << filename << std::endl;
			 * }
			 * ~~~~
			 */
			virtual VStr capture_many_to_jpeg( const size_t count, const std::string &directory = "." );

			/// Capture an image using memory mapped buffers.
			virtual v4l2_buffer capture_using_mmap_buffers( Bytes &bytes );

			/** Capture an image using @p read() or @p v4l2_read().  @return The @p v4l2_buffer returned is emulated,
			 * and most fields are set to zero.
			 */
			virtual v4l2_buffer capture_using_read( Bytes &bytes );

			/// Get the device name, such as @p /dev/video0.  This is not set until @ref initialize() is called.
			virtual std::string get_name( void ) const { return fname; }

			/** The V4L2 device capabilities are used to identify kernel devices compatible with this specification
			 * and to obtain information about driver and hardware capabilities.  This is automatically called during
			 * initialization, and normally does not need to be manually called.  @see get_capability_flags()
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-querycap.html
			 */
			virtual v4l2_capability get_capabilities( void ) const { return capabilities; }

			/** Returns a std::map that describes all the capabilities of this camera device.
			 * Example:
			 * ~~~~
			 * CC::Device camera_device;
			 * camera_device.initialize();
			 * for ( const auto iter : camera_device.get_capability_flags() )
			 * {
			 *     std::cout << "\t" << iter.second << std::endl;
			 * }
			 * ~~~~
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-querycap.html
			 */
			virtual MU32Str get_capability_flags( void ) const;

			/** Determines how many times the CamCap library will retry a capture if an incomplete buffer is returned
			 * from V4L.  Some cameras or drivers seem to return partial buffers.  CamCap can try multiple times to
			 * transfer an image from the camera when this condition is detected.
			 * @param [in] retries Determine the number of times a capture will be restarted.  Normally is a relatively
			 * small number greater than zero.
			 */
			virtual Device &retry_incomplete_captures( const size_t retries=5 );

			/// Determine if streaming is enabled.  @see @ref stream_start() @see @ref stream_stop()
			virtual bool is_streaming_enabled( void ) const { return streaming_enabled; }

			/// Query the camera to determine what image sizes are supported.
			virtual Dimensions get_dimensions( void );

			/** Decides if CamCap accesses the camera directly, or goes through the libv4l interface.  This is also a
			 * parameter to @ref initialize() since it can impact from very early on how the camera device is opened.
			 * @param [in] toggle Set to @p true to use the compatibility layer offered by @p libv4l.
			 * @param [in] toggle Set to @p false if @p libv4l should be ignored.
			 */
			virtual Device &set_libv4l_interface( const bool toggle = true ) { use_libv4l = toggle; return *this; }

			/** Toggle between memory mapped buffers and direct read when capturing images.
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/rw.html
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/mmap.html
			 */
			virtual Device &set_transfer_method( const ETransferMethod method = ETransferMethod::kMmap );

			/** Get all of the controls supported by this camera.  @note Controls typically keep their values
			 * even when the device is closed and re-opened.
			 *
			 * This does @b not include
			 *  @see @ref set_all_controls_to_defaults()
			 */
			virtual VCtrls get_all_controls( void );

			/** Iterate through the available controls and set all to their default values.  @note Controls typically
			 * keep their values even when the device is closed and re-opened.  It may be beneficial to call this method
			 * to ensure the camera controls are in a known state prior to capturing images.
			 */
			virtual Device &set_all_controls_to_defaults( void );

			/** Set a specific control to a certain value.  Examples (not the full list!) of common V4L control indexes
			 * defined in videodev2.h which may be supported by cameras are:
			 * - @p V4L2_CID_BRIGHTNESS
			 * - @p V4L2_CID_CONTRAST
			 * - @p V4L2_CID_SATURATION
			 * - @p V4L2_CID_HUE
			 * - @p V4L2_CID_AUTO_WHITE_BALANCE
			 * - @p V4L2_CID_GAMMA
			 * - @p V4L2_CID_GAIN
			 * - @p V4L2_CID_POWER_LINE_FREQUENCY
			 * - @p V4L2_CID_WHITE_BALANCE_TEMPERATURE
			 * - @p V4L2_CID_SHARPNESS
			 * - @p V4L2_CID_BACKLIGHT_COMPENSATION
			 * - @p V4L2_CID_EXPOSURE_AUTO
			 * - @p V4L2_CID_EXPOSURE_ABSOLUTE
			 * - @p V4L2_CID_EXPOSURE_AUTO_PRIORITY
			 *
			 * @see @ref get_all_controls()
			 * @see @ref set_all_controls_to_defaults()
			 */
			virtual Device &set_control( const uint32_t index, const int32_t value );

			/// Get the current value for the specified control.  @see @ref set_control()  @see @ref get_all_controls()
			virtual int32_t get_control( const uint32_t index );

			/** Map control names to control indexes.  The match is partial-word and case-insensitive.  The full list
			 * of control names is specific to each camera, though some common names found on most devices include:
			 * - @p brightness
			 * - @p contrast
			 * - @p sturation
			 * - @p hue
			 * - @p gamma
			 * - @p gain
			 * - @p sharpness
			 *
			 * @note If a text match cannot be found, this method will also attempt to parse the name as a hexadecimal value.
			 */
			virtual uint32_t control_name_to_index( std::string name );

			/** Similar to the other @ref set_control(), but takes a name instead of the control index.
			 * The name comparison is case-insitive.  For example, "gain" and "GaIn" both match "Gain".
			 */
			virtual Device &set_control( const std::string &name, const int32_t value );

			/** Similar to the other @ref get_control(), but takes a name instead of the control index.  The name
			 * comparison is partial-word and case-insensitive.  For example, "gain" and "GaIn" both match "Gain".
			 */
			virtual int32_t get_control( const std::string &name );

			/** Return a text string showing the state of all camera device controls.
			 * For example:
			 * ~~~~{.txt}
			 *  Index  Min  Def  Max Value Description
			 * ------ ---- ---- ---- ----- -----------
			 * 980900  -64    0   64     0 Brightness
			 * 980901    0   32   64    32 Contrast
			 * 980902    0   64  128    64 Saturation
			 * 980903  -40    0   40     0 Hue
			 * 98090c    0    1    1     1 White Balance Temperature, Auto
			 * 980910   72  100  500   100 Gamma
			 * 980913    0    0  100     0 Gain
			 * 980918    0    1    2     2 Power Line Frequency
			 * 98091a 2800 4600 6500  4600 White Balance Temperature
			 * 98091b    0    3    6     3 Sharpness
			 * 98091c    0    1    2     1 Backlight Compensation
			 * 9a0901    0    3    3     3 Exposure, Auto
			 * 9a0902    1  157 5000   157 Exposure (Absolute)
			 * 9a0903    0    0    1     1 Exposure, Auto Priority
			 * ~~~~
			 */
			virtual std::string describe_controls( void );

	protected:

			/// Name of the device that was opened.  Set in the constructor or @ref initialize().  Will normally be @p /dev/video0.
			std::string fname;

			/// File descriptor to the opened camera device.
			int fd;

			/// This is set to @p true once the device has been properly initialized.  @see @ref is_initialized()
			bool initialized;

			/// Needed to track V4L2 messages.  @see @ref set_v4l2_msg_handling() @see @ref get_v4l2_messages() @{
			char *v4l2_message_ptr;
			size_t v4l2_message_size;
			/// @}

			/** Device capabilities are obtained in @ref initialize().
			 * @see @ref get_capabilities()
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-querycap.html#c.v4l2_capability
			 */
			v4l2_capability capabilities;

			/// Map of all the inputs supported by a camera.  For most (all?) webcams, there is a single input at index #0.
			MInputs inputs;

			/// Map of all the formats supported by the camera.
			MFormats formats;

			/** The most recently set image format.  @see @ref get_format()
			 * @see https://linuxtv.org/downloads/v4l-dvb-apis/uapi/v4l/vidioc-g-fmt.html
			 */
			v4l2_format selected_format;

			/// Keep track of every memory mapped buffer.
			MIdxBufferDetail buffer_details;

			/// Determine if partial captures are kept or rejected.  @see @ref retry_incomplete_captures()
			size_t retry_count;

			/// State of streaming.  @see @ref stream_start() @see @ref stream_stop()
			bool streaming_enabled;

			/// Should CamCap directly access the camera, or use the libv4l interface layer.
			bool use_libv4l;

			/// Selected transfer method.  @see @ref set_transfer_method()
			ETransferMethod transfer_method;
	};
}
