/* CamCap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: CamCap_mmap.cpp 2154 2017-02-13 19:50:22Z stephane $
 */

/** @file
 * Methods dealing with @p mmap() buffers.
 */

#include "CamCap.hpp"
#include "CamCap_defines.h"


CC::Device &CC::Device::request_memory_mapped_buffers( size_t number_of_buffers_to_use )
{
	if (is_not_initialized())
	{
		initialize();
	}

	release_memory_mapped_buffers();

	// tell the camera driver we want it to allocate some buffers
	v4l2_requestbuffers requestbuffers;
	ZERO(requestbuffers);
	requestbuffers.count	= std::max( (uint32_t)1, (uint32_t)number_of_buffers_to_use );
	requestbuffers.type		= V4L2_BUF_TYPE_VIDEO_CAPTURE;
	requestbuffers.memory	= V4L2_MEMORY_MMAP;
	xioctl( VIDIOC_REQBUFS, &requestbuffers, "VIDIOC_REQBUFS" );

	// buffers have been allocated, but we know nothing about them -- query V4L to get the details
	for (uint32_t idx = 0; idx < requestbuffers.count; idx ++)
	{
		BufferDetail bd;
		bd.buffer.index		= idx;
		bd.buffer.type		= V4L2_BUF_TYPE_VIDEO_CAPTURE;
		bd.buffer.field		= selected_format.fmt.pix.field;
		bd.buffer.memory	= V4L2_MEMORY_MMAP;
		xioctl( VIDIOC_QUERYBUF, &bd.buffer, "VIDIOC_QUERYBUF" );

		// remember this buffer
		buffer_details[idx] = bd;
	}

	// next we use mmap() to get a usable address for each of the buffers
	for ( auto iter : buffer_details )
	{
		const uint32_t idx = iter.first;
		BufferDetail bd = iter.second;
		size_t len = bd.buffer.length;

		// mmap the necessary memory and remember the address; we're going to need it when dequeing buffers and to call unmmap()
		buffer_details[idx].address = xmmap( len, bd.buffer.m.offset );
	}

	// lastly, queue all of the buffers so they're ready to use
	for ( auto iter : buffer_details )
	{
		xioctl( VIDIOC_QBUF, &iter.second.buffer, "VIDIOC_QBUF" );
	}

	return *this;
}


CC::Device &CC::Device::release_memory_mapped_buffers( void )
{
	if (buffer_details.empty() == false)
	{
		stream_stop();

		// unmap the buffers
		for ( auto iter : buffer_details )
		{
			BufferDetail &bd = iter.second;

			xmunmap( bd.address, bd.buffer.length );
		}
		buffer_details.clear();

		// tell V4L2 we don't want any buffers, thus freeing them up
		v4l2_requestbuffers requestbuffers;
		ZERO(requestbuffers);
		requestbuffers.count	= 0;
		requestbuffers.type		= V4L2_BUF_TYPE_VIDEO_CAPTURE;
		requestbuffers.memory	= V4L2_MEMORY_MMAP;
		xioctl( VIDIOC_REQBUFS, &requestbuffers, "VIDIOC_REQBUFS" );
	}

	return *this;
}


CC::Device &CC::Device::set_transfer_method( const CC::ETransferMethod method )
{
	if (is_not_initialized())
	{
		initialize();
	}

	transfer_method = method;

	switch (transfer_method)
	{
		case CC::ETransferMethod::kMmap:
		{
			request_memory_mapped_buffers();
			break;
		}
		case CC::ETransferMethod::kRead:
		{
			release_memory_mapped_buffers();
			break;
		}
	}

	return *this;
}
