/* CamCap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: main.cpp 2161 2017-02-19 18:46:24Z stephane $
 */

/** @file
 * Command-line tool that uses libcamcap to access web cams.
 */

#include <map>
#include <string>
#include <vector>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <CamCap.hpp>
#include <thread>
#include <chrono>
#include <unistd.h>
#include <sys/ioctl.h>

/// Vector of strings.
typedef std::vector<std::string> VStr;

/// Map of strings.
typedef std::map<std::string, std::string> MStr;


/// Display usage text on the camcap tool.
void usage( const std::string &fname )
{
	const MStr m =
	{
		{ "control <index> <value>"	, "Set the control to the specific value.  Note the control index is the hexadecimal value shown in \"describe\".  For example, \"0x980913 20\" or \"980913 20\" would set the gain to 20 on many camera models." },
		{ "control <name> <value>"	, "Similar to the other \"control\" command, but use partial-word case-insensitive matching instead of a numerical index.  For example, \"control brightness 32\"." },
		{ "controls"				, "Display a table with all the control details." },
		{ "defaults"				, "Set all controls back to their default values.  See \"describe\" to see a list of available controls, as every camera is different." },
		{ "describe"				, "Display a large amount of information on the camera device, including image pixel formats, controls, etc." },
		{ "device <filename>"		, "Similar to \"initialize\", but specify the exact V4L2 device that must be opened.  For example, \"device /dev/video1\"." },
		{ "format"					, "Display some information on the currently selected image format.  This is a subset of the information shown by \"describe\"." },
		{ "formatindex <index>"		, "Set the format to the given index. See \"describe\" to get a list of all the supported formats.  Note the list is camera-specific." },
		{ "help"					, "Show this help text." },
		{ "ignore <count>"			, "When the camera is set to auto-adjust the image, the first image may be over- or under-exposed.  It may be beneficial to skip the first few images returned by the camera until calibration has finished." },
		{ "initialize"				, "Immediately initialize the camera.  This uses autodetection to find the first V4L2-capable web camera.  This doesn't have to be explicitely called.  The CamCap library will ensure the camera is properly initialized." },
		{ "jpg <filename>"			, "Take a picture and save it as a .jpg file." },
		{ "jpgs <count>"			, "Similar to \"jpg\", but take a sequence of pictures instead of a single one.  Images will be saved to the current directory." },
		{ "libv4l enable|disable"	, "Enable or disable libv4l.  When disabled, CamCap will directly access the camera via V4L2.  When enabled, the compatibility library libv4l will be used.  This command completely resets and re-initializes the device." },
		{ "mmap"					, "Use the mmap() transfer method.  This is normally the most efficient method and supported by most cameras.  It is the default transfer method.  Also see \"read\"." },
		{ "pause <milliseconds>"	, "Pause for the specified number of milliseconds." },
		{ "raw <filename>"			, "Take a picture in the currently selected format (see \"format\") and save it as a .raw file." },
		{ "read"					, "Use the read() transfer method.  Try this method if the default \"mmap\" method isn't working." },
		{ "retry <count>"			, "Set the number of retry attempts if CamCap fails to transfer a full buffer from V4L2." },
		{ "size <width> <height>"	, "Select an image size to use.  Note the camera driver may use a different image size.  Typically, the nearest supported size will be used.  For example, \"size 0 0\" will select the smallest image size supported by a camera." },
		{ "stream"					, "Turn on image streaming.  This is automatically called prior to capturing raw or jpg images." },
		{ "version"					, "Show version information." }
	};

	std::cout	<< ""													<< std::endl
				<< "Usage:"												<< std::endl
				<< ""													<< std::endl
				<< "\t" << fname << " <cmd1> <cmd2> ..."				<< std::endl
				<< ""													<< std::endl
				<< "Where commands are interpreted from left-to-right."	<< std::endl
				<< "Valid commands are:"								<< std::endl;

	size_t len_header	= 4;
	size_t len_cmd		= 0;
	size_t len_spacer	= 2;
	for (const auto iter : m)
	{
		len_cmd = std::max( len_cmd, iter.first.size() );
	}

	// get the current width of the window
	winsize w;
	ioctl(STDOUT_FILENO, TIOCGWINSZ, &w);

	size_t offset_txt	= len_header + len_cmd + len_spacer;
	size_t len_txt		= (w.ws_col < offset_txt ? 20 : w.ws_col - offset_txt);

	for (const auto iter : m)
	{
		std::string cmd = iter.first;
		std::string txt = iter.second;

		// display the command and optional parameters
		std::cout << std::setw(len_header) << " " << std::left << std::setw(len_cmd) << cmd << std::setw(len_spacer) << " ";

		// now we display the description, but take into account the length of the line and the size of the window
		while (txt.empty() == false)
		{
			std::string to_print;
			if (txt.size() >= len_txt)
			{
				// need to truncate the text -- find the whitespace where we will cut the line
				size_t cutoff = len_txt - 1;
				while (cutoff > 10 && txt[cutoff] != ' ')
				{
					cutoff --;
				}
				to_print = txt.substr( 0, cutoff );
				txt.erase( 0, cutoff );
			}
			else
			{
				to_print.swap( txt );
			}

			// trim any leading whitespace
			while (to_print[0] == ' ')
			{
				to_print.erase( 0, 1 );
			}

			std::cout << to_print << std::endl;

			// if there is more text to display, then we need to insert some blank spaces at the start of the line so everything lines up nicely
			if (txt.empty() == false)
			{
				std::cout << std::setw(offset_txt) << " ";
			}
		}
	}

	std::cout	<< ""																					<< std::endl
				<< "Possible examples:"																	<< std::endl
				<< ""																					<< std::endl
				<< "Take a picture and save it as a JPEG file:"											<< std::endl
				<< "\t" << fname << " jpg output.jpg"													<< std::endl
				<< ""																					<< std::endl
				<< "Specify a specific size and take a JPEG picture:"									<< std::endl
				<< "\t" << fname << " size 800 600 jpg output.jpg"										<< std::endl
				<< ""																					<< std::endl
				<< "Set the contrast to zero, ignore the first image, then take 5 JPEG images:"			<< std::endl
				<< "\t" << fname << " size 640 480 control 980901 0 ignore 1 jpgs 5"					<< std::endl
				<< ""																					<< std::endl
				<< "Using control names instead of indexes:"											<< std::endl
				<< "\t" << fname << " control gain 100 control brightness 32 contrast 48 jpg test.jpg"	<< std::endl
				<< ""																					<< std::endl;

	exit(1);
}


int main( int argc, char *argv[] )
{
	try
	{
		// I'd rather be working with std::string than char*[], so convert argv
		VStr v;
		for (int idx = 0; idx < argc; idx ++)
		{
			std::string cmd = argv[idx];
			while (cmd.size() > 0 && cmd[0] == '-')
			{
				cmd.erase(0, 1);
			}
			v.push_back( cmd );
		}
		if (v.size() < 2)
		{
			v.push_back( "help" );
		}

		CC::Device camera_device;
		bool libv4l_enabled = true;

		// loop through all of the parameters
		for ( size_t idx = 1; idx < v.size(); idx ++ )
		{
			// get some information on this parameter and the next few in case we want to use them as arguments
			const size_t remaining		= v.size() - idx - 1;
			const std::string cmd		= v[idx];
			const std::string arg1		= (remaining >= 1 ? v[idx+1] : "");
			const std::string arg2		= (remaining >= 2 ? v[idx+2] : "");
			const bool arg1_is_numeric	= arg1.empty() ? false : (arg1.find_first_not_of("0123456789") == std::string::npos ? true : false);
			const bool arg2_is_numeric	= arg2.empty() ? false : (arg2.find_first_not_of("0123456789") == std::string::npos ? true : false);

			if (cmd == "usage" || cmd == "help")
			{
				usage( v[0] );
			}
			else if (cmd == "initialize")
			{
				std::cout << "- initializing camera:  ";
				camera_device.initialize( libv4l_enabled );
				std::cout << camera_device.get_name() << std::endl;
			}
			else if (cmd == "device")
			{
				if (arg1.empty())
				{
					std::cout << "Command \"device\" requires a filename." << std::endl;
					exit(1);
				}
				std::cout << "- initializing camera:  ";
				camera_device.initialize( arg1, libv4l_enabled );
				std::cout << camera_device.get_name() << std::endl;
				idx += 1;
			}
			else if (cmd == "format")
			{
				v4l2_format format = camera_device.get_format();
				std::string fourcc = CC::fourcc_to_string(format.fmt.pix.pixelformat);
				std::cout << "- current format: " << format.fmt.pix.width << " x " << format.fmt.pix.height << " " << fourcc << ": " << format.fmt.pix.bytesperline << " bytes per line for a total of " << format.fmt.pix.sizeimage << " bytes" << std::endl;
			}
			else if (cmd == "formatindex")
			{
				if (arg1_is_numeric == false)
				{
					std::cout << "Command \"formatindex\" needs a numeric parameter." << std::endl;
					exit(1);
				}

				const CC::MFormats all_formats = camera_device.get_formats();
				const size_t format_index = std::stoul( arg1 );
				if (format_index >= all_formats.size())
				{
					std::cout << "Index " << format_index << " is out of range. Available formats are 0 to " << all_formats.size() - 1 << std::endl;
					exit(1);
				}

				// figure out what size we're using
				v4l2_format format = camera_device.get_format();

				// set the new format but try to keep the same image sizes
				std::cout << "- setting format:  ";
				format = camera_device.set_format( all_formats.at(format_index), format.fmt.pix.width, format.fmt.pix.height );
				std::string fourcc = CC::fourcc_to_string(format.fmt.pix.pixelformat);
				std::cout << format.fmt.pix.width << " x " << format.fmt.pix.height << " " << fourcc << ": " << format.fmt.pix.bytesperline << " bytes per line for a total of " << format.fmt.pix.sizeimage << " bytes" << std::endl;
				idx += 1;
			}
			else if (cmd == "describe")
			{
				std::cout << camera_device.describe() << std::endl;
			}
			else if (cmd == "size")
			{
				if (arg1_is_numeric == false || arg2_is_numeric == false)
				{
					std::cout << "Command \"size\" needs 2 numeric parameters." << std::endl;
					exit(1);
				}

				const size_t w = std::stoul( arg1 );
				const size_t h = std::stoul( arg2 );
				std::cout << "- setting format to " << w << " x " << h << std::endl;
				const v4l2_format format = camera_device.set_format( w, h );
				if (format.fmt.pix.width	!= w ||
					format.fmt.pix.height	!= h )
				{
					std::cout << "\tNOTE: requested " << w << " x " << h << ", but size selected is " << format.fmt.pix.width << " x " << format.fmt.pix.height << std::endl;
				}
				idx += 2;
			}
			else if (cmd == "jpeg" || cmd == "jpg")
			{
				if (arg1.empty())
				{
					std::cout << "Command \"jpeg\" needs a filename." << std::endl;
					exit(1);
				}
				std::cout << "- taking JPEG image: " << arg1 << std::endl;
				v4l2_buffer info = camera_device.capture_to_jpeg( arg1 );
				idx += 1;

				if (info.length != info.bytesused &&
					info.length > 0)
				{
					std::cout << "\tWARNING: incomplete transfer from camera!  (" << (info.bytesused * 100 / info.length) << "%)" << std::endl;
				}
			}
			else if (cmd == "jpegs" || cmd == "jpgs")
			{
				if (arg1_is_numeric == false)
				{
					std::cout << "Command \"jpgs\" needs an image count." << std::endl;
					exit(1);
				}
				unsigned long count = std::stoul(arg1);
				std::cout << "- taking " << count << " JPEG images" << std::endl;

				const VStr v = camera_device.capture_many_to_jpeg( count );
				for (const auto &str : v)
				{
					std::cout << "\t" << str << std::endl;
				}
				idx += 1;
			}
			else if (cmd == "ignore")
			{
				if (arg1_is_numeric == false)
				{
					std::cout << "Command \"ignore\" needs a numeric parameter." << std::endl;
					exit(1);
				}

				CC::Bytes bytes;
				const size_t count = std::stoul(arg1);
				for (size_t idx=0; idx < count; idx++)
				{
					std::cout << "- capturing and discarding image" << std::endl;
					camera_device.capture( bytes );
				}
				idx += 1;
			}
			else if (cmd == "raw")
			{
				if (arg1.empty())
				{
					std::cout << "Command \"raw\" needs a filename." << std::endl;
					exit(1);
				}
				v4l2_format format = camera_device.get_format();
				std::string fourcc = CC::fourcc_to_string(format.fmt.pix.pixelformat);
				std::cout << "- taking RAW image: " << arg1 << " (format=" << fourcc << ")" << std::endl;
				CC::Bytes bytes = camera_device.capture();
				idx += 1;
				std::ofstream out( arg1 );
				out.write( reinterpret_cast<const char *>(bytes.data()), bytes.size() );
			}
			else if (cmd == "retry")
			{
				if (arg1_is_numeric == false)
				{
					std::cout << "Command \"retry\" needs a numeric parameter." << std::endl;
					exit(1);
				}
				const size_t retry_count = std::stoul( arg1 );
				std::cout << "- setting retry count to " << retry_count << std::endl;
				camera_device.retry_incomplete_captures(retry_count);
				idx += 1;
			}
			else if (cmd == "libv4l")
			{
				if (arg1.empty())
				{
					std::cout << "Command \"libv4l\" requires an additional parameter." << std::endl;
					exit(1);
				}
				libv4l_enabled = (arg1 == "off" || arg1 == "disable" || arg1 == "false") ? false : true;
				if (libv4l_enabled)
				{
					std::cout << "- enabling libv4l interface layer" << std::endl;
				}
				else
				{
					std::cout << "- directly accessing the camera (libv4l is disabled)" << std::endl;
				}
				camera_device.initialize( libv4l_enabled );
				idx ++;
			}
			else if (cmd == "mmap")
			{
				std::cout << "- selecting mmap() transfer method" << std::endl;
				camera_device.set_transfer_method( CC::ETransferMethod::kMmap );
			}
			else if (cmd == "read")
			{
				std::cout << "- selecting read() transfer method" << std::endl;
				camera_device.set_transfer_method( CC::ETransferMethod::kRead );
			}
			else if (cmd == "stream")
			{
				std::cout << "- turning on image streaming" << std::endl;
				camera_device.stream_start();
			}
			else if (cmd == "defaults")
			{
				std::cout << "- setting controls to their default values" << std::endl;
				camera_device.set_all_controls_to_defaults();
			}
			else if (cmd == "control")
			{
				if (arg1.empty() || arg2.empty())
				{
					std::cout << "Command \"control\" requires 2 parameters." << std::endl;
					exit(1);
				}

				const uint32_t index = camera_device.control_name_to_index(arg1);

				if (arg2_is_numeric == false)
				{
					std::cout << "Command \"control\" requires numeric parameters." << std::endl;
					exit(1);
				}
				const int value = std::stoi(arg2);

				// would be nice to know the name of the control -- see if we can find it
				static CC::MU32Str m;
				if (m.empty())
				{
					CC::VCtrls v = camera_device.get_all_controls();
					for (const auto &ctrl : v)
					{
						m[ ctrl.id ] = (char*)ctrl.name;
					}
				}

				std::string name = "unknown";
				if (m.count(index) == 1)
				{
					name = m.at(index);
				}

				std::cout << "- setting control 0x" << std::hex << index << std::dec << " (" << name << ") to " << value << "." << std::endl;
				const int32_t old_value = camera_device.get_control( index );
				camera_device.set_control( index, value );
				const int32_t new_value = camera_device.get_control( index );

				if (new_value != value)
				{
					std::cout << "\tWARNING: set to " << value << ", but current value is " << new_value << std::endl;
				}
				else if (old_value != new_value)
				{
					std::cout << "\t- was " << old_value << ", now set to " << new_value << std::endl;
				}

				idx += 2;
			}
			else if (cmd == "controls")
			{
				std::cout	<< "- controls:" << std::endl
							<< camera_device.describe_controls();
			}
			else if (cmd == "pause" || cmd == "sleep")
			{
				if (arg1_is_numeric == false)
				{
					std::cout << "Command \"pause\" requires a numeric parameter." << std::endl;
					exit(1);
				}

				const unsigned long value = std::stoul( arg1 );
				std::cout << "- sleeping for " << value << " milliseconds" << std::endl;
				std::this_thread::sleep_for( std::chrono::milliseconds(value) );
				idx ++;
			}
			else if (cmd == "version")
			{
				std::cout	<< "CamCap v" << CC::get_version()								<< std::endl
							<< "(C) 2017 Stephane Charette <stephanecharette@gmail.com>"	<< std::endl
							<< "https://www.ccoderun.ca/programming/doxygen/camcap/"		<< std::endl;
			}
			else
			{
				std::cout << "Unknown command \"" << cmd << "\"." << std::endl;
				usage( v[0] );
			}
		}
	}
	catch ( const std::exception &e )
	{
		std::cout << "ERROR: " << e.what() << std::endl;
		return 1;
	}

	return 0;
}
