/* C Code Run (C) 2016-2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: ccr_default_user_passwd.cpp 2205 2017-04-16 08:23:43Z stephane $
 *
 * https://www.ccoderun.ca/bbtools/
 *
 * Check to see if an account is still using the default password.
 *
 * Returns 0 if the password has been changed.
 * Returns 1 if the password is still the default.
 * Returns 2 or more for errors.
 */


#include <algorithm>
#include <iostream>
#include <shadow.h>
#include <crypt.h>


int main( int argc, char *argv[] )
{
	if (argc != 2)
	{
		std::cout << "ERROR -- must specify username" << std::endl;
		return 4;
	}

	struct spwd *ent = getspnam( argv[1] );
	if (ent == NULL)
	{
		std::cout << "ERROR -- failed to retrieve entry for " << argv[1] << std::endl;
		return 3;
	}

	/* The password contains 3 fields:
	 *		1) encryption type (see man crypt: 1=md5, 5=sha-256, 6=sha-512)
	 *		2) salt
	 *		3) password
	 *
	 *		$6$9aw4f6N0$YFE.sqVmgWbFM8C5md6orCaUvUOxiyM68rHEhT806PbcBmd6BYlTPFa2W5wjrXOXllVZlkfkzRUbMX4rmjiox0
	 *		 | \      / \                                                                                   /
	 *		 |  \____/   \_________________________________________________________________________________/
	 *		 |    |                    |
	 *		 |    |                     \___ password hash
	 *		 |    |
	 *		 |     \________________________ salt
	 *		 |
	 *		  \_____________________________ encryption type
	 */

	std::string pw;
	if (ent->sp_pwdp)
	{
		pw = ent->sp_pwdp;
	}
	if (std::count(pw.begin(), pw.end(), '$') != 3)
	{
		std::cout << "ERROR -- unknown password format" << std::endl;
		return 2;
	}

	const std::size_t pos							= pw.find_last_of("$");
	const std::string encryption_method_and_salt	= pw.substr( 0,	pos + 1 );
	const std::string encrypted_password_hash		= pw.substr(	pos + 1 );

	if (encryption_method_and_salt.empty() || encryption_method_and_salt[0] != '$')
	{
		std::cout << "ERROR -- unrecognized password type and salt" << std::endl;
		return 2;
	}

	// CCR scripts uses "icx100", RCN's images uses "temppwd"
	const char *encrypted_icx100	= crypt( "icx100"	, encryption_method_and_salt.c_str() );
	const char *encrypted_temppwd	= crypt( "temppwd"	, encryption_method_and_salt.c_str() );

	if (pw == encrypted_icx100 || pw == encrypted_temppwd)
	{
		std::cout << "User \"" << ent->sp_namp << "\" is still using the default password." << std::endl;
		return 1;
	}

	return 0;
}
